@extends('highSuperAdmin.layout')
@section('title', "All Admins")
@section('page-name', "Admins Management")
@section('page-detail', "All Admins Information Across Tenants")
@section('body')

    <div class="content-are-dashboard p-3">
        <div class="card rounded-3">
            <div class="card-body">
                <div class="border rounded-3 p-3">
                    <div class="row">
                        <div class="col-md-5 col-lg-4">
                            <div class="serch">
                                <div class="input-group mb-3">
                                    <span class="input-group-text bg-transparent" id="basic-addon1">
                                        <i class="fas fa-search search-icon"></i></span>
                                    <input id="searchInput" type="text" class="form-control shadow-none"
                                        placeholder="Search Admins" aria-label="serch" aria-describedby="basic-addon1">
                                </div>
                            </div>
                        </div>
                        <div class="col-md-5 ms-auto">
                            <div class="d-flex justify-content-end">
                                <a href="#" class="btn btn-primary w-auto-fix" data-bs-toggle="modal"
                                   data-bs-target="#addAdminModal">
                                    <i class="fas fa-plus-circle me-3"></i>Add Admin
                                </a>
                            </div>
                        </div>
                    </div>

                    <hr>

                    @if($admins->isEmpty())
                        <h4> No Admins Yet! </h4>
                    @else
                        <div class="row">
                            <div class="col-12">
                                <div class="table-responsive">
                                    <table id="entryTable" class="table">
                                        <thead>
                                            <tr>
                                                <th scope="col">Email</th>
                                                <th scope="col">Tenant</th>
                                                <th scope="col">Domain</th>
                                                <th scope="col">Created Date</th>
                                                <th scope="col">Status</th>
                                                <th scope="col">Action</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            @foreach($admins as $admin)
                                                <tr id="admin-row-{{ $admin->id }}">
                                                    <td class="align-middle">{{ $admin->email }}</td>
                                                    <td class="align-middle">{{ $admin->tenant->name ?? 'N/A' }}</td>
                                                    <td class="align-middle">
                                                        @if($admin->tenant)
                                                            <div class="d-flex align-items-center">
                                                                <span id="domain-text-{{ $admin->id }}">
                                                                    https://{{ $admin->tenant->domain }}.platform.carismacar.org
                                                                </span>
                                                                <button class="btn btn-sm btn-light ms-2"
                                                                    onclick="copyToClipboard('domain-text-{{ $admin->id }}')">
                                                                    <i class="fas fa-copy"></i>
                                                                </button>
                                                            </div>
                                                        @else
                                                            N/A
                                                        @endif
                                                    </td>
                                                    <td class="align-middle">
                                                        {{ \Carbon\Carbon::parse($admin->created_at)->format('m-d-y') }}</td>
                                                    <td class="align-middle">
                                                        @if($admin->tenant && $admin->tenant->active)
                                                            <span class="badge bg-success">Active</span>
                                                        @else
                                                            <span class="badge bg-danger">Inactive</span>
                                                        @endif
                                                    </td>
                                                    <td>
                                                        <div class="btn-group" role="group">
                                                            <a href="#" onclick="viewAdminDetails({{$admin->id}})"
                                                                class="btn btn-primary btn-sm" data-bs-toggle="modal"
                                                                data-bs-target="#viewAdminModal">
                                                                <i class="fas fa-eye"></i> View
                                                            </a>
                                                            
                                                            <a href="#" onclick="editAdmin({{$admin->id}})"
                                                                class="btn btn-warning btn-sm" data-bs-toggle="modal"
                                                                data-bs-target="#editAdminModal">
                                                                <i class="fas fa-edit"></i> Edit
                                                            </a>

                                                            <a href="#" onclick="resetAdminPassword({{$admin->id}})" 
                                                                class="btn btn-info btn-sm">
                                                                <i class="fas fa-key"></i> Reset
                                                            </a>

                                                            <a href="#" onclick="deleteAdmin({{$admin->id}})" 
                                                                class="btn btn-danger btn-sm">
                                                                <i class="fas fa-trash"></i> Delete
                                                            </a>
                                                        </div>
                                                    </td>
                                                </tr>
                                            @endforeach
                                        </tbody>
                                    </table>
                                </div>

                                <!-- Pagination -->
                                <div class="d-flex justify-content-center mt-3">
                                    <nav aria-label="Page navigation">
                                        <ul class="pagination">
                                            @if ($admins->onFirstPage())
                                                <li class="page-item disabled">
                                                    <span class="page-link">Previous</span>
                                                </li>
                                            @else
                                                <li class="page-item">
                                                    <a class="page-link" href="{{ $admins->previousPageUrl() }}" aria-label="Previous">
                                                        <span aria-hidden="true">&laquo;</span>
                                                    </a>
                                                </li>
                                            @endif

                                            @foreach ($admins->links()->elements[0] as $page => $url)
                                                <li class="page-item {{ $page == $admins->currentPage() ? 'active' : '' }}">
                                                    <a class="page-link" href="{{ $url }}">{{ $page }}</a>
                                                </li>
                                            @endforeach

                                            @if ($admins->hasMorePages())
                                                <li class="page-item">
                                                    <a class="page-link" href="{{ $admins->nextPageUrl() }}" aria-label="Next">
                                                        <span aria-hidden="true">&raquo;</span>
                                                    </a>
                                                </li>
                                            @else
                                                <li class="page-item disabled">
                                                    <span class="page-link">Next</span>
                                                </li>
                                            @endif
                                        </ul>
                                    </nav>
                                </div>
                            </div>
                        </div>
                    @endif

                </div>
            </div>
        </div>
    </div>

    <!-- View Admin Details Modal -->
    <div class="modal fade" id="viewAdminModal" tabindex="-1" aria-labelledby="viewAdminModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-centered modal-dialog-scrollable">
            <div class="modal-content border-radious-20">
                <div class="modal-header">
                    <h4 class="modal-title" id="viewAdminModalLabel">Admin Details</h4>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-4">
                    <div id="adminDetailsContent">
                        <div class="text-center">
                            <div class="spinner-border" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Admin Modal -->
    <div class="modal fade" id="addAdminModal" tabindex="-1" aria-labelledby="addAdminModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-centered">
            <div class="modal-content border-radious-20">
                <div class="modal-body p-4">
                    <h4>Add New Admin</h4>
                    <hr>
                    <form id="addAdminForm">
                        @csrf
                        <div class="row">
                            <!-- Email -->
                            <div class="col-md-6">
                                <div class="mb-md-4 mb-3">
                                    <label class="form-label">Email Address</label>
                                    <input name="email" type="email" class="form-control shadow-none" 
                                           placeholder="admin@example.com" required>
                                </div>
                            </div>

                            <!-- Password -->
                            <div class="col-md-6">
                                <div class="mb-md-4 mb-3">
                                    <label class="form-label">Password</label>
                                    <input name="password" type="password" class="form-control shadow-none" 
                                           placeholder="Password" required>
                                </div>
                            </div>

                            <!-- Tenant Selection -->
                            <div class="col-12">
                                <div class="mb-md-4 mb-3">
                                    <label class="form-label">Select Tenant</label>
                                    <select name="tenant_id" class="form-control" required>
                                        <option value="">Choose Tenant</option>
                                        @foreach($tenants as $tenant)
                                            <option value="{{ $tenant->id }}">{{ $tenant->name }} ({{ $tenant->domain }})</option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>

                            <div class="d-flex justify-content-end">
                                <button type="button" class="btn btn-primary-white shadow-none mb-3 px-5"
                                        data-bs-dismiss="modal" aria-label="Close">Cancel
                                </button>
                                <button type="submit" id="add-admin-btn"
                                        class="btn btn-primary shadow-none mb-3 px-5 ms-md-3">Add Admin
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Admin Modal -->
    <div class="modal fade" id="editAdminModal" tabindex="-1" aria-labelledby="editAdminModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-centered">
            <div class="modal-content border-radious-20">
                <div class="modal-body p-4">
                    <h4>Edit Admin</h4>
                    <hr>
                    <form id="editAdminForm">
                        @csrf
                        @method('PUT')
                        <input type="hidden" id="edit_admin_id" name="admin_id">
                        
                        <div class="row">
                            <!-- Email -->
                            <div class="col-md-6">
                                <div class="mb-md-4 mb-3">
                                    <label class="form-label">Email Address</label>
                                    <input name="email" id="edit_admin_email" type="email" class="form-control shadow-none" 
                                           placeholder="admin@example.com" required>
                                </div>
                            </div>

                            <!-- Password (Optional) -->
                            <div class="col-md-6">
                                <div class="mb-md-4 mb-3">
                                    <label class="form-label">New Password (Leave blank to keep current)</label>
                                    <input name="password" id="edit_admin_password" type="password" class="form-control shadow-none" 
                                           placeholder="New Password (Optional)">
                                </div>
                            </div>

                            <!-- Tenant Selection -->
                            <div class="col-12">
                                <div class="mb-md-4 mb-3">
                                    <label class="form-label">Select Tenant</label>
                                    <select name="tenant_id" id="edit_admin_tenant" class="form-control" required>
                                        <option value="">Choose Tenant</option>
                                        @foreach($tenants as $tenant)
                                            <option value="{{ $tenant->id }}">{{ $tenant->name }} ({{ $tenant->domain }})</option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>

                            <div class="d-flex justify-content-end">
                                <button type="button" class="btn btn-primary-white shadow-none mb-3 px-5"
                                        data-bs-dismiss="modal" aria-label="Close">Cancel
                                </button>
                                <button type="submit" id="update-admin-btn"
                                        class="btn btn-primary shadow-none mb-3 px-5 ms-md-3">Update Admin
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

@endsection

@section('custom-scripts')
    <script>
        // Search functionality
        $('#searchInput').on('keyup', function () {
            var value = $(this).val().toLowerCase();
            $("#entryTable tbody tr").filter(function () {
                $(this).toggle($(this).text().toLowerCase().indexOf(value) > -1)
            });
        });

        // View Admin Details
        function viewAdminDetails(adminId) {
            $('#adminDetailsContent').html(`
            <div class="text-center">
                <div class="spinner-border" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
            </div>
        `);

            $.ajax({
                url: `/highsuperadmin/admin/${adminId}/details`,
                type: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function (response) {
                    if (response.success) {
                        const admin = response.admin;
                        const additionalData = response.additional_data;

                        let detailsHtml = `
                        <div class="row">
                            <div class="col-md-6">
                                <h5>Admin Information</h5>
                                <hr>
                                <p><strong>Email:</strong> ${admin.email}</p>
                                <p><strong>Created:</strong> ${new Date(admin.created_at).toLocaleDateString()}</p>
                                <p><strong>Last Login:</strong> ${admin.last_login ? new Date(admin.last_login).toLocaleDateString() : 'Never'}</p>
                                <p><strong>Login Count:</strong> ${additionalData.login_count}</p>
                            </div>

                            <div class="col-md-6">
                                <h5>Tenant Information</h5>
                                <hr>
                                <p><strong>Tenant Name:</strong> ${admin.tenant ? admin.tenant.name : 'N/A'}</p>
                               <p><strong>Domain:</strong> ${admin.tenant ? `
        <span id="modal-domain-text">${admin.tenant.domain}.platform.carismacar.org</span>
        <button class="btn btn-sm btn-light ms-2" onclick="copyToClipboard('modal-domain-text')">
            <i class="fas fa-copy"></i>
        </button>
    ` : 'N/A'}</p>

                                <p><strong>Plan:</strong> ${admin.tenant ? admin.tenant.plan : 'N/A'}</p>
                                <p><strong>Status:</strong> 
                                    ${admin.tenant && admin.tenant.active ?
                                '<span class="badge bg-success">Active</span>' :
                                '<span class="badge bg-danger">Inactive</span>'
                            }
                                </p>
                            </div>
                        </div>

                        <div class="row mt-4">
                            <div class="col-12">
                                <h5>Additional Details</h5>
                                <hr>
                                <p><strong>Total Employees Managed:</strong> ${additionalData.total_employees}</p>
                                <p><strong>Total Departments:</strong> ${additionalData.total_departments}</p>
                                <p><strong>Points Approved:</strong> ${additionalData.points_approved}</p>
                                <p><strong>Last Activity:</strong> ${additionalData.last_activity}</p>
                            </div>
                        </div>
                    `;

                        $('#adminDetailsContent').html(detailsHtml);
                    } else {
                        $('#adminDetailsContent').html(`
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle"></i>
                            Failed to load admin details. Please try again.
                        </div>
                    `);
                    }
                },
                error: function (xhr, status, error) {
                    console.error('Error fetching admin details:', error);
                    $('#adminDetailsContent').html(`
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle"></i>
                        Error loading admin details: ${xhr.responseJSON ? xhr.responseJSON.message : 'Unknown error'}
                    </div>
                `);
                }
            });
        }

        // Edit Admin
        function editAdmin(adminId) {
            // First fetch the admin details
            $.ajax({
                url: `/highsuperadmin/admin/${adminId}/details`,
                type: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function (response) {
                    if (response.success) {
                        const admin = response.admin;
                        
                        // Populate the edit form
                        $('#edit_admin_id').val(admin.id);
                        $('#edit_admin_email').val(admin.email);
                        $('#edit_admin_password').val(''); // Always empty for security
                        $('#edit_admin_tenant').val(admin.tenant_id);
                    }
                },
                error: function (xhr, status, error) {
                    Swal.fire({
                        icon: 'error',
                        title: 'Error!',
                        text: 'Failed to load admin details for editing.',
                        timer: 2000,
                        showConfirmButton: false
                    });
                }
            });
        }

        // Reset Admin Password
        function resetAdminPassword(adminId) {
            Swal.fire({
                title: 'Reset Password?',
                text: "This will generate a new random password for the admin",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#3085d6',
                cancelButtonColor: '#d33',
                confirmButtonText: 'Yes, reset it!'
            }).then((result) => {
                if (result.isConfirmed) {
                    $.ajax({
                        url: `/highsuperadmin/admins/${adminId}/reset-password`,
                        type: 'POST',
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest',
                            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                        },
                        success: function (response) {
                            if (response.success) {
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Password Reset!',
                                    html: `<p>${response.message}</p><p><strong>New Password:</strong> <code>${response.new_password}</code></p>`,
                                    confirmButtonText: 'Copy Password'
                                }).then((result) => {
                                    if (result.isConfirmed) {
                                        navigator.clipboard.writeText(response.new_password);
                                        Swal.fire({
                                            icon: 'success',
                                            title: 'Copied!',
                                            text: 'Password copied to clipboard.',
                                            timer: 1500,
                                            showConfirmButton: false
                                        });
                                    }
                                });
                            }
                        },
                        error: function (xhr, status, error) {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error!',
                                text: 'Failed to reset password. Please try again.',
                                timer: 2000,
                                showConfirmButton: false
                            });
                        }
                    });
                }
            });
        }

        // Delete Admin
        function deleteAdmin(adminId) {
            Swal.fire({
                title: 'Are you sure?',
                text: "This action cannot be undone!",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Yes, delete it!'
            }).then((result) => {
                if (result.isConfirmed) {
                    $.ajax({
                        url: `/highsuperadmin/admins/${adminId}`,
                        type: 'DELETE',
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest',
                            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                        },
                        success: function (response) {
                            if (response.success) {
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Deleted!',
                                    text: response.message,
                                    timer: 2000,
                                    showConfirmButton: false
                                });
                                
                                // Remove the row from table
                                $(`#admin-row-${adminId}`).fadeOut(500, function() {
                                    $(this).remove();
                                });
                            }
                        },
                        error: function (xhr, status, error) {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error!',
                                text: 'Failed to delete admin. Please try again.',
                                timer: 2000,
                                showConfirmButton: false
                            });
                        }
                    });
                }
            });
        }

        // Clear modals on close
        $('#addAdminModal, #editAdminModal').on('hidden.bs.modal', function () {
            $(this).find('form')[0].reset();
        });

        // Add Admin Form Submission
        $('#addAdminForm').on('submit', function (e) {
            e.preventDefault();
            
            var formData = $(this).serialize();
            $('#add-admin-btn').prop('disabled', true).text('Adding...');
            
            $.ajax({
                type: 'POST',
                url: '/highsuperadmin/admins/create',
                data: formData,
                dataType: 'json',
                success: function (res) {
                    if (res.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Success!',
                            text: res.message,
                            timer: 2000,
                            showConfirmButton: false
                        });
                        $('#addAdminModal').modal('hide');
                        setTimeout(function() {
                            window.location.reload();
                        }, 2000);
                    }
                },
                error: function (xhr) {
                    if (xhr.status === 422) {
                        var errors = xhr.responseJSON.errors;
                        var errorMessage = '';
                        $.each(errors, function(key, value) {
                            errorMessage += value[0] + '<br>';
                        });
                        Swal.fire({
                            icon: 'error',
                            title: 'Validation Error',
                            html: errorMessage
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error!',
                            text: 'Something went wrong. Please try again.'
                        });
                    }
                },
                complete: function () {
                    $('#add-admin-btn').prop('disabled', false).text('Add Admin');
                }
            });
        });

        // Edit Admin Form Submission
        $('#editAdminForm').on('submit', function (e) {
            e.preventDefault();
            
            var adminId = $('#edit_admin_id').val();
            var formData = $(this).serialize();
            $('#update-admin-btn').prop('disabled', true).text('Updating...');
            
            $.ajax({
                type: 'PUT',
                url: `/highsuperadmin/admins/${adminId}`,
                data: formData,
                dataType: 'json',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                },
                success: function (res) {
                    if (res.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Success!',
                            text: res.message,
                            timer: 2000,
                            showConfirmButton: false
                        });
                        $('#editAdminModal').modal('hide');
                        setTimeout(function() {
                            window.location.reload();
                        }, 2000);
                    }
                },
                error: function (xhr) {
                    if (xhr.status === 422) {
                        var errors = xhr.responseJSON.errors;
                        var errorMessage = '';
                        $.each(errors, function(key, value) {
                            errorMessage += value[0] + '<br>';
                        });
                        Swal.fire({
                            icon: 'error',
                            title: 'Validation Error',
                            html: errorMessage
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error!',
                            text: 'Something went wrong. Please try again.'
                        });
                    }
                },
                complete: function () {
                    $('#update-admin-btn').prop('disabled', false).text('Update Admin');
                }
            });
        });

        // Copy to clipboard function
        function copyToClipboard(elementId) {
            const textElement = document.getElementById(elementId);
            const text = textElement?.innerText || textElement?.textContent;

            if (text) {
                navigator.clipboard.writeText(text).then(() => {
                    Swal.fire({
                        icon: 'success',
                        title: 'Copied!',
                        text: 'Domain URL copied to clipboard.',
                        timer: 1500,
                        showConfirmButton: false
                    });
                }).catch(err => {
                    console.error('Copy failed', err);
                    Swal.fire({
                        icon: 'error',
                        title: 'Failed!',
                        text: 'Unable to copy text.',
                        timer: 1500,
                        showConfirmButton: false
                    });
                });
            }
        }
    </script>
@endsection