<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Business Card Template</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
</head>
<body>

<div class="card-container">
    <!-- Download Button -->
    <button id="downloadBtn" class="download-button" title="Download Business Card">
        <i class="fas fa-download"></i>
    </button>

    {{-- Business Card Template - Harper Russo Style with Dynamic Data and QR Code --}}
    <div class="business-card-harper" id="businessCard">
        <!-- Decorative diagonal lines -->
        <div class="diagonal-lines">
            <div class="line line-1"></div>
            <div class="line line-2"></div>
        </div>
        
        <!-- Profile photo section -->
        <div class="profile-section">
            <div class="profile-circle">
                <div class="profile-image">
                    @if($businessCard->profile_photo)
                        <img src="{{ $businessCard->profile_photo_url }}" alt="{{ $businessCard->name }}" class="profile-photo">
                    @else
                        <div class="profile-initials">
                            {{ strtoupper(substr($businessCard->name, 0, 2)) }}
                        </div>
                    @endif
                </div>
            </div>
        </div>
        
        <!-- Main content section -->
        <div class="content-section">
            <!-- Company Logo Section -->
            @if($businessCard->company_logo)
                <div class="company-logo-section">
                    <img src="{{ $businessCard->company_logo_url }}" alt="{{ $businessCard->company_name }}" class="company-logo">
                </div>
            @endif

            <div class="name-section">
                <h1 class="name">{{ $businessCard->name }}</h1>
                @if($businessCard->title)
                    <h2 class="title">{{ $businessCard->title }}</h2>
                @endif
                @if($businessCard->company_name)
                    <h3 class="company">{{ $businessCard->company_name }}</h3>
                @endif
            </div>
            
            <div class="contact-info">
                @if($businessCard->phone)
                    <div class="contact-item">
                        <i class="fas fa-phone-alt contact-icon"></i>
                        <span class="contact-text">{{ $businessCard->phone }}</span>
                    </div>
                @endif
                
                @if($businessCard->email)
                    <div class="contact-item">
                        <i class="fas fa-envelope contact-icon"></i>
                        <span class="contact-text">{{ strtoupper($businessCard->email) }}</span>
                    </div>
                @endif
                
                @if($businessCard->website)
                    <div class="contact-item">
                        <i class="fas fa-globe contact-icon"></i>
                        <span class="contact-text">{{ strtoupper($businessCard->website) }}</span>
                    </div>
                @endif
                
                @if($businessCard->location)
                    <div class="contact-item">
                        <i class="fas fa-map-marker-alt contact-icon"></i>
                        <span class="contact-text">{{ strtoupper($businessCard->location) }}</span>
                    </div>
                @endif
            </div>

            <!-- Bio Section -->
            @if($businessCard->bio)
                <div class="bio-section">
                    <p class="bio-text">{{ $businessCard->bio }}</p>
                </div>
            @endif

            <!-- Social Links -->
            @if($businessCard->linkedin || $businessCard->facebook || $businessCard->instagram || $businessCard->twitter)
                <div class="social-section">
                    <div class="social-links">
                        @if($businessCard->linkedin)
                            <a href="{{ $businessCard->linkedin }}" target="_blank" class="social-link linkedin">
                                <i class="fab fa-linkedin-in"></i>
                            </a>
                        @endif
                        @if($businessCard->facebook)
                            <a href="{{ $businessCard->facebook }}" target="_blank" class="social-link facebook">
                                <i class="fab fa-facebook-f"></i>
                            </a>
                        @endif
                        @if($businessCard->instagram)
                            <a href="{{ $businessCard->instagram }}" target="_blank" class="social-link instagram">
                                <i class="fab fa-instagram"></i>
                            </a>
                        @endif
                        @if($businessCard->twitter)
                            <a href="{{ $businessCard->twitter }}" target="_blank" class="social-link twitter">
                                <i class="fab fa-twitter"></i>
                            </a>
                        @endif
                    </div>
                </div>
            @endif

            <!-- QR Code Section -->
            <div class="qr-section">
                <div class="qr-container">
                    <div id="businessCardQR" class="qr-code"></div>
                    <!-- <p class="qr-label">Scan to Save Contact</p> -->
                </div>
            </div>

            <!-- Additional Notes -->
            @if($businessCard->additional_notes)
                <div class="notes-section">
                    <p class="notes-text">{{ $businessCard->additional_notes }}</p>
                </div>
            @endif
        </div>
    </div>
</div>

<!-- QR Code Libraries -->
<script src="https://cdn.jsdelivr.net/npm/qrious@4.0.2/dist/qrious.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/qrcode-generator/1.4.4/qrcode.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/qrcode/1.5.3/qrcode.min.js"></script>

<style>
    body {
        margin: 0;
        padding: 20px;
        background: #f5f5f5;
        font-family: 'Poppins', sans-serif;
        min-height: 100vh;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .card-container {
        position: relative;
        display: inline-block;
    }

    /* Download Button Styles */
    .download-button {
        position: absolute;
        top: -60px;
        right: 0;
        background: linear-gradient(135deg, #d4af37, #f4d03f);
        border: none;
        border-radius: 12px;
        width: 50px;
        height: 50px;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        color: #2d5016;
        font-size: 20px;
        box-shadow: 0 8px 25px rgba(212, 175, 55, 0.3);
        transition: all 0.3s ease;
        z-index: 1000;
    }

    .download-button:hover {
        transform: translateY(-2px);
        box-shadow: 0 12px 35px rgba(212, 175, 55, 0.4);
        background: linear-gradient(135deg, #f4d03f, #d4af37);
    }

    .download-button:active {
        transform: translateY(0);
        box-shadow: 0 6px 20px rgba(212, 175, 55, 0.3);
    }

    .download-button i {
        transition: transform 0.2s ease;
    }

    .download-button:hover i {
        transform: scale(1.1);
    }

    /* Loading state for download button */
    .download-button.loading {
        pointer-events: none;
        opacity: 0.7;
    }

    .download-button.loading i {
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }

    .business-card-harper {
        width: 100%;
        max-width: 800px;
        min-height: 500px;
        height: auto;
        background: linear-gradient(135deg, #2d5016 0%, #4a7c59 50%, #1a3a0f 100%);
        border-radius: 20px;
        position: relative;
        overflow: visible;
        display: flex;
        align-items: stretch;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        margin: 0 auto;
        font-family: 'Poppins', -apple-system, BlinkMacSystemFont, sans-serif;
        padding: 0;
    }

    /* Decorative diagonal lines */
    .diagonal-lines {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        overflow: hidden;
        pointer-events: none;
    }

    .line {
        position: absolute;
        background: linear-gradient(45deg, #d4af37, #f4d03f);
        transform-origin: left center;
    }

    .line-1 {
        top: 30%;
        left: -5%;
        width: 35%;
        height: 8px;
        transform: rotate(25deg);
        opacity: 0.9;
    }

    .line-2 {
        top: 45%;
        left: -3%;
        width: 30%;
        height: 6px;
        transform: rotate(25deg);
        opacity: 0.7;
    }

    /* Profile section */
    .profile-section {
        flex: 0 0 200px;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 40px 60px;
        position: relative;
        z-index: 2;
        min-height: 100%;
    }

    .profile-circle {
        position: relative;
        flex-shrink: 0;
    }

    .profile-circle::before {
        content: '';
        position: absolute;
        top: -15px;
        left: -15px;
        right: -15px;
        bottom: -15px;
        border: 4px solid #d4af37;
        border-radius: 50%;
        background: linear-gradient(135deg, #d4af37, #f4d03f);
        z-index: -1;
    }

    .profile-image {
        width: 200px;
        height: 200px;
        border-radius: 50%;
        overflow: hidden;
        background: white;
        padding: 8px;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .profile-photo {
        width: 100%;
        height: 100%;
        object-fit: cover;
        border-radius: 50%;
    }

    .profile-initials {
        width: 100%;
        height: 100%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 60px;
        font-weight: 800;
        color: #2d5016;
        background: #f8f9fa;
        border-radius: 50%;
    }

    /* Content section */
    .content-section {
        flex: 1;
        padding: 40px 60px 40px 10px;
        display: flex;
        flex-direction: column;
        justify-content: center;
        gap: 20px;
        min-height: 100%;
        overflow: visible;
    }

    /* Company Logo Section */
    .company-logo-section {
        margin-bottom: 5px;
        flex-shrink: 0;
        text-align: left;
    }

    .company-logo {
        height: 45px;
        width: auto;
        max-width: 200px;
    }

    .name-section {
        margin-bottom: 5px;
        flex-shrink: 0;
    }

    .name {
        font-size: clamp(28px, 6vw, 40px);
        font-weight: 800;
        color: #d4af37;
        margin-bottom: 8px;
        line-height: 1.1;
        text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.3);
        letter-spacing: -1px;
    }

    .title {
        font-size: clamp(14px, 3.5vw, 18px);
        font-weight: 400;
        color: #d4af37;
        opacity: 0.9;
        line-height: 1.2;
        text-transform: none;
        margin-bottom: 4px;
    }

    .company {
        font-size: clamp(12px, 2.5vw, 16px);
        font-weight: 500;
        color: #d4af37;
        opacity: 0.8;
        line-height: 1.2;
        text-transform: none;
        margin: 0;
    }

    .contact-info {
        display: flex;
        flex-direction: column;
        gap: 10px;
        flex-shrink: 0;
    }

    .contact-item {
        display: flex;
        align-items: center;
        gap: 15px;
    }

    .contact-icon {
        width: 20px;
        font-size: 16px;
        color: #d4af37;
        flex-shrink: 0;
        text-align: center;
    }

    .contact-text {
        font-size: clamp(12px, 2.5vw, 16px);
        color: #d4af37;
        font-weight: 500;
        line-height: 1.2;
        letter-spacing: 0.5px;
        word-break: break-word;
    }

    /* Bio section */
    .bio-section {
        margin-top: 10px;
        flex-shrink: 0;
    }

    .bio-text {
        font-size: clamp(11px, 2vw, 14px);
        color: #d4af37;
        opacity: 0.8;
        line-height: 1.4;
        font-style: italic;
        margin: 0;
        word-wrap: break-word;
    }

    /* Social section */
    .social-section {
        margin-top: 5px;
        display: flex;
        justify-content: flex-start;
        flex-shrink: 0;
    }

    .social-links {
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
    }

    .social-link {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 36px;
        height: 36px;
        background: rgba(212, 175, 55, 0.2);
        border: 2px solid #d4af37;
        border-radius: 8px;
        color: #d4af37;
        text-decoration: none;
        font-size: 16px;
        transition: all 0.3s ease;
        flex-shrink: 0;
    }

    .social-link:hover {
        background: #d4af37;
        color: #2d5016;
        transform: scale(1.1);
    }

    /* QR Code Section */
    .qr-section {
        margin-top: 10px;
        display: flex;
        justify-content: flex-start;
        flex-shrink: 0;
    }

    .qr-container {
        display: flex;
        flex-direction: column;
        align-items: center;
        /* gap: 8px; */
        /* padding: 15px; */
        /* border-radius: 12px; */
        /* transition: all 0.3s ease; */
    }

    .qr-code {
        width: 100px;
        height: 100px;
        display: flex;
        align-items: center;
        justify-content: center;
        /* background: white; */
        /* border-radius: 8 px; */
        /* padding: 4px; */
    }

    .qr-code canvas,
    .qr-code img {
        width: 100%;
        height: 100%;
        border-radius: 4px;
    }

    .qr-label {
        font-size: clamp(8px, 1.5vw, 10px);
        color: #d4af37;
        font-weight: 500;
        text-align: center;
        margin: 0;
        opacity: 0.8;
        letter-spacing: 0.3px;
        text-transform: uppercase;
    }

    /* Notes section */
    .notes-section {
        margin-top: 10px;
        padding-top: 15px;
        border-top: 1px solid rgba(212, 175, 55, 0.3);
        flex-shrink: 0;
    }

    .notes-text {
        font-size: clamp(9px, 1.8vw, 12px);
        color: #d4af37;
        opacity: 0.7;
        line-height: 1.3;
        margin: 0;
        text-align: left;
        word-wrap: break-word;
    }

    /* Responsive Design - Enhanced */
    @media (max-width: 1000px) {
        .business-card-harper {
            max-width: 750px;
        }
        
        .profile-section {
            flex: 0 0 350px;
            padding: 35px 40px;
        }
        
        .profile-image {
            width: 180px;
            height: 180px;
        }

        .qr-code {
            width: 75px;
            height: 75px;
        }

        .download-button {
            top: -50px;
            width: 45px;
            height: 45px;
            font-size: 18px;
        }
    }

    @media (max-width: 900px) {
        .business-card-harper {
            width: 95vw;
            min-height: 450px;
            max-width: 800px;
        }
        
        .profile-section {
            flex: 0 0 300px;
            padding: 30px 35px;
        }
        
        .profile-image {
            width: 160px;
            height: 160px;
        }
        
        .content-section {
            padding: 30px 40px 30px 15px;
            gap: 15px;
        }

        .qr-code {
            width: 70px;
            height: 70px;
        }
    }

    @media (max-width: 700px) {
        .business-card-harper {
            flex-direction: column;
            height: auto;
            min-height: 650px;
            width: 95vw;
            max-width: 450px;
            padding: 20px;
            box-sizing: border-box;
        }

        .profile-section {
            flex: 0 0 auto;
            padding: 30px 20px 20px 20px;
            order: 1;
        }

        .profile-image {
            width: 140px;
            height: 140px;
        }

        .profile-initials {
            font-size: 42px;
        }

        .content-section {
            flex: 1;
            padding: 20px 20px 30px 20px;
            gap: 15px;
            text-align: left;
            order: 2;
        }

        .line-1,
        .line-2 {
            display: none;
        }

        .name {
            font-size: clamp(24px, 6vw, 32px);
        }

        .title {
            font-size: clamp(14px, 3.5vw, 18px);
        }

        .company {
            font-size: clamp(12px, 3vw, 16px);
        }

        .contact-text {
            font-size: clamp(10px, 2.8vw, 13px);
        }
        
        .social-links {
            gap: 10px;
        }
        
        .social-link {
            width: 32px;
            height: 32px;
            font-size: 14px;
        }

        .qr-section {
            justify-content: center;
            margin-top: 15px;
        }

        .qr-code {
            width: 65px;
            height: 65px;
        }

        .qr-label {
            font-size: clamp(7px, 1.8vw, 9px);
        }

        .download-button {
            top: -45px;
            width: 40px;
            height: 40px;
            font-size: 16px;
        }
    }

    @media (max-width: 480px) {
        .business-card-harper {
            min-height: 600px;
            padding: 15px;
        }

        .profile-image {
            width: 120px;
            height: 120px;
        }

        .profile-initials {
            font-size: 36px;
        }

        .content-section {
            padding: 15px 15px 25px 15px;
            gap: 12px;
        }

        .contact-item {
            gap: 10px;
        }

        .contact-info {
            gap: 8px;
        }

        .name {
            font-size: clamp(20px, 7vw, 28px);
        }

        .title {
            font-size: clamp(12px, 4vw, 16px);
        }
        
        .company {
            font-size: clamp(11px, 3vw, 14px);
        }

        .social-link {
            width: 28px;
            height: 28px;
            font-size: 12px;
        }

        .qr-code {
            width: 60px;
            height: 60px;
        }

        .qr-container {
            padding: 12px;
        }

        .download-button {
            top: -40px;
            width: 35px;
            height: 35px;
            font-size: 14px;
        }
    }

    @media (max-width: 380px) {
        .business-card-harper {
            padding: 10px;
            min-height: 550px;
        }
        
        .contact-text {
            font-size: clamp(9px, 3vw, 11px);
            line-height: 1.1;
        }
        
        .bio-text {
            font-size: clamp(8px, 2.5vw, 10px);
        }
        
        .notes-text {
            font-size: clamp(7px, 2vw, 9px);
        }

        .qr-code {
            width: 55px;
            height: 55px;
        }

        .qr-container {
            padding: 10px;
        }
    }

    .business-card-harper * {
        box-sizing: border-box;
    }

    @media print {
        .download-button {
            display: none;
        }
        
        .business-card-harper {
            width: 800px;
            height: auto;
            min-height: 500px;
            break-inside: avoid;
        }
        
        body {
            background: white !important;
        }
    }
</style>

<script>
// Generate Contact Details QR Code for the business card
function generateBusinessCardQR() {
    // Prepare contact details as plain text
    const contactData = generateContactDetailsText();
    
    const qrContainer = document.getElementById('businessCardQR');
    if (!qrContainer) return;
    
    // Clear previous QR code
    qrContainer.innerHTML = '';
    
    // Try different QR generation methods
    try {
        // Method 1: Using QRious (most reliable)
        if (typeof QRious !== 'undefined') {
            generateQRWithQRious(contactData, qrContainer);
        }
        // Method 2: Using QRCode.js
        else if (typeof QRCode !== 'undefined') {
            generateQRWithQRCodeJS(contactData, qrContainer);
        }
        // Method 3: Using qrcode-generator
        else if (typeof qrcode !== 'undefined') {
            generateQRWithQRCodeGenerator(contactData, qrContainer);
        }
        // Fallback: Online service
        else {
            generateQRWithOnlineService(contactData, qrContainer);
        }
    } catch (error) {
        console.error('QR generation failed:', error);
        generateQRWithOnlineService(contactData, qrContainer);
    }
}

// Generate simple contact details as plain text (scannable format)
function generateContactDetailsText() {
    const name = '{{ $businessCard->name ?? "" }}';
    const title = '{{ $businessCard->title ?? "" }}';
    const company = '{{ $businessCard->company_name ?? "" }}';
    const phone = '{{ $businessCard->phone ?? "" }}';
    const email = '{{ $businessCard->email ?? "" }}';
    const website = '{{ $businessCard->website ?? "" }}';
    const location = '{{ $businessCard->location ?? "" }}';
    
    // Create simple contact information (no emojis, minimal formatting)
    let contactInfo = 'CONTACT DETAILS\n\n';
    
    if (name) {
        contactInfo += `Name: ${name}\n`;
    }
    
    if (title) {
        contactInfo += `Title: ${title}\n`;
    }
    
    if (company) {
        contactInfo += `Company: ${company}\n`;
    }
    
    if (phone) {
        contactInfo += `Phone: ${phone}\n`;
    }
    
    if (email) {
        contactInfo += `Email: ${email}\n`;
    }
    
    if (website) {
        contactInfo += `Website: ${website}\n`;
    }
    
    if (location) {
        contactInfo += `Location: ${location}\n`;
    }
    
    return contactInfo || 'Contact Information';
}

// Method 1: QRious
function generateQRWithQRious(data, container) {
    const canvas = document.createElement('canvas');
    const qr = new QRious({
        element: canvas,
        value: data,
        size: 200,
        level: 'M',
        foreground: '#000000',
        background: '#ffffff'
    });
    container.appendChild(canvas);
}

// Method 2: QRCode.js
function generateQRWithQRCodeJS(data, container) {
    QRCode.toCanvas(container, data, {
        width: 200,
        height: 200,
        colorDark: '#000000',
        colorLight: '#ffffff',
        correctLevel: QRCode.CorrectLevel.M,
        margin: 1
    }, function (error) {
        if (error) {
            console.error('QRCode.js failed:', error);
            generateQRWithOnlineService(data, container);
        }
    });
}

// Method 3: qrcode-generator
function generateQRWithQRCodeGenerator(data, container) {
    const qr = qrcode(0, 'M');
    qr.addData(data);
    qr.make();
    
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    const modules = qr.getModuleCount();
    const cellSize = 8;
    const margin = 8;
    
    canvas.width = canvas.height = modules * cellSize + margin * 2;
    
    // Fill background
    ctx.fillStyle = '#ffffff';
    ctx.fillRect(0, 0, canvas.width, canvas.height);
    
    // Draw QR modules
    ctx.fillStyle = '#000000';
    for (let row = 0; row < modules; row++) {
        for (let col = 0; col < modules; col++) {
            if (qr.isDark(row, col)) {
                ctx.fillRect(
                    col * cellSize + margin,
                    row * cellSize + margin,
                    cellSize,
                    cellSize
                );
            }
        }
    }
    
    container.appendChild(canvas);
}

// Fallback: Online service
function generateQRWithOnlineService(data, container) {
    const encodedData = encodeURIComponent(data);
    const qrApiUrl = `https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=${encodedData}&ecc=M`;
    
    const img = document.createElement('img');
    img.src = qrApiUrl;
    img.alt = 'Contact QR Code';
    img.style.width = '100%';
    img.style.height = '100%';
    
    img.onload = function() {
        container.appendChild(img);
    };
    
    img.onerror = function() {
        container.innerHTML = '<div style="display:flex;align-items:center;justify-content:center;font-size:10px;color:#666;">QR Error</div>';
    };
}

// FIXED Download functionality - Only captures the business card without download button
function downloadBusinessCard() {
    const downloadBtn = document.getElementById('downloadBtn');
    const businessCard = document.getElementById('businessCard');
    
    if (!businessCard) {
        alert('Business card not found!');
        return;
    }

    // Add loading state
    downloadBtn.classList.add('loading');
    downloadBtn.innerHTML = '<i class="fas fa-spinner"></i>';
    
    // Temporarily hide the download button during capture
    downloadBtn.style.display = 'none';
    
    // Configure html2canvas options for high quality
    const options = {
        allowTaint: true,
        useCORS: true,
        backgroundColor: null,
        scale: 3, // Higher scale for better quality
        width: businessCard.scrollWidth,
        height: businessCard.scrollHeight,
        scrollX: 0,
        scrollY: 0,
        windowWidth: window.innerWidth,
        windowHeight: window.innerHeight,
        ignoreElements: function(element) {
            // Ignore any element with download-button class
            return element.classList && element.classList.contains('download-button');
        }
    };

    // Small delay to ensure button is hidden
    setTimeout(() => {
        // Capture only the business card element
        html2canvas(businessCard, options).then(canvas => {
            try {
                // Create download link using dynamic name
                const businessCardName = '{{ $businessCard->name ?? "business-card" }}';
                const fileName = businessCardName.toLowerCase().replace(/\s+/g, '-').replace(/[^a-z0-9-]/g, '') + '-business-card.png';
                
                const link = document.createElement('a');
                link.download = fileName;
                link.href = canvas.toDataURL('image/png', 1.0);
                
                // Trigger download
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
                
                // Show success feedback
                showDownloadSuccess();
                
            } catch (error) {
                console.error('Download failed:', error);
                alert('Download failed. Please try again.');
            } finally {
                // Show the download button again and reset state
                downloadBtn.style.display = 'flex';
                downloadBtn.classList.remove('loading');
                downloadBtn.innerHTML = '<i class="fas fa-download"></i>';
            }
        }).catch(error => {
            console.error('html2canvas failed:', error);
            alert('Failed to capture business card. Please try again.');
            
            // Show the download button again and reset state
            downloadBtn.style.display = 'flex';
            downloadBtn.classList.remove('loading');
            downloadBtn.innerHTML = '<i class="fas fa-download"></i>';
        });
    }, 50);
}

// Show download success feedback
function showDownloadSuccess() {
    const downloadBtn = document.getElementById('downloadBtn');
    
    // Temporarily show success icon
    downloadBtn.innerHTML = '<i class="fas fa-check"></i>';
    downloadBtn.style.background = 'linear-gradient(135deg, #4CAF50, #45a049)';
    
    // Reset after 2 seconds
    setTimeout(() => {
        downloadBtn.innerHTML = '<i class="fas fa-download"></i>';
        downloadBtn.style.background = 'linear-gradient(135deg, #d4af37, #f4d03f)';
    }, 2000);
}

// Initialize QR code when document is ready
document.addEventListener('DOMContentLoaded', function() {
    // Small delay to ensure all libraries are loaded
    setTimeout(generateBusinessCardQR, 100);
    
    // Add download button event listener
    const downloadBtn = document.getElementById('downloadBtn');
    if (downloadBtn) {
        downloadBtn.addEventListener('click', downloadBusinessCard);
    }
});

// Also generate QR if the page is already loaded
if (document.readyState === 'complete') {
    setTimeout(generateBusinessCardQR, 100);
}
</script>

</body>
</html>