@extends('chat.layout')
@section('title',"My Business Cards")
@section('page-name',"My Business Cards")
@section('page-detail',"Create and manage your professional digital business cards")
@section('body')

<div class="content-are-dashboard p-3">
    <div class="card">
        <div class="card-body">
            <!-- Header with Create Button -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="d-flex justify-content-between align-items-center border rounded-3 p-3 bg-light">
                        <div>
                            <h4 class="mb-1">
                                <i class="fas fa-id-card me-2 text-primary"></i>
                                Digital Business Cards
                            </h4>
                            <p class="text-muted mb-0">Share your professional profile with unique links</p>
                        </div>
                        <a href="{{ route('business-cards.create') }}" class="btn btn-primary">
                            <i class="fas fa-plus me-2"></i>Create New Card
                        </a>
                    </div>
                </div>
            </div>

            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle me-2"></i>
                    {{ session('success') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            @endif

            @if(session('error'))
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    {{ session('error') }}
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            @endif

            <!-- Business Cards Grid -->
            @if($businessCards->count() > 0)
                <div class="row">
                    @foreach($businessCards as $card)
                        <div class="col-lg-4 col-md-6 mb-4">
                            <div class="card business-card-preview h-100 shadow-sm">
                                <div class="card-body">
                                    <!-- Card Header -->
                                    <div class="d-flex align-items-center mb-3">
                                        <div class="position-relative me-3">
                                            @if($card->profile_photo)
                                                <img src="{{ $card->profile_photo_url }}" alt="Profile" 
                                                     class="rounded-circle" style="width: 50px; height: 50px; object-fit: cover;">
                                            @else
                                                <div class="initials-container" 
                                                     style="width: 50px; height: 50px; background: {{ $card->primary_color }};">
                                                    {{ strtoupper(substr($card->name, 0, 2)) }}
                                                </div>
                                            @endif
                                            <!-- Status Badge -->
                                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill {{ $card->is_active ? 'bg-success' : 'bg-secondary' }}">
                                                <i class="fas fa-{{ $card->is_active ? 'eye' : 'eye-slash' }}" style="font-size: 10px;"></i>
                                            </span>
                                        </div>
                                        <div class="flex-grow-1">
                                            <h6 class="mb-1 fw-bold">{{ $card->name }}</h6>
                                            <small class="text-muted">{{ $card->title }}</small>
                                        </div>
                                        <!-- Template Badge -->
                                        <span class="badge bg-primary">{{ $card->template_name }}</span>
                                    </div>

                                    <!-- Card Info -->
                                    <div class="mb-3">
                                        <p class="text-muted mb-2">
                                            <i class="fas fa-building me-2"></i>{{ $card->company_name }}
                                        </p>
                                        @if($card->location)
                                            <p class="text-muted mb-2">
                                                <i class="fas fa-map-marker-alt me-2"></i>{{ $card->location }}
                                            </p>
                                        @endif
                                        @if($card->specialties)
                                            <p class="text-muted mb-2">
                                                <i class="fas fa-tools me-2"></i>{{ $card->formatted_specialties }}
                                            </p>
                                        @endif
                                    </div>

                                    <!-- Statistics -->
                                    <div class="row text-center mb-3">
                                        <div class="col-6">
                                            <div class="stat-box">
                                                <div class="stat-number">{{ $card->view_count }}</div>
                                                <div class="stat-label">Views</div>
                                            </div>
                                        </div>
                                        <div class="col-6">
                                            <div class="stat-box">
                                                <div class="stat-number">
                                                    {{ $card->created_at->diffForHumans() }}
                                                </div>
                                                <div class="stat-label">Created</div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Action Buttons -->
                                    <div class="d-flex gap-2">
                                        <a href="{{ $card->public_url }}" target="_blank" 
                                           class="btn btn-outline-primary btn-sm flex-fill">
                                            <i class="fas fa-eye me-1"></i>View
                                        </a>
                                        <a href="{{ route('business-cards.edit', $card) }}" 
                                           class="btn btn-outline-secondary btn-sm flex-fill">
                                            <i class="fas fa-edit me-1"></i>Edit
                                        </a>
                                        <button class="btn btn-outline-secondary btn-sm action-menu-btn" 
                                                onclick="showActionMenu(event, {{ $card->id }}, '{{ $card->public_url }}', {{ $card->is_active ? 'true' : 'false' }})">
                                            <i class="fas fa-ellipsis-v"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>
            @else
                <!-- Empty State -->
                <div class="text-center py-5">
                    <div class="mb-4">
                        <i class="fas fa-id-card fa-5x text-muted"></i>
                    </div>
                    <h4 class="text-muted">No Business Cards Yet</h4>
                    <p class="text-muted mb-4">
                        Create your first digital business card to start sharing your professional profile
                    </p>
                    <a href="{{ route('business-cards.create') }}" class="btn btn-primary">
                        <i class="fas fa-plus me-2"></i>Create Your First Card
                    </a>
                </div>
            @endif
        </div>
    </div>
</div>

<!-- Custom Action Menu Overlay -->
<div id="actionMenuOverlay" class="action-menu-overlay" onclick="hideActionMenu()">
    <div class="action-menu" onclick="event.stopPropagation()">
        <div class="action-menu-item" onclick="copyCardLink()">
            <i class="fas fa-copy"></i>
            <span>Copy Link</span>
        </div>
        <div class="action-menu-item" onclick="downloadVCard()">
            <i class="fas fa-download"></i>
            <span>Download vCard</span>
        </div>
        <div class="action-menu-divider"></div>
        <div class="action-menu-item" onclick="toggleCardStatus()">
            <i class="fas fa-eye" id="statusIcon"></i>
            <span id="statusText">Activate</span>
        </div>
        <div class="action-menu-item danger" onclick="deleteCurrentCard()">
            <i class="fas fa-trash"></i>
            <span>Delete</span>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                Are you sure you want to delete this business card? This action cannot be undone.
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmDelete">Delete</button>
            </div>
        </div>
    </div>
</div>

<!-- Delete Form -->
<form id="deleteForm" method="POST" style="display: none;">
    @csrf
    @method('DELETE')
</form>

@endsection

@section('custom-scripts')

<style>
.business-card-preview {
    border: 1px solid #e9ecef;
    transition: all 0.3s ease;
}

.business-card-preview:hover {
    transform: translateY(-3px);
    box-shadow: 0 4px 15px rgba(0,0,0,0.1) !important;
}

.initials-container {
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 18px;
    color: white;
    text-transform: uppercase;
}

.stat-box {
    padding: 10px;
    border-radius: 8px;
    background: #f8f9fa;
}

.stat-number {
    font-size: 16px;
    font-weight: bold;
    color: #495057;
}

.stat-label {
    font-size: 12px;
    color: #6c757d;
    text-transform: uppercase;
}

.bg-light {
    background: linear-gradient(135deg, #f8f9ff 0%, #f0f4ff 100%) !important;
}

.btn-primary {
    background: linear-gradient(45deg, #007bff, #0056b3);
    border: none;
}

.btn-primary:hover {
    background: linear-gradient(45deg, #0056b3, #004085);
    transform: translateY(-1px);
}

/* Custom Action Menu Overlay */
.action-menu-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100vw;
    height: 100vh;
    background: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(2px);
    z-index: 9999;
    display: none;
    align-items: center;
    justify-content: center;
    opacity: 0;
    transition: opacity 0.2s ease;
}

.action-menu-overlay.show {
    display: flex;
    opacity: 1;
}

.action-menu {
    background: white;
    border-radius: 12px;
    box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2);
    min-width: 200px;
    overflow: hidden;
    transform: scale(0.9);
    transition: transform 0.2s ease;
}

.action-menu-overlay.show .action-menu {
    transform: scale(1);
}

.action-menu-item {
    display: flex;
    align-items: center;
    padding: 15px 20px;
    cursor: pointer;
    transition: all 0.2s ease;
    border-bottom: 1px solid #f0f0f0;
}

.action-menu-item:last-child {
    border-bottom: none;
}

.action-menu-item:hover {
    background: #f8f9fa;
    transform: translateX(5px);
}

.action-menu-item.danger {
    color: #dc3545;
}

.action-menu-item.danger:hover {
    background: #f8d7da;
}

.action-menu-item i {
    width: 20px;
    margin-right: 12px;
    font-size: 14px;
    text-align: center;
}

.action-menu-item span {
    font-size: 14px;
    font-weight: 500;
}

.action-menu-divider {
    height: 1px;
    background: #e9ecef;
    margin: 8px 0;
}

.action-menu-btn {
    position: relative;
    transition: all 0.2s ease;
}

.action-menu-btn:hover {
    background: #e9ecef !important;
    transform: scale(1.1);
}

.action-menu-btn:active {
    transform: scale(0.95);
}

.alert {
    border-left: 4px solid;
}

.alert-success {
    border-left-color: #28a745;
}

.alert-danger {
    border-left-color: #dc3545;
}

@media (max-width: 768px) {
    .d-flex.gap-2 {
        flex-direction: column;
        gap: 0.5rem !important;
    }
    
    .action-menu {
        margin: 20px;
        width: calc(100vw - 40px);
        max-width: 300px;
    }
    
    .action-menu-item {
        padding: 18px 20px;
        font-size: 16px;
    }
}

/* Pulse animation for action button */
@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.05); }
    100% { transform: scale(1); }
}

.action-menu-btn.pulse {
    animation: pulse 0.3s ease;
}
</style>

<script>
// Global variables for current card
let currentCardId = null;
let currentCardUrl = null;
let currentCardActive = null;

// Show action menu at cursor position
function showActionMenu(event, cardId, cardUrl, isActive) {
    event.stopPropagation();
    
    // Store current card data
    currentCardId = cardId;
    currentCardUrl = cardUrl;
    currentCardActive = isActive;
    
    // Update menu items based on card status
    const statusIcon = document.getElementById('statusIcon');
    const statusText = document.getElementById('statusText');
    
    if (isActive) {
        statusIcon.className = 'fas fa-eye-slash';
        statusText.textContent = 'Deactivate';
    } else {
        statusIcon.className = 'fas fa-eye';
        statusText.textContent = 'Activate';
    }
    
    // Add pulse effect to button
    event.target.classList.add('pulse');
    setTimeout(() => event.target.classList.remove('pulse'), 300);
    
    // Show overlay menu
    const overlay = document.getElementById('actionMenuOverlay');
    overlay.classList.add('show');
    
    // Prevent body scroll
    document.body.style.overflow = 'hidden';
}

// Hide action menu
function hideActionMenu() {
    const overlay = document.getElementById('actionMenuOverlay');
    overlay.classList.remove('show');
    document.body.style.overflow = 'auto';
    
    // Reset current card data
    currentCardId = null;
    currentCardUrl = null;
    currentCardActive = null;
}

// Copy card link
function copyCardLink() {
    if (!currentCardUrl) return;
    
    navigator.clipboard.writeText(currentCardUrl).then(function() {
        showToast('Link copied to clipboard!', 'success');
        hideActionMenu();
    }, function() {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = currentCardUrl;
        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();
        try {
            document.execCommand('copy');
            showToast('Link copied to clipboard!', 'success');
        } catch (err) {
            showToast('Failed to copy link', 'error');
        }
        document.body.removeChild(textArea);
        hideActionMenu();
    });
}

// Download vCard
function downloadVCard() {
    if (!currentCardId) return;
    
    window.location.href = `/business-cards/${currentCardId}/download-vcard`;
    hideActionMenu();
}

// Toggle card status
function toggleCardStatus() {
    if (!currentCardId) return;
    
    fetch(`/business-cards/${currentCardId}/toggle-status`, {
        method: 'PATCH',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast(data.message, 'success');
            hideActionMenu();
            setTimeout(() => location.reload(), 1000);
        } else {
            showToast('Failed to update status', 'error');
        }
    })
    .catch(error => {
        showToast('An error occurred', 'error');
    });
}

// Delete current card
function deleteCurrentCard() {
    hideActionMenu();
    
    if (!currentCardId) return;
    
    const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
    modal.show();
}

// Confirm delete
document.getElementById('confirmDelete').addEventListener('click', function() {
    if (currentCardId) {
        const form = document.getElementById('deleteForm');
        form.action = `/business-cards/${currentCardId}`;
        form.submit();
    }
});

// Show toast notification
function showToast(message, type) {
    const toastHTML = `
        <div class="toast align-items-center text-white bg-${type === 'success' ? 'success' : 'danger'} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    let toastContainer = document.querySelector('.toast-container');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
        toastContainer.style.zIndex = '10000';
        document.body.appendChild(toastContainer);
    }
    
    toastContainer.insertAdjacentHTML('beforeend', toastHTML);
    
    const toastElement = toastContainer.lastElementChild;
    const toast = new bootstrap.Toast(toastElement);
    toast.show();
    
    toastElement.addEventListener('hidden.bs.toast', function() {
        toastElement.remove();
    });
}

// Close menu on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        hideActionMenu();
    }
});

// Prevent menu from closing when clicking inside
document.querySelector('.action-menu').addEventListener('click', function(e) {
    e.stopPropagation();
});
</script>

@endsection