@extends('SuperAdmin.layout.layout')
@section('title',"Leader Board")
@section('page-name',"Leader Board")
@section('page-detail',"Your Dealership Ranking Information")
@section('pagelink', 'leaderboard')
@section('body')
    <div class="content-are-dashboard p-3">
        <div class="card rounded-3">
            <div class="card-body">
                <div class="border rounded-3 p-3 leaderboard-container">
                    <div class="row align-items-center">
                        <div class="col-md-5 col-lg-4 mb-3">
                            <h4>Leader Board</h4>
                        </div>
                        <div class="col-md-7 ms-auto mb-3">
                            <div class="d-flex justify-content-end gap-2 align-items-center flex-wrap">
                                <!-- Advanced Category Filter -->
                                <div class="filter-group" id="categoryFilterGroup" style="display: none;">
                                    <label class="form-label small">Specific Category</label>
                                    <select class="form-select shadow-none" id="categoryFilter" style="width: 150px" onchange="filterBySpecificCategory()">
                                        <option value="all">All Categories</option>
                                        @foreach($availableCategories as $category)
                                            <option value="{{ $category }}">{{ $category }}</option>
                                        @endforeach
                                    </select>
                                </div>
                                
                                <!-- Month Dropdown -->
                                <div class="filter-group">
                                    <label class="form-label small">Month</label>
                                    <select class="form-select shadow-none" style="width: 150px" id="dateFilter" onchange="filterByDate()">
                                        @foreach($months as $month)
                                            <option value="{{ $month }}">{{ \Carbon\Carbon::parse($month)->format('M-y') }}</option>
                                        @endforeach
                                    </select>
                                </div>
                                
                                <!-- Filter Dropdown -->
                                <div class="filter-group">
                                    <label class="form-label small">View</label>
                                    <select class="form-select shadow-none" style="width: 150px" id="filter" onchange="switchBoards()">
                                        <option value="emp">Employees</option>
                                        <option value="dep">Department</option>
                                        <option value="cat">Category</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    <!-- Department Leader Board -->
                    <div class="row d-none" id="depLeaderBoard">
                        @include('partials.departmentLeaderBoard', ['departments' => $departments])
                    </div>


                    <!-- Employee Leader Board -->
                    <div class="row" id="empLeaderBoard">
                        @include('partials.employeeLeaderBoard', ['employees' => $employees])
                    </div>

                    <!-- Category Leader Board -->
                    <div class="row d-none" id="catLeaderBoard">
                        @include('partials.categoryLeaderBoard', ['categories' => $categories])
                    </div>

                    <!-- Dealership Leader Board -->
                    <div class="row d-none" id="dealerLeaderBoard">
                        {{--                        @include('partials.dealershipLeaderBoard', ['dealerships' => $dealerships])--}}
                    </div>

                </div>
            </div>
        </div>
    </div>
@endsection

@section('custom-scripts')
    <script>

        //Function to switch between Leader Boards
        function switchBoards() {
            let choice = $('#filter').val();
            const categoryFilterGroup = document.getElementById('categoryFilterGroup');
            
            // Show/hide category filter based on view selection
            if (choice === 'cat') {
                categoryFilterGroup.style.display = 'flex';
                // When category view is selected, show employee leaderboard for category filtering
                $('#empLeaderBoard').removeClass('d-none');
                $('#depLeaderBoard').addClass('d-none');
                $('#catLeaderBoard').addClass('d-none');
            } else {
                categoryFilterGroup.style.display = 'none';
                // Reset category filter when switching away from category view
                document.getElementById('categoryFilter').value = 'all';
            }
            
            if(choice == 'emp')
            {
                $('#empLeaderBoard').removeClass('d-none');
                $('#depLeaderBoard').addClass('d-none');
                $('#catLeaderBoard').addClass('d-none');
            }
            else if(choice == 'dep')
            {
                $('#depLeaderBoard').removeClass('d-none');
                $('#empLeaderBoard').addClass('d-none');
                $('#catLeaderBoard').addClass('d-none');
            }
            else if(choice == 'cat')
            {
                // Category filtering is handled above
                // This will show employee leaderboard with category filtering
            }
            filterByDate();
        }

        //Function to get Filtered Data By Date
        function filterByDate() {
            let selectedMonth = $('#dateFilter').val();
            let choice = $('#filter').val();
            if(choice == 'emp')
            {
                let url = "{{route('empFilter.leaderboard', ['currentMonth' => ':currentMonth'])}}".replace(':currentMonth', selectedMonth);
                console.log(url);
                $.ajax({
                    url: url, // Create this route in Laravel
                    type: "GET",
                    success: function(response) {
                        $("#empLeaderBoard").html(response); // Replace the leaderboard div with the updated content
                    },
                    error: function(xhr) {
                        console.error("Error fetching leaderboard:", xhr.responseText);
                    }
                });
            }
            else if(choice == 'dep')
            {
                let url = "{{ route('depFilter.leaderboard', ['currentMonth' => ':currentMonth']) }}".replace(':currentMonth', selectedMonth);
                $.ajax({
                    url: url, // Create this route in Laravel
                    type: "GET",
                    success: function(response) {
                        $("#depLeaderBoard").html(response); // Replace the leaderboard div with the updated content
                    },
                    error: function(xhr) {
                        console.error("Error fetching leaderboard:", xhr.responseText);
                    }
                });
            }
            else
            {
                let url = "{{ route('catFilter.leaderboard', ['currentMonth' => ':currentMonth']) }}".replace(':currentMonth', selectedMonth);
                $.ajax({
                    url: url, // Create this route in Laravel
                    type: "GET",
                    success: function(response) {
                        $("#catLeaderBoard").html(response); // Replace the leaderboard div with the updated content
                    },
                    error: function(xhr) {
                        console.error("Error fetching leaderboard:", xhr.responseText);
                    }
                });
            }
        }

        // Enhanced Leaderboard with Advanced Filtering
        let currentPrizes = {};
        let currentCategory = 'all';
        let currentMonth = '{{ $months[0] ?? date('Y-m') }}';
        
        // Load prizes when page loads
        $(document).ready(function() {
            loadPrizes();
        });
        
        async function loadPrizes() {
            try {
                const response = await fetch('/super-admin/prizes', {
                    method: 'GET',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                });
                
                const result = await response.json();
                
                if (result.success) {
                    currentPrizes = result.data.reduce((acc, prize) => {
                        if (!acc[prize.position]) {
                            acc[prize.position] = [];
                        }
                        acc[prize.position].push(prize);
                        return acc;
                    }, {});
                    
                    // Add prize display to existing leaderboard
                    addPrizeDisplayToLeaderboard();
                }
            } catch (error) {
                console.error('Error loading prizes:', error);
            }
        }
        
        function addPrizeDisplayToLeaderboard() {
            // Add prize display to top 3 positions in existing leaderboard
            $('.leaderboard-item').each(function(index) {
                if (index < 3) {
                    const position = index + 1;
                    const prize = getPrizeForPosition(position);
                    
                    if (prize) {
                        const prizeHtml = generatePrizeHTML(prize);
                        $(this).find('.employee-info, .department-info, .category-info').append(prizeHtml);
                    }
                }
            });
        }
        
        function getPrizeForPosition(position) {
            return currentPrizes[position] ? currentPrizes[position][0] : null;
        }
        
        function generatePrizeHTML(prize) {
            let html = `
                <div class="prize-info mt-2">
                    <div class="prize-badge">
                        <i class="fas fa-trophy me-1"></i>
                        <strong>${prize.title}</strong>
                    </div>
            `;
            
            if (prize.prize_amount) {
                html += `
                    <div class="prize-amount">
                        ${prize.formatted_amount || prize.currency + ' ' + parseFloat(prize.prize_amount).toFixed(2)}
                    </div>
                `;
            }
            
            if (prize.description) {
                html += `
                    <div class="prize-description">
                        <small>${prize.description}</small>
                    </div>
                `;
            }
            
            html += `</div>`;
            return html;
        }
        
        // Specific Category Filtering Function (for when Category view is selected)
        function filterBySpecificCategory() {
            const category = document.getElementById('categoryFilter').value;
            const currentMonth = document.getElementById('dateFilter').value;
            console.log('Filtering by specific category:', category, 'for month:', currentMonth);
            
            // Use the existing employee leaderboard route with category filtering
            let url = '/super-admin/leader-board/emp-leader-board/' + currentMonth;
            
            console.log('Loading URL:', url);
            
            // Show loading state
            showLoadingState();
            
            // Load the filtered content using existing route
            $.ajax({
                url: url,
                method: 'GET',
                data: {
                    category: category // Pass category as parameter
                },
                success: function(response) {
                    console.log('Successfully loaded category filter data');
                    // Replace the employee leaderboard content
                    $('#empLeaderBoard').html(response);
                    hideLoadingState();
                },
                error: function(xhr, status, error) {
                    console.error('Error loading category filter:', error);
                    showErrorMessage('Failed to load filtered data for ' + category);
                    hideLoadingState();
                }
            });
        }
        
        function showAllCategories() {
            // Show all categories without filtering
            $('#catLeaderBoard').removeClass('d-none');
            // Reset any filtered content
            $('.col-lg-8 .row').html(''); // Clear any custom ranking content
        }
        
        function updateCategoryLeaderboardWithFilteredData(data, category) {
            try {
                // Update the category leaderboard with filtered data
                $('#catLeaderBoard').removeClass('d-none');
                
                // Update top 3 positions with filtered data
                updateTop3Positions(data.top3 || [], category);
                
                // Update complete ranking for this specific category
                updateCompleteRankingForCategory(data.complete_ranking || [], category);
            } catch (error) {
                console.error('Error updating category leaderboard:', error);
                showErrorMessage('Error updating category leaderboard display');
            }
        }
        
        function updateCompleteRankingForCategory(ranking, category) {
            try {
                // Update the complete ranking section for specific category
                const rankingContainer = $('.col-lg-8 .row');
                if (rankingContainer.length > 0) {
                    let rankingHtml = `
                        <div class="col-12">
                            <div class="border rounded-3 p-3">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h5>Complete ${category} Ranking</h5>
                                    <span class="badge bg-primary">${ranking.length} Total</span>
                                </div>
                    `;
                    
                    if (ranking.length > 0) {
                        rankingHtml += `
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Rank</th>
                                            <th>Name</th>
                                            <th>Category</th>
                                            <th>Points</th>
                                            <th>Performance</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                        `;
                        
                        ranking.forEach((performer, index) => {
                            const rank = index + 4; // Starting from 4th place
                            const points = performer.earned_points || 0;
                            const progressWidth = Math.min((points / 100) * 100, 100);
                            
                            rankingHtml += `
                                <tr>
                                    <td><span class="badge bg-secondary">${rank}</span></td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <img src="${performer.image || '/assets/img/placeholder-img-2.png'}" 
                                                 alt="${performer.employee_name || performer.department_name}" 
                                                 class="rounded-circle me-2" width="32" height="32">
                                            <span>${performer.employee_name || performer.department_name || 'N/A'}</span>
                                        </div>
                                    </td>
                                    <td>${performer.category || 'N/A'}</td>
                                    <td><strong>${points}</strong></td>
                                    <td>
                                        <div class="progress" style="width: 100px;">
                                            <div class="progress-bar" style="width: ${progressWidth}%"></div>
                                        </div>
                                    </td>
                                </tr>
                            `;
                        });
                        
                        rankingHtml += `
                                    </tbody>
                                </table>
                            </div>
                        `;
                    } else {
                        rankingHtml += `
                            <div class="text-center py-4">
                                <p class="text-muted">No additional ${category} rankings available.</p>
                            </div>
                        `;
                    }
                    
                    rankingHtml += `
                            </div>
                        </div>
                    `;
                    
                    rankingContainer.html(rankingHtml);
                }
            } catch (error) {
                console.error('Error updating complete ranking for category:', error);
            }
        }

        // Advanced Category Filtering Function
        async function filterByCategory() {
            const category = document.getElementById('categoryFilter').value;
            currentCategory = category;
            
            console.log('Filtering by category:', category);
            
            // If "all" is selected, just show the original leaderboard
            if (category === 'all') {
                console.log('Showing original leaderboard...');
                showOriginalLeaderboard();
                return;
            }
            
            try {
                // Show loading state
                showLoadingState();
                
                // Get CSRF token
                const csrfToken = document.querySelector('meta[name="csrf-token"]');
                if (!csrfToken) {
                    throw new Error('CSRF token not found');
                }
                
                console.log('Making AJAX request to:', '/super-admin/leader-board/filtered');
                
                // Create a timeout promise
                const timeoutPromise = new Promise((_, reject) => {
                    setTimeout(() => reject(new Error('Request timeout')), 10000); // 10 second timeout
                });
                
                // Fetch filtered data based on category
                const fetchPromise = fetch('/super-admin/leader-board/filtered', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken.getAttribute('content')
                    },
                    body: JSON.stringify({
                        category: category,
                        time_period: 'current_month',
                        month: currentMonth
                    })
                });
                
                // Race between fetch and timeout
                const response = await Promise.race([fetchPromise, timeoutPromise]);
                
                console.log('Response status:', response.status);
                
                const result = await response.json();
                console.log('Response data:', result);
                
                if (result.success) {
                    console.log('Successfully received filtered data');
                    // Update the leaderboard with filtered data
                    updateLeaderboardWithFilteredData(result, category);
                } else {
                    console.error('Error filtering leaderboard:', result.message);
                    showErrorMessage('Failed to load filtered data');
                }
            } catch (error) {
                console.error('Error in filterByCategory:', error);
                showErrorMessage('An error occurred while filtering');
            } finally {
                // Always hide loading state
                hideLoadingState();
            }
        }
        
        function showOriginalLeaderboard() {
            // Show the original leaderboard without filtering
            $('#empLeaderBoard, #depLeaderBoard, #catLeaderBoard').addClass('d-none');
            
            // Show the appropriate leaderboard based on current view
            const currentView = document.getElementById('filter').value;
            let targetBoard = '';
            
            switch(currentView) {
                case 'emp':
                    targetBoard = '#empLeaderBoard';
                    break;
                case 'dep':
                    targetBoard = '#depLeaderBoard';
                    break;
                case 'cat':
                    targetBoard = '#catLeaderBoard';
                    break;
            }
            
            if (targetBoard) {
                $(targetBoard).removeClass('d-none');
            }
        }
        
        function updateLeaderboardWithFilteredData(data, category) {
            try {
                // Hide all existing leaderboards
                $('#empLeaderBoard, #depLeaderBoard, #catLeaderBoard').addClass('d-none');
                
                // Show the appropriate leaderboard based on current filter
                const currentView = document.getElementById('filter').value;
                let targetBoard = '';
                
                switch(currentView) {
                    case 'emp':
                        targetBoard = '#empLeaderBoard';
                        break;
                    case 'dep':
                        targetBoard = '#depLeaderBoard';
                        break;
                    case 'cat':
                        targetBoard = '#catLeaderBoard';
                        break;
                }
                
                // Update the leaderboard content
                if (targetBoard) {
                    $(targetBoard).removeClass('d-none');
                    
                    // Update top 3 positions with filtered data
                    updateTop3Positions(data.top3 || [], category);
                    
                    // Update complete ranking
                    updateCompleteRanking(data.complete_ranking || [], category);
                }
            } catch (error) {
                console.error('Error updating leaderboard:', error);
                showErrorMessage('Error updating leaderboard display');
            }
        }
        
        function updateTop3Positions(top3, category) {
            try {
                // Update the top 3 positions with new data
                $('.leaderboard-item').each(function(index) {
                    if (index < 3) {
                        const performer = top3[index] || null;
                        const position = index + 1;
                        const prize = getPrizeForPosition(position);
                        
                        // Update the content
                        const infoDiv = $(this).find('.employee-info, .department-info, .category-info');
                        
                        // Clear existing prize info
                        infoDiv.find('.prize-info').remove();
                        
                        if (performer) {
                            // Update name and points
                            infoDiv.find('h6').text(performer.category || performer.department || performer.employee_name || 'N/A');
                            infoDiv.find('.count').text((performer.earned_points || 'N/A') + ' pts');
                            infoDiv.find('.u-name').text(performer.employee_name || performer.department_name || performer.manager || 'N/A');
                            
                            // Add prize info if available
                            if (prize) {
                                const prizeHtml = generatePrizeHTML(prize);
                                infoDiv.append(prizeHtml);
                            }
                        } else {
                            // No data for this position
                            infoDiv.find('h6').text('N/A');
                            infoDiv.find('.count').text('0 pts');
                            infoDiv.find('.u-name').text('No data');
                        }
                    }
                });
            } catch (error) {
                console.error('Error updating top 3 positions:', error);
            }
        }
        
        function updateCompleteRanking(ranking, category) {
            try {
                // Update the complete ranking section
                const rankingContainer = $('.col-lg-8 .row');
                if (rankingContainer.length > 0) {
                    let rankingHtml = `
                        <div class="col-12">
                            <div class="border rounded-3 p-3">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h5>Complete ${category} Ranking</h5>
                                    <span class="badge bg-primary">${ranking.length} Total</span>
                                </div>
                    `;
                    
                    if (ranking.length > 0) {
                        rankingHtml += `
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Rank</th>
                                            <th>Name</th>
                                            <th>Category</th>
                                            <th>Points</th>
                                            <th>Performance</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                        `;
                        
                        ranking.forEach((performer, index) => {
                            const rank = index + 4; // Starting from 4th place
                            const points = performer.earned_points || 0;
                            const progressWidth = Math.min((points / 100) * 100, 100);
                            
                            rankingHtml += `
                                <tr>
                                    <td><span class="badge bg-secondary">${rank}</span></td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <img src="${performer.image || '/assets/img/placeholder-img-2.png'}" 
                                                 alt="${performer.employee_name || performer.department_name}" 
                                                 class="rounded-circle me-2" width="32" height="32">
                                            <span>${performer.employee_name || performer.department_name || 'N/A'}</span>
                                        </div>
                                    </td>
                                    <td>${performer.category || 'N/A'}</td>
                                    <td><strong>${points}</strong></td>
                                    <td>
                                        <div class="progress" style="width: 100px;">
                                            <div class="progress-bar" style="width: ${progressWidth}%"></div>
                                        </div>
                                    </td>
                                </tr>
                            `;
                        });
                        
                        rankingHtml += `
                                    </tbody>
                                </table>
                            </div>
                        `;
                    } else {
                        rankingHtml += `
                            <div class="text-center py-4">
                                <p class="text-muted">No additional rankings available for this category.</p>
                            </div>
                        `;
                    }
                    
                    rankingHtml += `
                            </div>
                        </div>
                    `;
                    
                    rankingContainer.html(rankingHtml);
                }
            } catch (error) {
                console.error('Error updating complete ranking:', error);
            }
        }
        
        function showLoadingState() {
            // Show loading indicator
            console.log('Showing loading state...');
            $('.leaderboard-container').append('<div class="loading-overlay"><div class="spinner-border text-primary" role="status"><span class="visually-hidden">Loading...</span></div></div>');
        }
        
        function hideLoadingState() {
            // Hide loading indicator
            console.log('Hiding loading state...');
            $('.loading-overlay').remove();
        }
        
        function showErrorMessage(message) {
            // Show error message
            alert(message); // You can replace this with a better notification system
        }

    </script>

    <style>
        /* Filter Group Styles */
        .filter-group {
            display: flex;
            flex-direction: column;
            margin-right: 15px;
        }
        
        .filter-group .form-label {
            margin-bottom: 5px;
            font-weight: 500;
            color: #6c757d;
            font-size: 0.875rem;
        }
        
        /* Prize Display Styles */
        .prize-info {
            background: linear-gradient(135deg, #ffd700, #ffed4e);
            border-radius: 8px;
            padding: 8px;
            margin-top: 10px;
            border: 2px solid #ffd700;
        }
        
        .prize-badge {
            font-size: 0.8rem;
            margin-bottom: 5px;
            color: #333;
        }
        
        .prize-badge i {
            color: #ff6b35;
        }
        
        .prize-amount {
            font-size: 1.1rem;
            font-weight: bold;
            color: #2c3e50;
        }
        
        .prize-description {
            font-size: 0.7rem;
            color: #555;
            margin-top: 3px;
        }

        /* Enhanced styling for top 3 positions */
        .leaderboard-item:nth-child(1) .prize-info {
            background: linear-gradient(135deg, #ffd700, #ffed4e);
            border-color: #ffd700;
        }
        
        .leaderboard-item:nth-child(2) .prize-info {
            background: linear-gradient(135deg, #c0c0c0, #e8e8e8);
            border-color: #c0c0c0;
        }
        
        .leaderboard-item:nth-child(3) .prize-info {
            background: linear-gradient(135deg, #cd7f32, #daa520);
            border-color: #cd7f32;
        }
        
        /* Loading Overlay Styles */
        .loading-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.8);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
        }
        
        .leaderboard-container {
            position: relative;
        }
        
        /* Responsive design */
        @media (max-width: 768px) {
            .prize-info {
                padding: 6px;
                margin-top: 6px;
            }

            .prize-badge {
                font-size: 0.8em;
            }

            .prize-amount {
                font-size: 1em;
            }

            .prize-description {
                font-size: 0.75em;
            }
        }
    </style>
@endsection
