@php
    $admin = session('owner');
@endphp
<!-- header -->
<style>
    .notification-container {
        position: relative;
        display: inline-block;
    }

    .notification-bell {
        position: relative;
        background: none;
        border: none;
        font-size: 20px;
        color: #6b7280;
        cursor: pointer;
        padding: 8px;
        border-radius: 6px;
        transition: color 0.2s;
    }

    .notification-bell:hover {
        color: #374151;
        background-color: #f3f4f6;
    }

    .notification-badge {
        position: absolute;
        top: 2px;
        right: 2px;
        background-color: #ef4444;
        color: white;
        border-radius: 50%;
        width: 18px;
        height: 18px;
        font-size: 11px;
        display: none;
        align-items: center;
        justify-content: center;
        font-weight: bold;
    }

    .notification-badge.show {
        display: flex;
    }

    .notification-dropdown {
        position: absolute;
        top: 100%;
        right: 0;
        background: white;
        border: 1px solid #e5e7eb;
        border-radius: 8px;
        box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        width: 350px;
        max-height: 400px;
        z-index: 1000;
        display: none;
    }

    .notification-dropdown.show {
        display: block;
    }

    .notification-header {
        padding: 16px;
        border-bottom: 1px solid #e5e7eb;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .notification-title {
        font-size: 18px;
        font-weight: 600;
        color: #111827;
        margin: 0;
    }

    .mark-all-read {
        background: none;
        border: none;
        color: #3b82f6;
        cursor: pointer;
        font-size: 14px;
        padding: 4px 8px;
        border-radius: 4px;
        transition: background-color 0.2s;
    }

    .mark-all-read:hover {
        background-color: #eff6ff;
    }

    .notification-list {
        max-height: 300px;
        overflow-y: auto;
    }

    .notification-item {
        padding: 12px 16px;
        border-bottom: 1px solid #f3f4f6;
        cursor: pointer;
        transition: background-color 0.2s;
        display: flex;
        align-items: start;
        gap: 12px;
    }

    .notification-item:hover {
        background-color: #f9fafb;
    }

    .notification-item.unread {
        background-color: #eff6ff;
    }

    .notification-icon {
        width: 8px;
        height: 8px;
        border-radius: 50%;
        background-color: #3b82f6;
        margin-top: 6px;
        flex-shrink: 0;
    }

    .notification-icon.read {
        background-color: transparent;
    }

    .notification-content {
        flex: 1;
    }

    .notification-item-title {
        font-weight: 600;
        color: #111827;
        font-size: 14px;
        margin: 0 0 4px 0;
    }

    .notification-item-content {
        color: #6b7280;
        font-size: 13px;
        margin: 0 0 4px 0;
        line-height: 1.4;
    }

    .notification-time {
        color: #9ca3af;
        font-size: 12px;
    }

    .notification-actions {
        display: flex;
        gap: 8px;
        margin-top: 8px;
    }

    .notification-action {
        background: none;
        border: none;
        color: #6b7280;
        cursor: pointer;
        font-size: 12px;
        padding: 2px 4px;
        border-radius: 4px;
        transition: color 0.2s;
    }

    .notification-action:hover {
        color: #374151;
    }

    .notification-footer {
        padding: 12px 16px;
        border-top: 1px solid #e5e7eb;
        text-align: center;
    }

    .view-all-link {
        color: #3b82f6;
        text-decoration: none;
        font-size: 14px;
        font-weight: 500;
    }

    .view-all-link:hover {
        text-decoration: underline;
    }

    .no-notifications {
        padding: 32px 16px;
        text-align: center;
        color: #6b7280;
    }

    .no-notifications i {
        font-size: 32px;
        margin-bottom: 8px;
        display: block;
    }

    .notification-toggle {
        margin-right: 10px;
        transition: all 0.3s ease;
        border-radius: 6px;
        padding: 8px 12px;
        font-size: 14px;
        font-weight: 500;
    }

    .notification-toggle:disabled {
        cursor: not-allowed;
        opacity: 0.6;
    }

    .error-message {
        padding: 16px;
        text-align: center;
        color: #ef4444;
    }

    .error-message i {
        font-size: 24px;
        margin-bottom: 8px;
        display: block;
    }

    .chat-notification-item {
        padding: 10px 15px;
        border-bottom: 1px solid #eee;
        cursor: pointer;
        transition: background-color 0.2s;
    }

    .chat-notification-item:hover {
        background-color: #f8f9fa;
    }

    .chat-notification-item:last-child {
        border-bottom: none;
    }

    .notification-sender {
        font-weight: bold;
        color: #333;
    }
</style>

<header class="py-2">
    <div class="container-fluid px-md-5">
        <div class="d-flex align-items-center justify-content-between">

            @if (View::hasSection('sub-page-name'))
                <div class="ps-md-5">
                    <h3 class="mb-0">@yield('page-name')</h3>
                    <div class="d-flex">
                        <p class="p-light mb-0">@yield('page-name')</p>
                        <a href="#">
                            <i class="fas fa-angle-right px-3"></i>
                        </a>
                        <a href="#">
                            <p>@yield('sub-page-name')</p>
                        </a>
                    </div>
                </div>
            @else
                <div class="ps-md-5">
                    <h3 class="mb-0">@yield('page-name')</h3>
                    <div class="d-flex">
                        <p class="p-light mb-0">@yield('page-detail')</p>
                    </div>
                </div>
            @endif

            <div class="d-flex align-items-center">
                <!-- Push Notification Toggle Button -->
                <button type="button" id="notificationToggle" class="notification-toggle btn btn-outline-primary">
                    <i class="fas fa-bell-slash"></i> Enable Notifications
                </button>

                <!-- Notification Bell -->
                <div class="notification-container">
                    <button class="notification-bell" id="notificationBell">
                        <i class="fas fa-bell"></i>
                        <span class="notification-badge" id="notificationBadge">0</span>
                    </button>

                    <div class="notification-dropdown" id="notificationDropdown">
                        <div class="notification-header">
                            <h3 class="notification-title">Notifications</h3>
                            <button class="mark-all-read" id="markAllRead">Mark all read</button>
                        </div>

                        <div class="notification-list" id="notificationList">
                            <div class="no-notifications" id="noNotifications">
                                <i class="fas fa-bell-slash"></i>
                                <p>No notifications</p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- User Dropdown -->
                <div class="dropdown">
                    <a href="#" class="d-flex c-dropdown ms-0 align-items-center text-decoration-none dropdown-toggle"
                        id="userDropdown" role="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <img src="{{ asset('assets') }}/img/user.png" alt="User Image" class="me-2">
                        <div class="d-flex flex-column text-start">
                            <span class="fw-bold">{{$admin['name']}}</span>
                            <small class="p-light">Super Admin</small>
                        </div>
                    </a>
                    <ul class="dropdown-menu dropdown-menu-end ms-0 w-100 c-dropdown mt-1"
                        aria-labelledby="userDropdown">
                        <li><a class="dropdown-item text-danger bg-transparent" href="{{route('logout')}}">Logout</a>
                        </li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</header>

<script>
// Global Laravel configuration with CSRF token
window.Laravel = {!! json_encode([
    'csrfToken' => csrf_token(),
]) !!};

class PushNotificationManager {
    constructor() {
        this.isSupported = 'serviceWorker' in navigator && 'PushManager' in window;
        this.registration = null;
        this.subscription = null;
        this.vapidPublicKey = null;
        
        this.elements = {
            toggle: document.getElementById('notificationToggle')
        };

        this.init();
    }

    async init() {
        if (!this.isSupported) {
            this.updateToggleButton('not-supported');
            return false;
        }

        try {
            await this.registerServiceWorker();
            await this.loadStatus();
            await this.checkPermissionAndSubscription();
            this.bindEvents();
            
            return true;
        } catch (error) {
            console.error('Failed to initialize push notifications:', error);
            this.updateToggleButton('error');
            return false;
        }
    }

    async registerServiceWorker() {
        if (!this.registration) {
            this.registration = await navigator.serviceWorker.register('/sw.js');
            console.log('Service Worker registered:', this.registration);
        }
    }

    async loadStatus() {
        try {
            const response = await this.makeRequest('/chat/push/status', {
                method: 'GET'
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.vapidPublicKey = data.vapid_public_key;
                    
                    if (data.subscribed) {
                        this.updateToggleButton('subscribed');
                    } else {
                        this.updateToggleButton('granted-unsubscribed');
                    }
                }
            }
        } catch (error) {
            console.error('Failed to load push status:', error);
        }
    }

    async checkPermissionAndSubscription() {
        const permission = Notification.permission;
        
        if (permission === 'denied') {
            this.updateToggleButton('denied');
            return;
        }

        if (permission === 'granted') {
            this.subscription = await this.registration.pushManager.getSubscription();
            
            if (this.subscription) {
                await this.verifySubscriptionWithBackend();
            }
        }
    }

    bindEvents() {
        if (this.elements.toggle) {
            this.elements.toggle.addEventListener('click', () => this.handleToggleClick());
        }
    }

    async handleToggleClick() {
        const permission = Notification.permission;
        
        try {
            this.elements.toggle.disabled = true;
            this.updateToggleButton('loading');

            if (permission === 'default') {
                await this.requestPermissionAndSubscribe();
            } else if (permission === 'granted') {
                if (this.subscription) {
                    await this.unsubscribe();
                } else {
                    await this.subscribe();
                }
            }
        } catch (error) {
            console.error('Toggle click error:', error);
            this.showError('Failed to update notification settings');
        } finally {
            this.elements.toggle.disabled = false;
        }
    }

    async requestPermissionAndSubscribe() {
        const permission = await Notification.requestPermission();
        
        if (permission === 'granted') {
            await this.subscribe();
        } else if (permission === 'denied') {
            this.updateToggleButton('denied');
            this.showError('Notifications were blocked. Please enable them in your browser settings.');
        } else {
            this.updateToggleButton('default');
        }
    }

    async subscribe() {
        try {
            if (!this.vapidPublicKey) {
                throw new Error('VAPID public key not available');
            }

            const applicationServerKey = this.urlBase64ToUint8Array(this.vapidPublicKey);
            
            this.subscription = await this.registration.pushManager.subscribe({
                userVisibleOnly: true,
                applicationServerKey: applicationServerKey
            });

            await this.sendSubscriptionToBackend();
            
            this.updateToggleButton('subscribed');
            this.showSuccess('Push notifications enabled successfully!');
            
        } catch (error) {
            console.error('Failed to subscribe:', error);
            this.updateToggleButton('error');
            this.showError('Failed to enable push notifications');
        }
    }

    async unsubscribe() {
        try {
            if (this.subscription) {
                await this.removeSubscriptionFromBackend();
                await this.subscription.unsubscribe();
                
                this.subscription = null;
                this.updateToggleButton('granted-unsubscribed');
                this.showSuccess('Push notifications disabled successfully!');
            }
        } catch (error) {
            console.error('Failed to unsubscribe:', error);
            this.showError('Failed to disable push notifications');
        }
    }

    async sendSubscriptionToBackend() {
        const subscriptionData = {
            endpoint: this.subscription.endpoint,
            keys: {
                p256dh: this.arrayBufferToBase64(this.subscription.getKey('p256dh')),
                auth: this.arrayBufferToBase64(this.subscription.getKey('auth'))
            },
            contentEncoding: 'aes128gcm'
        };

        const response = await this.makeRequest('/chat/push/subscribe', {
            method: 'POST',
            body: JSON.stringify(subscriptionData)
        });

        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.error || 'Failed to save subscription to backend');
        }

        const result = await response.json();
        console.log('Subscription saved to backend:', result);
    }

    async removeSubscriptionFromBackend() {
        const response = await this.makeRequest('/chat/push/unsubscribe', {
            method: 'POST',
            body: JSON.stringify({
                endpoint: this.subscription.endpoint
            })
        });

        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.error || 'Failed to remove subscription from backend');
        }

        const result = await response.json();
        console.log('Subscription removed from backend:', result);
    }

    async verifySubscriptionWithBackend() {
        try {
            const response = await this.makeRequest('/chat/push/status', {
                method: 'GET'
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success && data.subscribed) {
                    this.updateToggleButton('subscribed');
                } else {
                    await this.sendSubscriptionToBackend();
                    this.updateToggleButton('subscribed');
                }
            }
        } catch (error) {
            console.error('Failed to verify subscription:', error);
        }
    }

    async makeRequest(url, options = {}) {
        const defaultOptions = {
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': this.getCSRFToken()
            }
        };

        const finalOptions = {
            ...defaultOptions,
            ...options,
            headers: {
                ...defaultOptions.headers,
                ...(options.headers || {})
            }
        };

        try {
            const response = await fetch(url, finalOptions);
            
            if (response.status === 419) {
                console.warn('CSRF token mismatch, attempting to refresh...');
                await this.refreshCSRFToken();
                
                finalOptions.headers['X-CSRF-TOKEN'] = this.getCSRFToken();
                return await fetch(url, finalOptions);
            }
            
            return response;
        } catch (error) {
            console.error('Request failed:', error);
            throw error;
        }
    }

    async refreshCSRFToken() {
        try {
            const response = await fetch('/csrf-token', {
                method: 'GET',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                
                const metaTag = document.querySelector('meta[name="csrf-token"]');
                if (metaTag) {
                    metaTag.setAttribute('content', data.token);
                }
                
                if (window.Laravel) {
                    window.Laravel.csrfToken = data.token;
                }
                
                console.log('CSRF token refreshed successfully');
            }
        } catch (error) {
            console.error('Failed to refresh CSRF token:', error);
        }
    }

    updateToggleButton(state) {
        if (!this.elements.toggle) return;

        const button = this.elements.toggle;
        
        switch (state) {
            case 'loading':
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
                button.className = 'notification-toggle btn btn-secondary';
                button.disabled = true;
                break;
                
            case 'not-supported':
                button.innerHTML = '<i class="fas fa-times"></i> Not Supported';
                button.className = 'notification-toggle btn btn-secondary';
                button.disabled = true;
                break;
                
            case 'denied':
                button.innerHTML = '<i class="fas fa-bell-slash"></i> Notifications Blocked';
                button.className = 'notification-toggle btn btn-danger';
                button.disabled = true;
                button.title = 'Notifications are blocked. Please enable them in your browser settings.';
                break;
                
            case 'subscribed':
                button.innerHTML = '<i class="fas fa-bell"></i> Notifications On';
                button.className = 'notification-toggle btn btn-success';
                button.disabled = false;
                button.title = 'Click to disable push notifications';
                break;
                
            case 'granted-unsubscribed':
                button.innerHTML = '<i class="fas fa-bell-slash"></i> Enable Notifications';
                button.className = 'notification-toggle btn btn-outline-primary';
                button.disabled = false;
                button.title = 'Click to enable push notifications';
                break;
                
            case 'default':
                button.innerHTML = '<i class="fas fa-bell-slash"></i> Enable Notifications';
                button.className = 'notification-toggle btn btn-outline-primary';
                button.disabled = false;
                button.title = 'Click to enable push notifications';
                break;
                
            case 'error':
                button.innerHTML = '<i class="fas fa-exclamation-triangle"></i> Error';
                button.className = 'notification-toggle btn btn-warning';
                button.disabled = false;
                break;
        }
    }

    urlBase64ToUint8Array(base64String) {
        const padding = '='.repeat((4 - base64String.length % 4) % 4);
        const base64 = (base64String + padding)
            .replace(/\-/g, '+')
            .replace(/_/g, '/');

        const rawData = window.atob(base64);
        const outputArray = new Uint8Array(rawData.length);

        for (let i = 0; i < rawData.length; ++i) {
            outputArray[i] = rawData.charCodeAt(i);
        }
        return outputArray;
    }

    arrayBufferToBase64(buffer) {
        const bytes = new Uint8Array(buffer);
        let binary = '';
        for (let i = 0; i < bytes.byteLength; i++) {
            binary += String.fromCharCode(bytes[i]);
        }
        return window.btoa(binary);
    }

    getCSRFToken() {
        const metaToken = document.querySelector('meta[name="csrf-token"]');
        if (metaToken && metaToken.getAttribute('content')) {
            return metaToken.getAttribute('content');
        }
        
        if (window.Laravel && window.Laravel.csrfToken) {
            return window.Laravel.csrfToken;
        }
        
        const formToken = document.querySelector('input[name="_token"]');
        if (formToken) {
            return formToken.value;
        }
        
        console.error('CSRF token not found in meta tag, Laravel object, or form inputs');
        return '';
    }

    showSuccess(message) {
        this.showToast(message, 'success');
    }

    showError(message) {
        this.showToast(message, 'error');
    }

    showToast(message, type) {
        const toast = document.createElement('div');
        toast.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 12px 20px;
            border-radius: 6px;
            color: white;
            font-weight: 500;
            z-index: 10000;
            animation: slideIn 0.3s ease;
            ${type === 'success' ? 'background-color: #059669;' : 'background-color: #dc2626;'}
        `;
        toast.textContent = message;
        
        document.body.appendChild(toast);
        
        setTimeout(() => {
            toast.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => {
                if (document.body.contains(toast)) {
                    document.body.removeChild(toast);
                }
            }, 300);
        }, 3000);
    }
}

class NotificationManager {
    constructor() {
        this.elements = {
            bell: document.getElementById('notificationBell'),
            badge: document.getElementById('notificationBadge'),
            dropdown: document.getElementById('notificationDropdown'),
            list: document.getElementById('notificationList'),
            markAllRead: document.getElementById('markAllRead'),
            loading: document.getElementById('loadingMessage')
        };

        this.isLoading = false;
        this.init();
    }

    init() {
        this.bindEvents();
        this.loadUnreadCount();
        this.startPolling();
    }

    bindEvents() {
        this.elements.bell.addEventListener('click', (e) => {
            e.stopPropagation();
            this.toggleDropdown();
        });

        this.elements.markAllRead.addEventListener('click', () => {
            this.markAllAsRead();
        });

        document.addEventListener('click', (e) => {
            if (!e.target.closest('.notification-container')) {
                this.hideDropdown();
            }
        });

        this.elements.list.addEventListener('click', (e) => {
            const notificationItem = e.target.closest('.notification-item');
            if (notificationItem) {
                const notificationId = notificationItem.dataset.id;

                if (e.target.classList.contains('mark-read-btn')) {
                    e.stopPropagation();
                    this.markAsRead(notificationId, notificationItem);
                    return;
                }

                if (e.target.classList.contains('delete-btn')) {
                    e.stopPropagation();
                    this.deleteNotification(notificationId);
                    return;
                }

                this.handleNotificationClick(notificationId, notificationItem);
            }
        });
    }

    toggleDropdown() {
        const isShowing = this.elements.dropdown.classList.contains('show');

        if (isShowing) {
            this.hideDropdown();
        } else {
            this.showDropdown();
            this.loadNotifications();
        }
    }

    showDropdown() {
        this.elements.dropdown.classList.add('show');
    }

    hideDropdown() {
        this.elements.dropdown.classList.remove('show');
    }

    async loadNotifications() {
        if (this.isLoading) return;

        this.isLoading = true;
        this.showLoading();

        try {
            const response = await this.makeRequest('/notifications/get?limit=10', {
                method: 'GET'
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const data = await response.json();

            if (data.success) {
                this.updateNotificationUI(data.notifications, data.unread_count);
            } else {
                this.showError('Failed to load notifications');
            }
        } catch (error) {
            console.error('Failed to load notifications:', error);
            this.showError('Unable to load notifications. Please try again.');
        } finally {
            this.isLoading = false;
            this.hideLoading();
        }
    }

    async loadUnreadCount() {
        try {
            const response = await this.makeRequest('/notifications/unread-count', {
                method: 'GET'
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const data = await response.json();

            if (data.success) {
                this.updateBadge(data.count);
            }
        } catch (error) {
            console.error('Failed to load unread count:', error);
        }
    }

    async makeRequest(url, options = {}) {
        const defaultOptions = {
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': this.getCSRFToken()
            }
        };

        const finalOptions = {
            ...defaultOptions,
            ...options,
            headers: {
                ...defaultOptions.headers,
                ...(options.headers || {})
            }
        };

        return await fetch(url, finalOptions);
    }

    updateNotificationUI(notifications, unreadCount) {
        this.updateBadge(unreadCount);
        this.renderNotifications(notifications);
    }

    updateBadge(count) {
        const badge = this.elements.badge;

        if (count > 0) {
            badge.textContent = count > 99 ? '99+' : count;
            badge.classList.add('show');
        } else {
            badge.classList.remove('show');
        }
    }

    renderNotifications(notifications) {
        const list = this.elements.list;

        if (notifications.length === 0) {
            list.innerHTML = `
                <div class="no-notifications">
                    <i class="fas fa-bell-slash"></i>
                    <p>No notifications</p>
                </div>
            `;
            return;
        }

        list.innerHTML = notifications.map(notification => `
            <div class="notification-item ${!notification.is_read ? 'unread' : ''}" data-id="${notification.id}">
                <div class="notification-icon ${notification.is_read ? 'read' : ''}"></div>
                <div class="notification-content">
                    <div class="notification-item-title">${this.escapeHtml(notification.title)}</div>
                    <div class="notification-item-content">${this.escapeHtml(notification.content)}</div>
                    ${notification.sender_name ? `<div class="notification-item-content"><small>From: ${this.escapeHtml(notification.sender_name)}</small></div>` : ''}
                    <div class="notification-time">${notification.time_ago}</div>
                    <div class="notification-actions">
                        ${!notification.is_read ? '<button class="notification-action mark-read-btn">Mark read</button>' : ''}
                        <button class="notification-action delete-btn">Delete</button>
                    </div>
                </div>
            </div>
        `).join('');
    }

    async handleNotificationClick(notificationId, element) {
        if (element.classList.contains('unread')) {
            const success = await this.markAsRead(notificationId, element);
            if (success) {
                element.classList.remove('unread');
                element.querySelector('.notification-icon').classList.add('read');

                const markReadBtn = element.querySelector('.mark-read-btn');
                if (markReadBtn) {
                    markReadBtn.remove();
                }

                this.loadUnreadCount();
            }
        }
    }

    async markAsRead(notificationId, element = null) {
        try {
            const response = await this.makeRequest(`/notifications/${notificationId}/read`, {
                method: 'POST'
            });

            const data = await response.json();

            if (data.success && element) {
                element.classList.remove('unread');
                element.querySelector('.notification-icon').classList.add('read');

                const markReadBtn = element.querySelector('.mark-read-btn');
                if (markReadBtn) {
                    markReadBtn.remove();
                }
            }

            return data.success;
        } catch (error) {
            console.error('Failed to mark notification as read:', error);
            return false;
        }
    }

    async markAllAsRead() {
        try {
            const response = await this.makeRequest('/notifications/mark-all-read', {
                method: 'POST'
            });

            const data = await response.json();

            if (data.success) {
                document.querySelectorAll('.notification-item').forEach(item => {
                    item.classList.remove('unread');
                    item.querySelector('.notification-icon').classList.add('read');

                    const markReadBtn = item.querySelector('.mark-read-btn');
                    if (markReadBtn) {
                        markReadBtn.remove();
                    }
                });

                this.updateBadge(0);
            }
        } catch (error) {
            console.error('Failed to mark all notifications as read:', error);
            this.showError('Failed to mark all as read');
        }
    }

    async deleteNotification(notificationId) {
        try {
            const response = await this.makeRequest(`/notifications/${notificationId}`, {
                method: 'DELETE'
            });

            const data = await response.json();

            if (data.success) {
                const element = document.querySelector(`[data-id="${notificationId}"]`);
                if (element) {
                    element.remove();
                }

                this.loadUnreadCount();

                const remainingItems = this.elements.list.querySelectorAll('.notification-item');
                if (remainingItems.length === 0) {
                    this.elements.list.innerHTML = `
                        <div class="no-notifications">
                            <i class="fas fa-bell-slash"></i>
                            <p>No notifications</p>
                        </div>
                    `;
                }
            }
        } catch (error) {
            console.error('Failed to delete notification:', error);
            this.showError('Failed to delete notification');
        }
    }

    showLoading() {
        if (this.elements.loading) {
            this.elements.loading.style.display = 'block';
        }
    }

    hideLoading() {
        if (this.elements.loading) {
            this.elements.loading.style.display = 'none';
        }
    }

    showError(message) {
        this.elements.list.innerHTML = `
            <div class="error-message">
                <i class="fas fa-exclamation-triangle"></i>
                <p>${message}</p>
            </div>
        `;
    }

    startPolling() {
        setInterval(() => {
            this.loadUnreadCount();
        }, 30000);

        document.addEventListener('visibilitychange', () => {
            if (!document.hidden) {
                this.loadUnreadCount();
            }
        });
    }

    getCSRFToken() {
        const metaToken = document.querySelector('meta[name="csrf-token"]');
        if (metaToken && metaToken.getAttribute('content')) {
            return metaToken.getAttribute('content');
        }
        
        if (window.Laravel && window.Laravel.csrfToken) {
            return window.Laravel.csrfToken;
        }
        
        const formToken = document.querySelector('input[name="_token"]');
        if (formToken) {
            return formToken.value;
        }
        
        console.error('CSRF token not found');
        return '';
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text || '';
        return div.innerHTML;
    }
}

// Initialize everything when DOM is loaded
document.addEventListener('DOMContentLoaded', async function() {
    window.pushNotificationManager = new PushNotificationManager();
    window.notificationManager = new NotificationManager();
    
    console.log('All notification systems initialized for Super Admin');
});

// Add CSS for animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
    }
    
    @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(100%); opacity: 0; }
    }
`;
document.head.appendChild(style);
</script>

@yield('custom-header')
<!-- header -->