@extends('SuperAdmin.layout.layout')
@section('title',"Prize Management")
@section('page-name',"Prize Management")
@section('page-detail',"Manage prizes for top 3 leaderboard positions")
@section('pagelink', 'prize-management')
@section('body')
    <div class="content-are-dashboard p-3">
        <div class="card rounded-3">
            <div class="card-body">
                <div class="border rounded-3 p-3">
                    <div class="row align-items-center mb-4">
                        <div class="col-md-6">
                            <h4>Prize Management</h4>
                            <p class="text-muted">Assign and manage prizes for top 3 leaderboard positions</p>
                        </div>
                        <div class="col-md-6 text-end">
                            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addPrizeModal">
                                <i class="fas fa-plus me-2"></i>Add New Prize
                            </button>
                        </div>
                    </div>

                    <!-- Filter Controls -->
                    <div class="row mb-4">
                        <div class="col-md-12">
                            <div class="d-flex gap-3 align-items-center flex-wrap">
                                <div class="filter-group">
                                    <label class="form-label small">Category</label>
                                    <select class="form-select shadow-none" id="categoryFilter" style="width: 150px">
                                        <option value="all">All Categories</option>
                                        <option value="BDC">BDC</option>
                                        <option value="Service Advisor">Service Advisor</option>
                                        <option value="Department">Department</option>
                                    </select>
                                </div>
                                
                                <div class="filter-group">
                                    <label class="form-label small">Time Period</label>
                                    <select class="form-select shadow-none" id="timeFilter" style="width: 150px">
                                        <option value="all">All Periods</option>
                                        <option value="current_month">Current Month</option>
                                        <option value="last_month">Last Month</option>
                                        <option value="quarter">This Quarter</option>
                                        <option value="year">This Year</option>
                                    </select>
                                </div>
                                
                                <div class="filter-group">
                                    <label class="form-label small">Status</label>
                                    <select class="form-select shadow-none" id="statusFilter" style="width: 150px">
                                        <option value="all">All Status</option>
                                        <option value="active">Active</option>
                                        <option value="inactive">Inactive</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Prizes Table -->
                    <div class="table-responsive">
                        <table class="table table-hover" id="prizesTable">
                            <thead class="table-light">
                                <tr>
                                    <th>Position</th>
                                    <th>Title</th>
                                    <th>Type</th>
                                    <th>Amount</th>
                                    <th>Category</th>
                                    <th>Time Period</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody id="prizesTableBody">
                                <!-- Prizes will be loaded here -->
                            </tbody>
                        </table>
                    </div>

                    <!-- Loading State -->
                    <div id="loadingState" class="text-center py-5" style="display: none;">
                        <div class="spinner-border text-primary" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        <p class="mt-2">Loading prizes...</p>
                    </div>

                    <!-- Empty State -->
                    <div id="emptyState" class="text-center py-5" style="display: none;">
                        <i class="fas fa-trophy fa-3x text-muted mb-3"></i>
                        <h4>No Prizes Found</h4>
                        <p class="text-muted">No prizes match your current filters.</p>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addPrizeModal">
                            <i class="fas fa-plus me-2"></i>Add First Prize
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Prize Modal -->
    <div class="modal fade" id="addPrizeModal" tabindex="-1" aria-labelledby="addPrizeModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addPrizeModalLabel">Add New Prize</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="prizeForm">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="position" class="form-label">Position <span class="text-danger">*</span></label>
                                    <select class="form-select" id="position" name="position" required>
                                        <option value="">Select Position</option>
                                        <option value="1">1st Place</option>
                                        <option value="2">2nd Place</option>
                                        <option value="3">3rd Place</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="prize_type" class="form-label">Prize Type <span class="text-danger">*</span></label>
                                    <select class="form-select" id="prize_type" name="prize_type" required>
                                        <option value="">Select Type</option>
                                        <option value="money">Cash Prize</option>
                                        <option value="gift">Gift</option>
                                        <option value="recognition">Recognition</option>
                                        <option value="bonus">Bonus</option>
                                        <option value="voucher">Voucher</option>
                                        <option value="trophy">Trophy</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="title" class="form-label">Title <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="title" name="title" required>
                        </div>

                        <div class="mb-3">
                            <label for="description" class="form-label">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="prize_amount" class="form-label">Prize Amount</label>
                                    <input type="number" class="form-control" id="prize_amount" name="prize_amount" step="0.01" min="0">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="currency" class="form-label">Currency</label>
                                    <select class="form-select" id="currency" name="currency">
                                        <option value="USD">USD ($)</option>
                                        <option value="EUR">EUR (€)</option>
                                        <option value="GBP">GBP (£)</option>
                                        <option value="CAD">CAD (C$)</option>
                                        <option value="AUD">AUD (A$)</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="category" class="form-label">Category</label>
                                    <select class="form-select" id="category" name="category">
                                        <option value="">All Categories</option>
                                        <option value="BDC">BDC</option>
                                        <option value="Service Advisor">Service Advisor</option>
                                        <option value="Department">Department</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="time_period" class="form-label">Time Period <span class="text-danger">*</span></label>
                                    <select class="form-select" id="time_period" name="time_period" required>
                                        <option value="">Select Period</option>
                                        <option value="current_month">Current Month</option>
                                        <option value="last_month">Last Month</option>
                                        <option value="quarter">This Quarter</option>
                                        <option value="year">This Year</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="start_date" class="form-label">Start Date</label>
                                    <input type="date" class="form-control" id="start_date" name="start_date">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="end_date" class="form-label">End Date</label>
                                    <input type="date" class="form-control" id="end_date" name="end_date">
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary" id="savePrizeBtn">
                            <i class="fas fa-save me-2"></i>Save Prize
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p>Are you sure you want to delete this prize? This action cannot be undone.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-danger" id="confirmDeleteBtn">
                        <i class="fas fa-trash me-2"></i>Delete Prize
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('custom-scripts')
<script>
class PrizeManager {
    constructor() {
        this.currentFilters = {
            category: 'all',
            time_period: 'all',
            status: 'all'
        };
        this.editingPrizeId = null;
        this.deletePrizeId = null;
        
        this.initializeEventListeners();
        this.loadPrizes();
    }
    
    initializeEventListeners() {
        // Filter changes
        $('#categoryFilter, #timeFilter, #statusFilter').on('change', () => {
            this.updateFilters();
            this.loadPrizes();
        });
        
        // Form submission
        $('#prizeForm').on('submit', (e) => {
            e.preventDefault();
            this.savePrize();
        });
        
        // Modal events
        $('#addPrizeModal').on('hidden.bs.modal', () => {
            this.resetForm();
        });
        
        // Delete confirmation
        $('#confirmDeleteBtn').on('click', () => {
            this.deletePrize();
        });
    }
    
    updateFilters() {
        this.currentFilters = {
            category: $('#categoryFilter').val(),
            time_period: $('#timeFilter').val(),
            status: $('#statusFilter').val()
        };
    }
    
    async loadPrizes() {
        this.showLoading();
        
        try {
            const params = new URLSearchParams(this.currentFilters);
            const response = await fetch(`/super-admin/prizes?${params}`, {
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.renderPrizes(result.data);
            } else {
                this.showError('Failed to load prizes');
            }
        } catch (error) {
            console.error('Error loading prizes:', error);
            this.showError('An error occurred while loading prizes');
        } finally {
            this.hideLoading();
        }
    }
    
    renderPrizes(prizes) {
        const tbody = $('#prizesTableBody');
        tbody.empty();
        
        if (prizes.length === 0) {
            this.showEmptyState();
            return;
        }
        
        this.hideEmptyState();
        
        prizes.forEach(prize => {
            const row = this.createPrizeRow(prize);
            tbody.append(row);
        });
    }
    
    createPrizeRow(prize) {
        const statusBadge = prize.is_active 
            ? '<span class="badge bg-success">Active</span>'
            : '<span class="badge bg-secondary">Inactive</span>';
            
        const amountDisplay = prize.prize_amount 
            ? `${prize.currency} ${parseFloat(prize.prize_amount).toFixed(2)}`
            : 'N/A';
            
        const positionBadge = this.getPositionBadge(prize.position);
        
        return `
            <tr>
                <td>${positionBadge}</td>
                <td>
                    <div>
                        <strong>${prize.title}</strong>
                        ${prize.description ? `<br><small class="text-muted">${prize.description}</small>` : ''}
                    </div>
                </td>
                <td><span class="badge bg-info">${this.getPrizeTypeDisplay(prize.prize_type)}</span></td>
                <td>${amountDisplay}</td>
                <td>${prize.category || 'All'}</td>
                <td>${this.getTimePeriodDisplay(prize.time_period)}</td>
                <td>${statusBadge}</td>
                <td>
                    <div class="btn-group" role="group">
                        <button class="btn btn-sm btn-outline-primary" onclick="prizeManager.editPrize(${prize.id})" title="Edit">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-${prize.is_active ? 'warning' : 'success'}" onclick="prizeManager.toggleStatus(${prize.id})" title="${prize.is_active ? 'Deactivate' : 'Activate'}">
                            <i class="fas fa-${prize.is_active ? 'pause' : 'play'}"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="prizeManager.confirmDelete(${prize.id})" title="Delete">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `;
    }
    
    getPositionBadge(position) {
        const badges = {
            1: '<span class="badge bg-warning text-dark">1st Place</span>',
            2: '<span class="badge bg-secondary">2nd Place</span>',
            3: '<span class="badge bg-danger">3rd Place</span>'
        };
        return badges[position] || `Position ${position}`;
    }
    
    getPrizeTypeDisplay(type) {
        const types = {
            'money': 'Cash Prize',
            'gift': 'Gift',
            'recognition': 'Recognition',
            'bonus': 'Bonus',
            'voucher': 'Voucher',
            'trophy': 'Trophy'
        };
        return types[type] || type;
    }
    
    getTimePeriodDisplay(period) {
        const periods = {
            'current_month': 'Current Month',
            'last_month': 'Last Month',
            'quarter': 'This Quarter',
            'year': 'This Year'
        };
        return periods[period] || period;
    }
    
    async savePrize() {
        const formData = new FormData(document.getElementById('prizeForm'));
        const data = Object.fromEntries(formData.entries());
        
        // Convert empty strings to null for optional fields
        Object.keys(data).forEach(key => {
            if (data[key] === '') {
                data[key] = null;
            }
        });
        
        const saveBtn = $('#savePrizeBtn');
        const originalText = saveBtn.html();
        saveBtn.html('<i class="fas fa-spinner fa-spin me-2"></i>Saving...').prop('disabled', true);
        
        try {
            const url = this.editingPrizeId 
                ? `/super-admin/prizes/${this.editingPrizeId}`
                : '/super-admin/prizes';
                
            const method = this.editingPrizeId ? 'PUT' : 'POST';
            
            const response = await fetch(url, {
                method: method,
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showSuccess(result.message);
                $('#addPrizeModal').modal('hide');
                this.loadPrizes();
            } else {
                this.showError(result.message || 'Failed to save prize');
            }
        } catch (error) {
            console.error('Error saving prize:', error);
            this.showError('An error occurred while saving the prize');
        } finally {
            saveBtn.html(originalText).prop('disabled', false);
        }
    }
    
    editPrize(prizeId) {
        // Load prize data and populate form
        fetch(`/super-admin/prizes/${prizeId}`)
            .then(response => response.json())
            .then(result => {
                if (result.success) {
                    const prize = result.data;
                    this.editingPrizeId = prizeId;
                    
                    // Populate form
                    $('#position').val(prize.position);
                    $('#prize_type').val(prize.prize_type);
                    $('#title').val(prize.title);
                    $('#description').val(prize.description || '');
                    $('#prize_amount').val(prize.prize_amount || '');
                    $('#currency').val(prize.currency || 'USD');
                    $('#category').val(prize.category || '');
                    $('#time_period').val(prize.time_period);
                    $('#start_date').val(prize.start_date ? prize.start_date.split('T')[0] : '');
                    $('#end_date').val(prize.end_date ? prize.end_date.split('T')[0] : '');
                    
                    $('#addPrizeModalLabel').text('Edit Prize');
                    $('#addPrizeModal').modal('show');
                } else {
                    this.showError('Failed to load prize data');
                }
            })
            .catch(error => {
                console.error('Error loading prize:', error);
                this.showError('An error occurred while loading the prize');
            });
    }
    
    async toggleStatus(prizeId) {
        try {
            const response = await fetch(`/super-admin/prizes/${prizeId}/toggle`, {
                method: 'PATCH',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showSuccess(result.message);
                this.loadPrizes();
            } else {
                this.showError(result.message || 'Failed to update prize status');
            }
        } catch (error) {
            console.error('Error toggling prize status:', error);
            this.showError('An error occurred while updating the prize status');
        }
    }
    
    confirmDelete(prizeId) {
        this.deletePrizeId = prizeId;
        $('#deleteModal').modal('show');
    }
    
    async deletePrize() {
        if (!this.deletePrizeId) return;
        
        try {
            const response = await fetch(`/super-admin/prizes/${this.deletePrizeId}`, {
                method: 'DELETE',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showSuccess(result.message);
                $('#deleteModal').modal('hide');
                this.loadPrizes();
            } else {
                this.showError(result.message || 'Failed to delete prize');
            }
        } catch (error) {
            console.error('Error deleting prize:', error);
            this.showError('An error occurred while deleting the prize');
        } finally {
            this.deletePrizeId = null;
        }
    }
    
    resetForm() {
        document.getElementById('prizeForm').reset();
        this.editingPrizeId = null;
        $('#addPrizeModalLabel').text('Add New Prize');
    }
    
    showLoading() {
        $('#loadingState').show();
        $('#prizesTable').hide();
        this.hideEmptyState();
    }
    
    hideLoading() {
        $('#loadingState').hide();
        $('#prizesTable').show();
    }
    
    showEmptyState() {
        $('#emptyState').show();
        $('#prizesTable').hide();
    }
    
    hideEmptyState() {
        $('#emptyState').hide();
    }
    
    showSuccess(message) {
        // You can implement a toast notification here
        alert('Success: ' + message);
    }
    
    showError(message) {
        // You can implement a toast notification here
        alert('Error: ' + message);
    }
}

// Initialize the prize manager
let prizeManager;

$(document).ready(function() {
    prizeManager = new PrizeManager();
});
</script>

<style>
.filter-group {
    margin-bottom: 15px;
}

.filter-group label {
    font-weight: 600;
    color: #495057;
    margin-bottom: 5px;
}

.table th {
    border-top: none;
    font-weight: 600;
    color: #495057;
}

.btn-group .btn {
    margin-right: 2px;
}

.btn-group .btn:last-child {
    margin-right: 0;
}

.modal-lg {
    max-width: 800px;
}

.badge {
    font-size: 0.75em;
}

@media (max-width: 768px) {
    .filter-group {
        margin-bottom: 10px;
    }
    
    .btn-group {
        display: flex;
        flex-direction: column;
    }
    
    .btn-group .btn {
        margin-bottom: 2px;
        margin-right: 0;
    }
}
</style>
@endsection
