@extends('SuperAdmin.layout.layout')

@section('title', 'Custom Emoji & GIF Management')

@section('body')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="mb-0">
                        <i class="fas fa-smile me-2"></i>Custom Emoji & GIF Management
                    </h4>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#uploadModal">
                        <i class="fas fa-plus me-2"></i>Upload New
                    </button>
                </div>
                <div class="card-body">
                    @if(session('success'))
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            {{ session('success') }}
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    @endif

                    @if(session('error'))
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            {{ session('error') }}
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    @endif

                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Type</th>
                                    <th>Content</th>
                                    <th>Status</th>
                                    <th>Created By</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($emojis as $emoji)
                                    <tr>
                                        <td>{{ $emoji->id }}</td>
                                        <td>{{ $emoji->name }}</td>
                                        <td>
                                            <span class="badge bg-{{ $emoji->type === 'emoji' ? 'info' : 'success' }}">
                                                {{ ucfirst($emoji->type) }}
                                            </span>
                                        </td>
                                        <td>
                                            @if($emoji->type === 'emoji')
                                                <span style="font-size: 2rem;">{{ $emoji->emoji_text }}</span>
                                            @else
                                                <img src="{{ asset($emoji->file_path) }}" alt="{{ $emoji->name }}" style="max-height: 40px;">
                                            @endif
                                        </td>
                                        <td>
                                            <span class="badge bg-{{ $emoji->is_active ? 'success' : 'danger' }}">
                                                {{ $emoji->is_active ? 'Active' : 'Inactive' }}
                                            </span>
                                        </td>
                                        <td>{{ $emoji->creator_name }}</td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button type="button" class="btn btn-sm btn-warning" onclick="editEmoji({{ $emoji->id }})">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <button type="button" class="btn btn-sm btn-{{ $emoji->is_active ? 'secondary' : 'success' }}" onclick="toggleStatus({{ $emoji->id }})">
                                                    <i class="fas fa-{{ $emoji->is_active ? 'pause' : 'play' }}"></i>
                                                </button>
                                                <button type="button" class="btn btn-sm btn-danger" onclick="deleteEmoji({{ $emoji->id }})">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                @empty
                                    <tr>
                                        <td colspan="7" class="text-center text-muted">No emojis or GIFs found</td>
                                    </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Upload Modal -->
<div class="modal fade" id="uploadModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="uploadForm" method="POST" enctype="multipart/form-data" onsubmit="return false;">
                @csrf
                <div class="modal-header">
                    <h5 class="modal-title">Upload New Emoji/GIF</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="name" class="form-label">Name</label>
                        <input type="text" class="form-control" id="name" name="name" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="type" class="form-label">Type</label>
                        <select class="form-control" id="type" name="type" required onchange="toggleFields()">
                            <option value="">Select Type</option>
                            <option value="emoji">Emoji (Text)</option>
                            <option value="gif">GIF (Image)</option>
                        </select>
                    </div>
                    
                    <div class="mb-3" id="emojiField">
                        <label for="emoji_text" class="form-label">Emoji Text</label>
                        <input type="text" class="form-control" id="emoji_text" name="emoji_text" placeholder="Enter emoji (e.g., 😀, 🎉, 👍)">
                    </div>
                    
                    <div class="mb-3" id="fileField">
                        <label for="file" class="form-label">Upload File</label>
                        <input type="file" class="form-control" id="file" name="file" accept=".png,.jpg,.jpeg,.gif,.webp">
                        <small class="text-muted">Supported: PNG, JPG, JPEG, GIF, WEBP (Max: 2MB)</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary" onclick="handleFormSubmit(event)">
                        <i class="fas fa-upload me-2"></i>Upload
                    </button>
                    <button type="button" class="btn btn-warning ms-2" onclick="testFunction()">
                        Test
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Modal -->
<div class="modal fade" id="editModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="editForm" method="POST">
                @csrf
                @method('PUT')
                <input type="hidden" id="editId" name="id">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Emoji/GIF</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="editName" class="form-label">Name</label>
                        <input type="text" class="form-control" id="editName" name="name" required>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="editIsActive" name="is_active" value="1">
                            <label class="form-check-label" for="editIsActive">Active</label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</div>
<script>
// Test function
function testFunction() {
    console.log('=== TEST FUNCTION CALLED ===');
    alert('JavaScript is working!');
}

// Refresh emoji table function
function refreshEmojiTable() {
    console.log('Refreshing emoji table...');
    
    fetch('/super-admin/custom-emoji-management', {
        method: 'GET',
        headers: {
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.text())
    .then(html => {
        // Extract the table content from the response
        const parser = new DOMParser();
        const doc = parser.parseFromString(html, 'text/html');
        const newTable = doc.querySelector('.table-responsive');
        
        if (newTable) {
            // Replace the existing table
            const currentTable = document.querySelector('.table-responsive');
            if (currentTable) {
                currentTable.innerHTML = newTable.innerHTML;
                console.log('Emoji table refreshed successfully');
            }
        }
    })
    .catch(error => {
        console.error('Error refreshing table:', error);
        // Fallback to page reload if AJAX fails
        location.reload();
    });
}

// Toast notification function
function showToast(type, title, message) {
    // Create toast element
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type === 'success' ? 'success' : 'danger'} border-0" role="alert" aria-live="assertive" aria-atomic="true">
            <div class="d-flex">
                <div class="toast-body">
                    <strong>${title}</strong><br>
                    ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
        </div>
    `;
    
    // Create toast container if it doesn't exist
    let toastContainer = document.getElementById('toast-container');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.id = 'toast-container';
        toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
        toastContainer.style.zIndex = '9999';
        document.body.appendChild(toastContainer);
    }
    
    // Add toast to container
    toastContainer.insertAdjacentHTML('beforeend', toastHtml);
    
    // Show toast
    const toastElement = toastContainer.lastElementChild;
    const toast = new bootstrap.Toast(toastElement);
    toast.show();
    
    // Remove toast element after it's hidden
    toastElement.addEventListener('hidden.bs.toast', function() {
        toastElement.remove();
    });
}
// Handle form submission (backup function)
function handleFormSubmit(event) {
    event.preventDefault();
    event.stopPropagation();
    
    console.log('=== FORM SUBMISSION START ===');
    console.log('Form submission handled by onclick handler');
    console.log('Event:', event);
    console.log('Form element:', document.getElementById('uploadForm'));
    
    const form = document.getElementById('uploadForm');
    if (!form) {
        console.error('Form not found!');
        return;
    }
    
    const formData = new FormData(form);
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    // Show loading state
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Uploading...';
    submitBtn.disabled = true;
    
    // Submit form
    fetch('/super-admin/custom-emojis', {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => {
        console.log('Response received:', response);
        return response.json();
    })
    .then(data => {
        console.log('Data received:', data);
        if (data.success) {
            // Show success toast notification
            showToast('success', 'Success!', data.message);
            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('uploadModal'));
            if (modal) {
                modal.hide();
            }
            // Reset form
            form.reset();
            // Refresh emoji table in real-time
            refreshEmojiTable();
        } else {
            showToast('error', 'Error!', data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('error', 'Error!', 'Something went wrong');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
}

// Toggle fields based on type selection
function toggleFields() {
    const type = document.getElementById('type').value;
    const emojiField = document.getElementById('emojiField');
    const fileField = document.getElementById('fileField');
    
    // Show both fields by default
    emojiField.style.display = 'block';
    fileField.style.display = 'block';
    
    // Hide irrelevant field based on selection
    if (type === 'emoji') {
        fileField.style.display = 'none';
        document.getElementById('emoji_text').required = true;
        document.getElementById('file').required = false;
    } else if (type === 'gif') {
        emojiField.style.display = 'none';
        document.getElementById('file').required = true;
        document.getElementById('emoji_text').required = false;
    } else {
        // No type selected, show both fields
        document.getElementById('emoji_text').required = false;
        document.getElementById('file').required = false;
    }
}

// Form submission - use event delegation to handle dynamically loaded forms
document.addEventListener('submit', function(e) {
    console.log('=== SUBMIT EVENT DETECTED ===');
    console.log('Target ID:', e.target.id);
    console.log('Event:', e);
    
    if (e.target.id === 'uploadForm') {
    e.preventDefault();
    e.stopPropagation();
    
    console.log('=== UPLOAD FORM SUBMISSION START ===');
    console.log('Form submission started via JavaScript');
    console.log('Form element:', e.target);
    console.log('Event prevented:', e.defaultPrevented);
    
    const formData = new FormData(e.target);
    const submitBtn = e.target.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    // Show loading state
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Uploading...';
    submitBtn.disabled = true;
    
    // Submit form
    fetch('/super-admin/custom-emojis', {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => {
        console.log('Response received:', response);
        return response.json();
    })
    .then(data => {
        console.log('Data received:', data);
        if (data.success) {
            // Show success toast notification
            showToast('success', 'Success!', data.message);
            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('uploadModal'));
            if (modal) {
                modal.hide();
            }
            // Reset form
            form.reset();
            // Refresh emoji table in real-time
            refreshEmojiTable();
        } else {
            showToast('error', 'Error!', data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('error', 'Error!', 'Something went wrong');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
    }
});

// Edit form submission
document.getElementById('editForm').addEventListener('submit', function(e) {
    e.preventDefault();
    
    const emojiId = document.getElementById('editId').value;
    const formData = new FormData(this);
    const submitBtn = this.querySelector('button[type="submit"]');
    const originalText = submitBtn.innerHTML;
    
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Saving...';
    submitBtn.disabled = true;
    
    fetch(`/super-admin/custom-emojis/${emojiId}`, {
        method: 'PUT',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('success', 'Success!', data.message);
            // Close edit modal
            const editModal = bootstrap.Modal.getInstance(document.getElementById('editModal'));
            if (editModal) {
                editModal.hide();
            }
            refreshEmojiTable();
        } else {
            showToast('error', 'Error!', data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('error', 'Error!', 'Something went wrong');
    })
    .finally(() => {
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    });
});

// Edit emoji function
function editEmoji(id) {
    fetch(`/super-admin/custom-emojis/${id}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                document.getElementById('editId').value = data.emoji.id;
                document.getElementById('editName').value = data.emoji.name;
                document.getElementById('editIsActive').checked = data.emoji.is_active;
                
                const editModal = new bootstrap.Modal(document.getElementById('editModal'));
                editModal.show();
            } else {
                showToast('error', 'Error!', data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading emoji data');
        });
}

// Toggle status function
function toggleStatus(id) {
    if (confirm('Are you sure you want to change the status?')) {
        fetch(`/super-admin/custom-emojis/${id}/toggle`, {
            method: 'PATCH',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast('success', 'Success!', data.message);
                refreshEmojiTable();
            } else {
                showToast('error', 'Error!', data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showToast('error', 'Error!', 'Something went wrong');
        });
    }
}

// Delete emoji function
function deleteEmoji(id) {
    if (confirm('Are you sure you want to delete this emoji/GIF? This action cannot be undone.')) {
        fetch(`/super-admin/custom-emojis/${id}`, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast('success', 'Success!', data.message);
                refreshEmojiTable();
            } else {
                showToast('error', 'Error!', data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showToast('error', 'Error!', 'Something went wrong');
        });
    }
}

// Reset form when modal is closed
document.getElementById('uploadModal').addEventListener('hidden.bs.modal', function () {
    document.getElementById('uploadForm').reset();
    // Show both fields by default
    document.getElementById('emojiField').style.display = 'block';
    document.getElementById('fileField').style.display = 'block';
});
</script>
@endsection