@extends('Manager.layout')
@section('title',"Enhanced Leader Board")
@section('page-name',"Enhanced Leader Board")
@section('page-detail',"Advanced Rankings with Advanced Filtering")
@section('pagelink', 'enhanced-leaderboard')
@section('body')
    <div class="content-are-dashboard p-3">
        <div class="card rounded-3">
            <div class="card-body">
                <div class="border rounded-3 p-3">
                    <div class="row align-items-center">
                        <div class="col-md-5 col-lg-4 mb-3">
                            <h4>Enhanced Leader Board</h4>
                            <p class="text-muted">Advanced filtering and real-time rankings</p>
                        </div>
                        <div class="col-md-7 ms-auto mb-3">
                            <div class="d-flex justify-content-end gap-2 align-items-center flex-wrap">
                                <!-- Enhanced Filter Controls -->
                                <div class="filter-group">
                                    <label class="form-label small">Category</label>
                                    <select class="form-select shadow-none" id="categoryFilter" style="width: 150px">
                                        <option value="employees">All Employees</option>
                                        <option value="bdc">BDC</option>
                                        <option value="service_advisors">Service Advisors</option>
                                        <option value="departments">Departments</option>
                                        <option value="categories">Categories</option>
                                    </select>
                                </div>
                                
                                <div class="filter-group">
                                    <label class="form-label small">Time Period</label>
                                    <select class="form-select shadow-none" id="timeFilter" style="width: 150px">
                                        <option value="current_month">Current Month</option>
                                        <option value="last_month">Last Month</option>
                                        <option value="quarter">This Quarter</option>
                                        <option value="year">This Year</option>
                                    </select>
                                </div>
                                
                                <div class="filter-group">
                                    <label class="form-label small">Department</label>
                                    <select class="form-select shadow-none" id="departmentFilter" style="width: 150px">
                                        <option value="all">All Departments</option>
                                        @foreach($departments as $department)
                                            <option value="{{ $department->id }}">{{ $department->name }}</option>
                                        @endforeach
                                    </select>
                                </div>
                                
                                <!-- Legacy Month Filter (for compatibility) -->
                                <div class="filter-group">
                                    <label class="form-label small">Month</label>
                                    <select class="form-select shadow-none" id="dateFilter" style="width: 150px">
                                        @foreach($months as $month)
                                            <option value="{{ $month }}">{{ \Carbon\Carbon::parse($month)->format('M-y') }}</option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Enhanced Leaderboard Container -->
                    <div id="leaderboardContainer" class="row">
                        <!-- Content will be loaded here -->
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('custom-scripts')
<script>
// Enhanced leaderboard JavaScript for Manager
class ManagerLeaderboardManager {
    constructor() {
        this.currentFilters = {
            category: 'employees',
            time_period: 'current_month',
            department_id: 'all'
        };
        this.isLoading = false;
        this.cache = new Map();
        
        this.initializeEventListeners();
        this.loadInitialData();
    }
    
    initializeEventListeners() {
        // Category filter change
        $('#categoryFilter').on('change', (e) => {
            this.currentFilters.category = e.target.value;
            this.refreshLeaderboard();
        });
        
        // Time period filter change
        $('#timeFilter').on('change', (e) => {
            this.currentFilters.time_period = e.target.value;
            this.refreshLeaderboard();
        });
        
        // Department filter change
        $('#departmentFilter').on('change', (e) => {
            this.currentFilters.department_id = e.target.value;
            this.refreshLeaderboard();
        });
        
        // Month filter change (legacy support)
        $('#dateFilter').on('change', (e) => {
            this.currentFilters.month = e.target.value;
            this.refreshLeaderboard();
        });
    }
    
    async refreshLeaderboard() {
        if (this.isLoading) return;
        
        this.isLoading = true;
        this.showLoadingState();
        
        try {
            const cacheKey = this.generateCacheKey();
            
            // Check cache first
            if (this.cache.has(cacheKey)) {
                this.renderLeaderboard(this.cache.get(cacheKey));
                this.isLoading = false;
                this.hideLoadingState();
                return;
            }
            
            // Fetch fresh data
            const response = await this.fetchLeaderboardData();
            
            if (response.success) {
                // Cache the result
                this.cache.set(cacheKey, response);
                
                // Render the leaderboard
                this.renderLeaderboard(response);
                
                // Update UI state
                this.updateFilterUI();
            } else {
                this.showError('Failed to load leaderboard data');
            }
        } catch (error) {
            console.error('Manager Leaderboard refresh error:', error);
            this.showError('An error occurred while loading the leaderboard');
        } finally {
            this.isLoading = false;
            this.hideLoadingState();
        }
    }
    
    async fetchLeaderboardData() {
        const url = this.getApiUrl();
        
        const response = await fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(this.currentFilters)
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    }
    
    renderLeaderboard(response) {
        const container = $('#leaderboardContainer');
        
        // Clear existing content
        container.empty();
        
        if (response.data && response.data.length > 0) {
            // Render top 3 section
            this.renderTopPerformers(response.top3, this.currentFilters.category);
            
            // Render complete ranking
            this.renderCompleteRanking(response.complete_ranking, this.currentFilters.category);
        } else {
            this.renderNoData();
        }
        
        // Add smooth animations
        container.hide().fadeIn(300);
    }
    
    renderTopPerformers(top3, category) {
        const top3Html = this.generateTop3HTML(top3, category);
        $('#leaderboardContainer').append(`
            <div class="col-lg-4">
                <div class="border rounded-3 p-3 top-performers-section">
                    ${top3Html}
                </div>
            </div>
        `);
    }
    
    renderCompleteRanking(ranking, category) {
        const rankingHtml = this.generateRankingHTML(ranking, category);
        $('#leaderboardContainer').append(`
            <div class="col-lg-8">
                <div class="row">
                    <div class="col-12">
                        <div class="border p-3 rounded-3">
                            ${rankingHtml}
                        </div>
                    </div>
                </div>
            </div>
        `);
    }
    
    generateTop3HTML(top3, category) {
        let html = `
            <div class="text-center mb-4">
                <h4>Top 3 ${category}</h4>
                <p class="text-muted">Real-time performance rankings</p>
            </div>
            <div class="d-flex align-items-end justify-content-center pt-6">
        `;
        
        // Generate podium positions
        const positions = ['2nd', '1st', '3rd'];
        const classes = ['leader-box-left', 'leader-box-middle', 'leader-box-right'];
        
        positions.forEach((position, index) => {
            if (top3[index]) {
                const performer = top3[index];
                const isFirst = position === '1st';
                
                html += `
                    <div class="${classes[index]} position-relative">
                        <div class="position-absolute text-center">
                            ${isFirst ? '<img src="/assets/img/crown.png" alt="Crown" class="img-fluid mb-1">' : ''}
                            <div class="img-circle-area">
                                <img src="${performer.image || '/assets/img/placeholder-img-2.png'}" 
                                     alt="${performer.employee_name || performer.department_name}" 
                                     class="rounded-circle img-fluid">
                            </div>
                            <span class="rank-badge rank-${index + 1}">${index + 1}</span>
                        </div>
                        <div class="text-center">
                            <h6>${performer.category || 'N/A'}</h6>
                            <div class="count ${isFirst ? 'my-2' : ''}">${performer.earned_points || 'N/A'} pts</div>
                            <div class="u-name">${performer.employee_name || performer.department_name || 'N/A'}</div>
                        </div>
                    </div>
                `;
            }
        });
        
        html += `
            </div>
        `;
        
        return html;
    }
    
    generateRankingHTML(ranking, category) {
        let html = `
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5>Complete ${category} Ranking</h5>
                <span class="badge bg-primary">${ranking.length} Total</span>
            </div>
            
            <div class="table-responsive">
                <table class="table leader-table">
                    <thead>
                        <tr>
                            <th scope="col">Rank</th>
                            <th scope="col">${category}</th>
                            <th scope="col">Category</th>
                            <th scope="col">Points Earned</th>
                            <th scope="col">Performance</th>
                        </tr>
                    </thead>
                    <tbody>
        `;
        
        ranking.forEach((performer, index) => {
            const performancePercentage = this.calculatePerformancePercentage(performer, ranking);
            const name = performer.employee_name || performer.department_name || 'N/A';
            const image = performer.image || '/assets/img/placeholder-img-2.png';
            
            html += `
                <tr class="ranking-row" data-rank="${performer.rank}">
                    <td class="align-middle">
                        <div class="table-circle-no">${performer.rank}</div>
                    </td>
                    <td class="align-middle">
                        <div class="d-flex align-items-center">
                            <img src="${image}" 
                                 alt="${name}" 
                                 class="rounded-circle me-2" 
                                 style="width: 32px; height: 32px;">
                            ${name}
                        </div>
                    </td>
                    <td class="align-middle">
                        <span class="badge bg-secondary">${performer.category || 'N/A'}</span>
                    </td>
                    <td class="align-middle">
                        <strong>${performer.earned_points} pts</strong>
                    </td>
                    <td class="align-middle">
                        <div class="progress" style="height: 8px;">
                            <div class="progress-bar" style="width: ${performancePercentage}%"></div>
                        </div>
                    </td>
                </tr>
            `;
        });
        
        html += `
                    </tbody>
                </table>
            </div>
        `;
        
        return html;
    }
    
    calculatePerformancePercentage(performer, allData) {
        if (!allData || allData.length === 0) return 0;
        
        const maxPoints = Math.max(...allData.map(p => p.earned_points));
        if (maxPoints === 0) return 0;
        
        return Math.round((performer.earned_points / maxPoints) * 100);
    }
    
    getApiUrl() {
        return '/manager/leaderboard/filtered';
    }
    
    generateCacheKey() {
        return JSON.stringify(this.currentFilters);
    }
    
    showLoadingState() {
        $('#leaderboardContainer').html(`
            <div class="col-12">
                <div class="text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2">Loading leaderboard data...</p>
                </div>
            </div>
        `);
    }
    
    hideLoadingState() {
        // Loading state is replaced by actual content
    }
    
    showError(message) {
        $('#leaderboardContainer').html(`
            <div class="col-12">
                <div class="alert alert-danger text-center">
                    <i class="fas fa-exclamation-triangle"></i>
                    <p class="mb-0">${message}</p>
                    <button class="btn btn-sm btn-outline-danger mt-2" onclick="managerLeaderboardManager.refreshLeaderboard()">
                        Try Again
                    </button>
                </div>
            </div>
        `);
    }
    
    renderNoData() {
        $('#leaderboardContainer').html(`
            <div class="col-12">
                <div class="text-center py-5">
                    <i class="fas fa-chart-line fa-3x text-muted mb-3"></i>
                    <h4>No Data Available</h4>
                    <p class="text-muted">No leaderboard data found for the selected filters.</p>
                </div>
            </div>
        `);
    }
    
    updateFilterUI() {
        // Update filter UI to reflect current state
        $('#categoryFilter').val(this.currentFilters.category);
        $('#timeFilter').val(this.currentFilters.time_period);
        $('#departmentFilter').val(this.currentFilters.department_id);
    }
    
    loadInitialData() {
        this.refreshLeaderboard();
    }
}

// Initialize the manager leaderboard manager
let managerLeaderboardManager;

$(document).ready(function() {
    managerLeaderboardManager = new ManagerLeaderboardManager();
});
</script>

<style>
/* Enhanced leaderboard styling for Manager */
.filter-group {
    margin-bottom: 15px;
}

.filter-group label {
    font-weight: 600;
    color: #495057;
    margin-bottom: 5px;
}

.top-performers-section {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
}

.leader-box-left, .leader-box-middle, .leader-box-right {
    transition: all 0.3s ease;
    cursor: pointer;
}

.leader-box-left:hover, .leader-box-middle:hover, .leader-box-right:hover {
    transform: translateY(-5px);
    box-shadow: 0 12px 30px rgba(0,0,0,0.15);
}

.rank-badge {
    background: rgba(255,255,255,0.2);
    color: white;
    border-radius: 50%;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 14px;
    margin: 0 auto;
    margin-top: 10px;
}

.rank-1 {
    background: linear-gradient(45deg, #ffd700, #ffed4e);
    color: #333;
    box-shadow: 0 4px 15px rgba(255, 215, 0, 0.4);
}

.rank-2 {
    background: linear-gradient(45deg, #c0c0c0, #e8e8e8);
    color: #333;
    box-shadow: 0 4px 15px rgba(192, 192, 192, 0.4);
}

.rank-3 {
    background: linear-gradient(45deg, #cd7f32, #daa520);
    color: white;
    box-shadow: 0 4px 15px rgba(205, 127, 50, 0.4);
}

.ranking-row {
    transition: all 0.2s ease;
}

.ranking-row:hover {
    background-color: #f8f9fa;
    transform: translateX(5px);
}

.table-circle-no {
    background: linear-gradient(45deg, #007bff, #0056b3);
    color: white;
    border-radius: 50%;
    width: 35px;
    height: 35px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    margin: 0 auto;
}

.progress {
    background-color: #e9ecef;
    border-radius: 10px;
}

.progress-bar {
    background: linear-gradient(45deg, #28a745, #20c997);
    border-radius: 10px;
    transition: width 0.6s ease;
}

/* Responsive design */
@media (max-width: 768px) {
    .filter-group {
        margin-bottom: 10px;
    }
    
    .top-performers-section {
        padding: 20px;
    }
    
    .leader-box-left, .leader-box-middle, .leader-box-right {
        margin-bottom: 20px;
    }
}
</style>
@endsection
