<?php
namespace App\Services;

use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class YouTubeService
{
    public static function extractVideoId(string $url): ?string
    {
        $patterns = [
            '/(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/)([a-zA-Z0-9_-]{11})/',
            '/youtube\.com\/v\/([a-zA-Z0-9_-]{11})/',
            '/youtube\.com\/watch\?.*v=([a-zA-Z0-9_-]{11})/'
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $url, $matches)) {
                return $matches[1];
            }
        }

        return null;
    }

    public static function getVideoInfo(string $videoId): ?array
    {
        try {
            // Method 1: Using oEmbed API (no API key required)
            $oembedUrl = "https://www.youtube.com/oembed?url=https://www.youtube.com/watch?v={$videoId}&format=json";
            $response = Http::timeout(10)->get($oembedUrl);

            if ($response->successful()) {
                $data = $response->json();
                
                return [
                    'title' => $data['title'] ?? 'Unknown Title',
                    'thumbnail_url' => $data['thumbnail_url'] ?? "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg",
                    'channel_name' => $data['author_name'] ?? 'Unknown Channel',
                    'description' => 'Fetched from YouTube', // oEmbed doesn't provide description
                    'duration' => null // oEmbed doesn't provide duration
                ];
            }

            // Fallback: Basic info with default thumbnail
            return [
                'title' => 'YouTube Video',
                'thumbnail_url' => "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg",
                'channel_name' => 'Unknown Channel',
                'description' => 'YouTube Video',
                'duration' => null
            ];

        } catch (Exception $e) {
            Log::error('YouTube API Error: ' . $e->getMessage());
            
            // Return basic info as fallback
            return [
                'title' => 'YouTube Video',
                'thumbnail_url' => "https://img.youtube.com/vi/{$videoId}/maxresdefault.jpg",
                'channel_name' => 'Unknown Channel',
                'description' => 'YouTube Video',
                'duration' => null
            ];
        }
    }

    // Alternative method using YouTube Data API v3 (requires API key)
    public static function getVideoInfoWithAPI(string $videoId, string $apiKey): ?array
    {
        try {
            $url = "https://www.googleapis.com/youtube/v3/videos";
            $response = Http::get($url, [
                'part' => 'snippet,contentDetails,statistics',
                'id' => $videoId,
                'key' => $apiKey
            ]);

            if ($response->successful()) {
                $data = $response->json();
                
                if (empty($data['items'])) {
                    return null;
                }

                $video = $data['items'][0];
                $snippet = $video['snippet'];
                $contentDetails = $video['contentDetails'];

                return [
                    'title' => $snippet['title'],
                    'description' => $snippet['description'],
                    'thumbnail_url' => $snippet['thumbnails']['maxres']['url'] ?? 
                                    $snippet['thumbnails']['high']['url'] ?? 
                                    $snippet['thumbnails']['medium']['url'],
                    'channel_name' => $snippet['channelTitle'],
                    'duration' => self::parseDuration($contentDetails['duration'])
                ];
            }

            return null;
        } catch (Exception $e) {
            Log::error('YouTube API Error: ' . $e->getMessage());
            return null;
        }
    }

    private static function parseDuration(string $duration): int
    {
        // Parse ISO 8601 duration format (PT4M13S)
        preg_match('/PT(?:(\d+)H)?(?:(\d+)M)?(?:(\d+)S)?/', $duration, $matches);
        
        $hours = isset($matches[1]) ? (int)$matches[1] : 0;
        $minutes = isset($matches[2]) ? (int)$matches[2] : 0;
        $seconds = isset($matches[3]) ? (int)$matches[3] : 0;
        
        return ($hours * 3600) + ($minutes * 60) + $seconds;
    }
}

