<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\Employee;
use App\Models\Admin;
use App\Models\PushSubscription;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Minishlink\WebPush\WebPush;
use Minishlink\WebPush\Subscription;

class UniversalNotificationService
{
    private $webPush;

    public function __construct()
    {
        // Initialize WebPush with VAPID keys
        $this->webPush = new WebPush([
            'VAPID' => [
                'subject' => config('app.url'),
                'publicKey' => config('app.vapid_public_key'),
                'privateKey' => config('app.vapid_private_key'),
            ],
        ]);
    }

    /**
     * Send notification to single user (enhanced with web push)
     */
    public function sendToUser(array $params): bool
    {
        try {
            // Create database notification
            $notification = $this->createNotification([
                'tenant_id' => $params['tenant_id'],
                'receiver_type' => $params['receiver_type'], // 'admin' or 'employee'
                'receiver_id' => $params['receiver_id'],
                'sender_type' => $params['sender_type'] ?? 'system',
                'sender_id' => $params['sender_id'] ?? null,
                'sender_name' => $params['sender_name'] ?? 'System',
                'title' => $params['title'],
                'content' => $params['content'],
                'type' => $params['type'] ?? 'general',
                'priority' => $params['priority'] ?? 'normal',
                'related_id' => $params['related_id'] ?? null,
                'action_url' => $params['action_url'] ?? null,
                'action_text' => $params['action_text'] ?? null,
                'expires_at' => $params['expires_at'] ?? null,
                'is_read' => false,
            ]);

            // Send web push notification
            $this->sendWebPushNotification(
                $params['tenant_id'],
                $params['receiver_type'],
                $params['receiver_id'],
                $params['title'],
                $params['content'],
                $params
            );

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send notification to user', [
                'params' => $params,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send web push notification to user
     */
    private function sendWebPushNotification(
        string $tenantId,
        string $receiverType,
        int $receiverId,
        string $title,
        string $content,
        array $additionalData = []
    ): bool {
        try {
            // Get user's push subscriptions
            $subscriptions = $this->getUserPushSubscriptions($tenantId, $receiverType, $receiverId);
            
            if ($subscriptions->isEmpty()) {
                Log::info('No push subscriptions found for user', [
                    'tenant_id' => $tenantId,
                    'receiver_type' => $receiverType,
                    'receiver_id' => $receiverId
                ]);
                return true; // Not an error, just no subscriptions
            }

            // Prepare notification payload
            $payload = json_encode([
                'title' => $title,
                'body' => $content,
                'icon' => asset('images/notification-icon.png'),
                'badge' => asset('images/notification-badge.png'),
                'tag' => $additionalData['type'] ?? 'general',
                'data' => [
                    'url' => $additionalData['action_url'] ?? config('app.url'),
                    'type' => $additionalData['type'] ?? 'general',
                    'priority' => $additionalData['priority'] ?? 'normal',
                    'related_id' => $additionalData['related_id'] ?? null,
                    'tenant_id' => $tenantId,
                    'timestamp' => now()->toISOString(),
                ],
                'actions' => $this->getNotificationActions($additionalData),
                'requireInteraction' => ($additionalData['priority'] ?? 'normal') === 'urgent',
                'silent' => false,
            ]);

            // Send to all user's subscriptions
            $successCount = 0;
            foreach ($subscriptions as $subscription) {
                try {
                    $webPushSubscription = Subscription::create([
                        'endpoint' => $subscription->endpoint,
                        'publicKey' => $subscription->public_key,
                        'authToken' => $subscription->auth_token,
                        'contentEncoding' => $subscription->content_encoding ?? 'aesgcm',
                    ]);

                    $report = $this->webPush->sendOneNotification($webPushSubscription, $payload);

                    if ($report->isSuccess()) {
                        $successCount++;
                        Log::info('Web push notification sent successfully', [
                            'subscription_id' => $subscription->id,
                            'endpoint' => substr($subscription->endpoint, 0, 50) . '...'
                        ]);
                    } else {
                        Log::warning('Web push notification failed', [
                            'subscription_id' => $subscription->id,
                            'reason' => $report->getReason(),
                            'expired' => $report->isSubscriptionExpired()
                        ]);

                        // Remove expired subscriptions
                        if ($report->isSubscriptionExpired()) {
                            $subscription->delete();
                            Log::info('Removed expired push subscription', [
                                'subscription_id' => $subscription->id
                            ]);
                        }
                    }

                } catch (\Exception $e) {
                    Log::error('Error sending individual web push notification', [
                        'subscription_id' => $subscription->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            Log::info('Web push notification batch completed', [
                'tenant_id' => $tenantId,
                'receiver_type' => $receiverType,
                'receiver_id' => $receiverId,
                'total_subscriptions' => $subscriptions->count(),
                'successful_sends' => $successCount
            ]);

            return $successCount > 0;

        } catch (\Exception $e) {
            Log::error('Failed to send web push notification', [
                'tenant_id' => $tenantId,
                'receiver_type' => $receiverType,
                'receiver_id' => $receiverId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Get user's push subscriptions
     */
    private function getUserPushSubscriptions(string $tenantId, string $receiverType, int $receiverId)
    {
        $query = PushSubscription::where('tenant_id', $tenantId);

        if ($receiverType === 'admin') {
            $query->where('admin_id', $receiverId);
        } else {
            $query->where('employee_id', $receiverId);
        }

        return $query->get();
    }

    /**
     * Get notification actions based on type
     */
    private function getNotificationActions(array $data): array
    {
        $actions = [];

        // Add action button if provided
        if (!empty($data['action_url']) && !empty($data['action_text'])) {
            $actions[] = [
                'action' => 'open',
                'title' => $data['action_text'],
                'url' => $data['action_url']
            ];
        }

        // Add type-specific actions
        switch ($data['type'] ?? 'general') {
            case 'chat':
                $actions[] = [
                    'action' => 'reply',
                    'title' => 'Reply',
                    'url' => $data['chat_url'] ?? config('app.url') . '/chat'
                ];
                break;

            case 'points':
                if (in_array($data['points_action'] ?? '', ['pending'])) {
                    $actions[] = [
                        'action' => 'view',
                        'title' => 'View Details',
                        'url' => config('app.url') . '/points'
                    ];
                }
                break;

            case 'announcement':
                $actions[] = [
                    'action' => 'view',
                    'title' => 'View Announcement',
                    'url' => config('app.url') . '/announcements'
                ];
                break;
        }

        // Always add dismiss action
        $actions[] = [
            'action' => 'dismiss',
            'title' => 'Dismiss'
        ];

        return array_slice($actions, 0, 2); // Limit to 2 actions (browser limitation)
    }

    /**
     * Send notification to multiple users (enhanced with web push)
     */
    public function sendToMultipleUsers(array $params): array
    {
        $results = [
            'successful' => 0,
            'failed' => 0,
            'failed_users' => []
        ];

        foreach ($params['receivers'] as $receiver) {
            $singleParams = array_merge($params, [
                'receiver_type' => $receiver['type'],
                'receiver_id' => $receiver['id'],
            ]);
            
            unset($singleParams['receivers']); // Remove the receivers array for single notification

            if ($this->sendToUser($singleParams)) {
                $results['successful']++;
            } else {
                $results['failed']++;
                $results['failed_users'][] = $receiver;
            }
        }

        return $results;
    }

    /**
     * Send to all users of a specific type in tenant (enhanced with web push)
     */
    public function sendToAllUsers(array $params): int
    {
        try {
            $users = $this->getUsersByTypeAndFilters(
                $params['tenant_id'],
                $params['receiver_type'],
                $params['filters'] ?? []
            );

            $receivers = $users->map(function ($user) use ($params) {
                return [
                    'type' => $params['receiver_type'],
                    'id' => $params['receiver_type'] === 'employee' ? $user->employee_id : $user->id
                ];
            })->toArray();

            $multiParams = array_merge($params, ['receivers' => $receivers]);
            $results = $this->sendToMultipleUsers($multiParams);

            return $results['successful'];

        } catch (\Exception $e) {
            Log::error('Failed to send notification to all users', [
                'params' => $params,
                'error' => $e->getMessage()
            ]);
            return 0;
        }
    }

    /**
     * Send to users by category/department (enhanced with web push)
     */
    public function sendByCategory(array $params): int
    {
        $filters = [
            'category' => $params['category'] ?? null,
            'department_id' => $params['department_id'] ?? null,
        ];

        return $this->sendToAllUsers(array_merge($params, ['filters' => $filters]));
    }

    /**
     * Send emoji notification (enhanced with web push)
     */
    public function sendEmoji(array $params): bool
    {
        $emojiTitle = 'Emoji Reaction';
        $emojiContent = $params['emoji'] . ' ' . ($params['message'] ?? '');
        
        return $this->sendToUser([
            'tenant_id' => $params['tenant_id'],
            'receiver_type' => $params['receiver_type'],
            'receiver_id' => $params['receiver_id'],
            'sender_type' => $params['sender_type'] ?? 'user',
            'sender_id' => $params['sender_id'],
            'sender_name' => $params['sender_name'] ?? 'Someone',
            'title' => $emojiTitle,
            'content' => $emojiContent,
            'type' => 'emoji',
            'priority' => 'low', // Emojis are low priority
            'related_id' => $params['related_id'] ?? null,
            'context' => $params['context'] ?? 'general'
        ]);
    }

    /**
     * Send chat notification (enhanced with web push)
     */
    public function sendChatNotification(array $params): bool
    {
        return $this->sendToUser([
            'tenant_id' => $params['tenant_id'],
            'receiver_type' => $params['receiver_type'],
            'receiver_id' => $params['receiver_id'],
            'sender_type' => $params['sender_type'],
            'sender_id' => $params['sender_id'],
            'sender_name' => $params['sender_name'] ?? 'Someone',
            'title' => 'New Message',
            'content' => $params['message'],
            'type' => 'chat',
            'priority' => 'normal',
            'related_id' => $params['chat_id'] ?? null,
            'chat_id' => $params['chat_id'] ?? null,
            'message_id' => $params['message_id'] ?? null,
            'is_group' => $params['is_group'] ?? false,
            'chat_url' => $params['chat_url'] ?? null
        ]);
    }

    /**
     * Send points-related notification (enhanced with web push)
     */
    public function sendPointsNotification(array $params): bool
    {
        $pointsTypes = [
            'awarded' => 'Points Awarded! 🎉',
            'approved' => 'Points Request Approved ✅',
            'rejected' => 'Points Request Rejected ❌',
            'pending' => 'Points Request Pending ⏳',
            'cancelled' => 'Points Cancelled ⚠️',
            'target_completed' => 'Target Completed! 🎯',
            'target_set' => 'New Target Set 📋'
        ];

        $title = $pointsTypes[$params['points_action']] ?? 'Points Update';
        
        // Set priority based on action type
        $priority = 'normal';
        if (in_array($params['points_action'], ['awarded', 'target_completed'])) {
            $priority = 'high';
        } elseif (in_array($params['points_action'], ['rejected', 'cancelled'])) {
            $priority = 'normal';
        }

        return $this->sendToUser([
            'tenant_id' => $params['tenant_id'],
            'receiver_type' => $params['receiver_type'],
            'receiver_id' => $params['receiver_id'],
            'sender_type' => $params['sender_type'] ?? 'admin',
            'sender_id' => $params['sender_id'] ?? null,
            'sender_name' => $params['sender_name'] ?? 'Admin',
            'title' => $title,
            'content' => $params['content'],
            'type' => 'points',
            'priority' => $priority,
            'related_id' => $params['points_id'] ?? null,
            'points_action' => $params['points_action'],
            'points_amount' => $params['points_amount'] ?? null,
            'target_name' => $params['target_name'] ?? null,
            'vin_no' => $params['vin_no'] ?? null
        ]);
    }

    /**
     * Send announcement (enhanced with web push)
     */
    public function sendAnnouncement(array $params): int
    {
        return $this->sendToAllUsers([
            'tenant_id' => $params['tenant_id'],
            'receiver_type' => $params['receiver_type'],
            'sender_type' => 'admin',
            'sender_id' => $params['sender_id'],
            'sender_name' => $params['sender_name'] ?? 'Admin',
            'title' => $params['title'],
            'content' => $params['content'],
            'type' => 'announcement',
            'priority' => $params['priority'] ?? 'high',
            'expires_at' => $params['expires_at'] ?? null,
            'filters' => $params['filters'] ?? []
        ]);
    }

    /**
     * Send reminder notification (enhanced with web push)
     */
    public function sendReminder(array $params): bool
    {
        return $this->sendToUser([
            'tenant_id' => $params['tenant_id'],
            'receiver_type' => $params['receiver_type'],
            'receiver_id' => $params['receiver_id'],
            'sender_type' => 'system',
            'sender_id' => null,
            'sender_name' => 'System',
            'title' => $params['title'],
            'content' => $params['content'],
            'type' => 'reminder',
            'priority' => $params['priority'] ?? 'normal',
            'action_url' => $params['action_url'] ?? null,
            'action_text' => $params['action_text'] ?? null,
            'related_id' => $params['related_id'] ?? null,
            'reminder_data' => $params['reminder_data'] ?? []
        ]);
    }

    /**
     * Send bulk web push notification to all subscriptions
     */
    public function sendBulkWebPush(string $tenantId, string $title, string $content, array $options = []): array
    {
        try {
            $subscriptions = PushSubscription::where('tenant_id', $tenantId)->get();
            
            if ($subscriptions->isEmpty()) {
                return ['success' => true, 'sent' => 0, 'message' => 'No subscriptions found'];
            }

            $payload = json_encode([
                'title' => $title,
                'body' => $content,
                'icon' => asset('images/notification-icon.png'),
                'badge' => asset('images/notification-badge.png'),
                'tag' => $options['tag'] ?? 'bulk',
                'data' => [
                    'url' => $options['url'] ?? config('app.url'),
                    'type' => $options['type'] ?? 'announcement',
                    'tenant_id' => $tenantId,
                    'timestamp' => now()->toISOString(),
                ],
                'requireInteraction' => $options['requireInteraction'] ?? false,
            ]);

            $notifications = [];
            foreach ($subscriptions as $subscription) {
                $notifications[] = [
                    'subscription' => Subscription::create([
                        'endpoint' => $subscription->endpoint,
                        'publicKey' => $subscription->public_key,
                        'authToken' => $subscription->auth_token,
                        'contentEncoding' => $subscription->content_encoding ?? 'aesgcm',
                    ]),
                    'payload' => $payload,
                ];
            }

            // Send all notifications
            $reports = $this->webPush->sendNotifications($notifications);
            
            $successCount = 0;
            foreach ($reports as $report) {
                if ($report->isSuccess()) {
                    $successCount++;
                } elseif ($report->isSubscriptionExpired()) {
                    // Handle expired subscriptions
                    $this->removeExpiredSubscription($report->getRequest()->getUri());
                }
            }

            return [
                'success' => true,
                'sent' => $successCount,
                'total' => $subscriptions->count(),
                'message' => "Sent {$successCount} out of {$subscriptions->count()} notifications"
            ];

        } catch (\Exception $e) {
            Log::error('Bulk web push failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return [
                'success' => false,
                'sent' => 0,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Remove expired subscription
     */
    private function removeExpiredSubscription(string $endpoint): void
    {
        try {
            PushSubscription::where('endpoint', $endpoint)->delete();
            Log::info('Removed expired push subscription', ['endpoint' => $endpoint]);
        } catch (\Exception $e) {
            Log::error('Failed to remove expired subscription', [
                'endpoint' => $endpoint,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get user notifications with filters
     */
    public function getUserNotifications(
        string $tenantId,
        string $userType,
        int $userId,
        int $limit = 20,
        array $filters = []
    ) {
        $query = Notification::forTenant($tenantId)
            ->forReceiver($userType, $userId);

        // Apply filters
        if (isset($filters['type'])) {
            $query->ofType($filters['type']);
        }

        if (isset($filters['unread_only']) && $filters['unread_only']) {
            $query->unread();
        }

        if (isset($filters['priority'])) {
            $query->where('priority', $filters['priority']);
        }

        if (isset($filters['from_date'])) {
            $query->where('created_at', '>=', $filters['from_date']);
        }

        return $query->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get()
            ->map(function ($notification) {
                return $this->formatNotification($notification);
            });
    }

    /**
     * Mark notification as read
     */
    public function markAsRead(int $notificationId, string $tenantId, string $userType, int $userId): bool
    {
        try {
            $notification = Notification::where('id', $notificationId)
                ->forTenant($tenantId)
                ->forReceiver($userType, $userId)
                ->first();

            if ($notification) {
                return $notification->markAsRead();
            }

            return false;

        } catch (\Exception $e) {
            Log::error('Failed to mark notification as read', [
                'notification_id' => $notificationId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead(string $tenantId, string $userType, int $userId): int
    {
        try {
            return Notification::forTenant($tenantId)
                ->forReceiver($userType, $userId)
                ->unread()
                ->update(['is_read' => true, 'read_at' => now()]);

        } catch (\Exception $e) {
            Log::error('Failed to mark all notifications as read', [
                'tenant_id' => $tenantId,
                'user_type' => $userType,
                'user_id' => $userId,
                'error' => $e->getMessage()
            ]);
            return 0;
        }
    }

    /**
     * Get unread count
     */
    public function getUnreadCount(string $tenantId, string $userType, int $userId): int
    {
        try {
            return Notification::forTenant($tenantId)
                ->forReceiver($userType, $userId)
                ->unread()
                ->count();

        } catch (\Exception $e) {
            Log::error('Failed to get unread count', [
                'error' => $e->getMessage()
            ]);
            return 0;
        }
    }

    /**
     * Delete notification
     */
    public function deleteNotification(int $notificationId, string $tenantId, string $userType, int $userId): bool
    {
        try {
            $deleted = Notification::where('id', $notificationId)
                ->forTenant($tenantId)
                ->forReceiver($userType, $userId)
                ->delete();

            return $deleted > 0;

        } catch (\Exception $e) {
            Log::error('Failed to delete notification', [
                'notification_id' => $notificationId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Clean up old notifications
     */
    public function cleanupOldNotifications(int $daysOld = 30): int
    {
        try {
            return Notification::where('created_at', '<', now()->subDays($daysOld))
                ->delete();

        } catch (\Exception $e) {
            Log::error('Failed to cleanup old notifications', [
                'error' => $e->getMessage()
            ]);
            return 0;
        }
    }

    /**
     * Get notification statistics
     */
    public function getNotificationStats(string $tenantId, array $filters = []): array
    {
        try {
            $query = Notification::forTenant($tenantId);

            if (isset($filters['user_type'])) {
                $query->where('receiver_type', $filters['user_type']);
            }

            if (isset($filters['from_date'])) {
                $query->where('created_at', '>=', $filters['from_date']);
            }

            $total = $query->count();
            $unread = (clone $query)->unread()->count();
            $byType = (clone $query)->select('type', DB::raw('count(*) as count'))
                ->groupBy('type')
                ->pluck('count', 'type')
                ->toArray();

            // Add push subscription stats
            $pushStats = $this->getPushSubscriptionStats($tenantId);

            return [
                'total_notifications' => $total,
                'unread_notifications' => $unread,
                'read_notifications' => $total - $unread,
                'by_type' => $byType,
                'read_percentage' => $total > 0 ? round((($total - $unread) / $total) * 100, 2) : 0,
                'push_subscriptions' => $pushStats
            ];

        } catch (\Exception $e) {
            Log::error('Failed to get notification stats', [
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    /**
     * Get push subscription statistics
     */
    public function getPushSubscriptionStats(string $tenantId): array
    {
        try {
            $total = PushSubscription::where('tenant_id', $tenantId)->count();
            $adminSubscriptions = PushSubscription::where('tenant_id', $tenantId)
                ->whereNotNull('admin_id')->count();
            $employeeSubscriptions = PushSubscription::where('tenant_id', $tenantId)
                ->whereNotNull('employee_id')->count();

            return [
                'total' => $total,
                'admin_subscriptions' => $adminSubscriptions,
                'employee_subscriptions' => $employeeSubscriptions,
            ];

        } catch (\Exception $e) {
            Log::error('Failed to get push subscription stats', [
                'error' => $e->getMessage()
            ]);
            return ['total' => 0, 'admin_subscriptions' => 0, 'employee_subscriptions' => 0];
        }
    }

    /**
     * Private helper methods
     */
    private function createNotification(array $data): Notification
    {
        // Add metadata
        $data['created_at'] = now();
        $data['updated_at'] = now();

        return Notification::create($data);
    }

    private function getUsersByTypeAndFilters(string $tenantId, string $userType, array $filters = [])
    {
        if ($userType === 'employee') {
            $query = Employee::where('tenant_id', $tenantId);
            
            if (isset($filters['category'])) {
                $query->where('category', $filters['category']);
            }
            
            if (isset($filters['department_id'])) {
                $query->where('department_id', $filters['department_id']);
            }
            
            return $query->get();
        } else {
            $query = Admin::where('tenant_id', $tenantId);
            
            if (isset($filters['role'])) {
                $query->where('role', $filters['role']);
            }
            
            return $query->get();
        }
    }

    private function formatNotification(Notification $notification): array
    {
        return [
            'id' => $notification->id,
            'title' => $notification->title,
            'content' => $notification->content,
            'sender_name' => $notification->sender_name, // Uses your existing accessor
            'type' => $notification->type,
            'priority' => $notification->priority ?? 'normal',
            'is_read' => $notification->is_read,
            'related_id' => $notification->related_id,
            'action_url' => $notification->action_url,
            'action_text' => $notification->action_text,
            'expires_at' => $notification->expires_at,
            'created_at' => $notification->created_at->format('Y-m-d H:i:s'),
            'time_ago' => $notification->time_ago, // Uses your existing accessor
        ];
    }
}