<?php
// app/Services/TenantManager.php
namespace App\Services;

use App\Models\Tenant;
use Illuminate\Support\Facades\Log;

class TenantManager
{
    /**
     * The current tenant instance.
     *
     * @var \App\Models\Tenant|null
     */
    protected $tenant;

    /**
     * Set the current tenant.
     *
     * @param  \App\Models\Tenant  $tenant
     * @return $this
     */
    public function setTenant(Tenant $tenant)
    {
        $this->tenant = $tenant;
        
        Log::info('TenantManager: Tenant set', [
            'tenant_id' => $tenant->id,
            'tenant_name' => $tenant->name
        ]);
        
        return $this;
    }

    /**
     * Get the current tenant.
     *
     * @return \App\Models\Tenant|null
     */
    public function getTenant()
    {
        return $this->tenant;
    }

    /**
     * Get the current tenant ID.
     *
     * @return int|null
     */
    public function getTenantId()
    {
        return $this->tenant ? $this->tenant->id : null;
    }
    
    /**
     * Get the legacy tenant ID (admin_id for compatibility).
     *
     * @return int|null
     */
    public function getLegacyTenantId()
    {
        if (!$this->tenant) {
            return null;
        }
        
        // If tenant has an admin_id, use that for legacy code
        if ($this->tenant->admin_id) {
            return $this->tenant->admin_id;
        }
        
        // Fall back to actual tenant ID if no admin_id is set
        return $this->tenant->id;
    }
    
    /**
     * Clear the current tenant.
     *
     * @return $this
     */
    public function clearTenant()
    {
        $this->tenant = null;
        return $this;
    }
    
    /**
     * Check if a tenant is set.
     *
     * @return bool
     */
    public function hasTenant()
    {
        return $this->tenant !== null;
    }
}