<?php

namespace App\Services;

use App\Models\Employee;
use App\Models\Admin;
use App\Models\PushSubscription;
use App\Models\Target;
use Minishlink\WebPush\WebPush;
use Minishlink\WebPush\Subscription;
use Illuminate\Support\Facades\Log;

class PointsNotificationService
{
    private NotificationService $notificationService;
    private WebPush $webPush;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        $this->webPush = new WebPush([
            'VAPID' => [
                'subject' => config('app.vapid_subject'),
                'publicKey' => config('app.vapid_public_key'),
                'privateKey' => config('app.vapid_private_key'),
            ]
        ]);
    }

    /**
     * Send comprehensive notification (database + push) to employee
     */
    public function sendPointsNotification(
        int $employeeId,
        int $tenantId,
        string $title,
        string $content,
        string $type = 'points',
        ?int $adminId = null
    ): bool {
        try {
            // 1. Send database notification
            $this->notificationService->sendToUser(
                $tenantId,
                'employee',
                $employeeId,
                $title,
                $content,
                'admin',
                $adminId,
                $type
            );

            // 2. Send browser push notification
            $this->sendBrowserPushNotification($employeeId, $tenantId, $title, $content, $type);

            Log::info('Points notification sent successfully', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'type' => $type,
                'title' => $title
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send points notification', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return false;
        }
    }

    /**
     * Send points awarded notification to all relevant stakeholders
     */
    public function sendPointsAwardedNotification(
        int $employeeId,
        int $tenantId,
        int $points,
        array $targetNames = [],
        ?int $adminId = null,
        string $category = null,
        int $departmentId = null
    ): bool {
        $targetsText = !empty($targetNames) 
            ? ' for ' . implode(', ', $targetNames)
            : '';

        $employee = Employee::where('employee_id', $employeeId)
                           ->where('tenant_id', $tenantId)
                           ->first();

        if (!$employee) {
            Log::warning('Employee not found for points awarded notification', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId
            ]);
            return false;
        }

        $employeeCategory = $category ?? $employee->category;
        $employeeDepartment = $departmentId ?? $employee->department_id;

        // Employee notification
        $employeeTitle = 'Points Awarded! 🎉';
        $employeeContent = "Congratulations! You have been awarded {$points} points{$targetsText}. Keep up the great work!";

        $this->sendPointsNotification(
            $employeeId,
            $tenantId,
            $employeeTitle,
            $employeeContent,
            'points_awarded',
            $adminId
        );

        // Notify all stakeholders
        $this->notifyAllStakeholders(
            $tenantId,
            $employeeId,
            $employeeCategory,
            $employeeDepartment,
            "Points Awarded to {$employee->name} 📊",
            "{$employee->name} has been awarded {$points} points{$targetsText}.",
            'points_awarded_notification',
            $adminId
        );

        return true;
    }

    /**
     * Send points approved notification to all relevant stakeholders
     */
    public function sendPointsApprovedNotification(
        int $employeeId,
        int $tenantId,
        int $points,
        string $targetName = '',
        ?int $adminId = null,
        string $category = null,
        int $departmentId = null
    ): bool {
        $targetText = $targetName ? " for {$targetName}" : '';
        
        $employee = Employee::where('employee_id', $employeeId)
                           ->where('tenant_id', $tenantId)
                           ->first();

        if (!$employee) {
            return false;
        }

        $employeeCategory = $category ?? $employee->category;
        $employeeDepartment = $departmentId ?? $employee->department_id;

        // Employee notification
        $employeeTitle = 'Points Request Approved! ✅';
        $employeeContent = "Great news! Your request for {$points} points has been approved{$targetText}.";

        $this->sendPointsNotification(
            $employeeId,
            $tenantId,
            $employeeTitle,
            $employeeContent,
            'points_approved',
            $adminId
        );

        // Notify all stakeholders
        $this->notifyAllStakeholders(
            $tenantId,
            $employeeId,
            $employeeCategory,
            $employeeDepartment,
            "Points Request Approved for {$employee->name} ✅",
            "{$employee->name}'s request for {$points} points has been approved{$targetText}.",
            'points_approved_notification',
            $adminId
        );

        return true;
    }

    /**
     * Send points rejected notification
     */
    public function sendPointsRejectedNotification(
        int $employeeId,
        int $tenantId,
        string $reason,
        ?int $adminId = null,
        string $category = null,
        int $departmentId = null
    ): bool {
        $employee = Employee::where('employee_id', $employeeId)
                           ->where('tenant_id', $tenantId)
                           ->first();

        if (!$employee) {
            return false;
        }

        $employeeCategory = $category ?? $employee->category;
        $employeeDepartment = $departmentId ?? $employee->department_id;

        // Employee notification
        $employeeTitle = 'Points Request Rejected ❌';
        $employeeContent = "Your points request has been rejected. Reason: {$reason}";

        $this->sendPointsNotification(
            $employeeId,
            $tenantId,
            $employeeTitle,
            $employeeContent,
            'points_rejected',
            $adminId
        );

        // Notify all stakeholders
        $this->notifyAllStakeholders(
            $tenantId,
            $employeeId,
            $employeeCategory,
            $employeeDepartment,
            "Points Request Rejected for {$employee->name} ❌",
            "{$employee->name}'s points request has been rejected. Reason: {$reason}",
            'points_rejected_notification',
            $adminId
        );

        return true;
    }

    /**
     * Send target completion notification
     */
    public function sendTargetCompletionNotification(
        int $employeeId,
        int $tenantId,
        string $targetName,
        ?int $adminId = null
    ): bool {
        $employee = Employee::where('employee_id', $employeeId)
                           ->where('tenant_id', $tenantId)
                           ->first();

        if (!$employee) {
            return false;
        }

        // Employee notification
        $employeeTitle = 'Target Completed! 🏆';
        $employeeContent = "Congratulations! You have successfully completed the target: {$targetName}";

        $this->sendPointsNotification(
            $employeeId,
            $tenantId,
            $employeeTitle,
            $employeeContent,
            'target_completed',
            $adminId
        );

        // Notify all stakeholders
        $this->notifyAllStakeholders(
            $tenantId,
            $employeeId,
            $employee->category,
            $employee->department_id,
            "Target Completed by {$employee->name} 🏆",
            "{$employee->name} has successfully completed the target: {$targetName}",
            'target_completed_notification',
            $adminId
        );

        return true;
    }

    /**
     * Send target set notification to all related users
     */
    public function sendTargetSetNotification(
        int $tenantId,
        int $targetId,
        string $targetName,
        string $category,
        int $departmentId,
        ?int $adminId = null
    ): bool {
        try {
            $title = 'New Target Set! 🎯';
            $content = "A new target '{$targetName}' has been set for your category/department.";

            // Get all employees in the same category and department
            $employees = Employee::where('tenant_id', $tenantId)
                               ->where('category', $category)
                               ->where('department_id', $departmentId)
                               ->get();

            // Notify employees
            foreach ($employees as $employee) {
                $this->sendPointsNotification(
                    $employee->employee_id,
                    $tenantId,
                    $title,
                    $content,
                    'target_set',
                    $adminId
                );
            }

            // Notify stakeholders
            $this->notifyAllStakeholders(
                $tenantId,
                null,
                $category,
                $departmentId,
                "New Target Set: {$targetName} 🎯",
                "A new target '{$targetName}' has been set for category: {$category}",
                'target_set_notification',
                $adminId
            );

            Log::info('Target set notifications sent', [
                'target_id' => $targetId,
                'tenant_id' => $tenantId,
                'category' => $category,
                'department_id' => $departmentId,
                'employees_notified' => $employees->count()
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send target set notifications', [
                'target_id' => $targetId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send points cancelled notification
     */
    public function sendPointsCancelledNotification(
        int $employeeId,
        int $tenantId,
        int $points,
        string $reason,
        ?int $adminId = null,
        string $category = null,
        int $departmentId = null
    ): bool {
        $employee = Employee::where('employee_id', $employeeId)
                           ->where('tenant_id', $tenantId)
                           ->first();

        if (!$employee) {
            return false;
        }

        $employeeCategory = $category ?? $employee->category;
        $employeeDepartment = $departmentId ?? $employee->department_id;

        // Employee notification
        $employeeTitle = 'Points Cancelled ⚠️';
        $employeeContent = "Your {$points} points have been cancelled. Reason: {$reason}";

        $this->sendPointsNotification(
            $employeeId,
            $tenantId,
            $employeeTitle,
            $employeeContent,
            'points_cancelled',
            $adminId
        );

        // Notify all stakeholders
        $this->notifyAllStakeholders(
            $tenantId,
            $employeeId,
            $employeeCategory,
            $employeeDepartment,
            "Points Cancelled for {$employee->name} ⚠️",
            "{$points} points have been cancelled for {$employee->name}. Reason: {$reason}",
            'points_cancelled_notification',
            $adminId
        );

        return true;
    }

    /**
     * Notify all relevant stakeholders (category point admins, superadmin, manager)
     */
    private function notifyAllStakeholders(
        int $tenantId,
        ?int $employeeId,
        string $category,
        int $departmentId,
        string $title,
        string $content,
        string $type,
        ?int $excludeAdminId = null
    ): void {
        try {
            // 1. Notify all point admins in the same category
            $this->notifyCategoryPointAdmins($tenantId, $category, $departmentId, $title, $content, $type, $excludeAdminId);

            // 2. Notify superadmin
            $this->notifySuperAdmin($tenantId, $title, $content, $type, $excludeAdminId);

            // 3. Notify manager (tenant admin)
            $this->notifyManager($tenantId, $title, $content, $type, $excludeAdminId);

        } catch (\Exception $e) {
            Log::error('Failed to notify stakeholders', [
                'tenant_id' => $tenantId,
                'employee_id' => $employeeId,
                'category' => $category,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Notify all point admins in the same category
     */
    private function notifyCategoryPointAdmins(
        int $tenantId,
        string $category,
        int $departmentId,
        string $title,
        string $content,
        string $type,
        ?int $excludeAdminId = null
    ): void {
        // Get all admins in the same tenant and category
        // Assuming admins have access to specific categories/departments
        $categoryAdmins = Admin::where('tenant_id', $tenantId)
                             ->where('category', $category)
                             ->when($excludeAdminId, function ($query, $excludeAdminId) {
                                 return $query->where('id', '!=', $excludeAdminId);
                             })
                             ->get();

        foreach ($categoryAdmins as $admin) {
            $this->notificationService->sendToUser(
                $tenantId,
                'admin',
                $admin->id,
                $title,
                $content,
                'system',
                null,
                $type
            );

            // Send push notification to admin
            $this->sendAdminPushNotification($admin->id, $tenantId, $title, $content, $type);
        }

        Log::info('Category point admins notified', [
            'tenant_id' => $tenantId,
            'category' => $category,
            'admins_count' => $categoryAdmins->count()
        ]);
    }

    /**
     * Notify superadmin
     */
    private function notifySuperAdmin(
        int $tenantId,
        string $title,
        string $content,
        string $type,
        ?int $excludeAdminId = null
    ): void {
        // Get superadmin (assuming role field or specific identification)
        $superAdmin = Admin::where('tenant_id', $tenantId)
                          ->where('role', 'superadmin')
                          ->orWhere('subadmin', '0') // assuming 0 means main admin
                          ->when($excludeAdminId, function ($query, $excludeAdminId) {
                              return $query->where('id', '!=', $excludeAdminId);
                          })
                          ->first();

        if ($superAdmin) {
            $this->notificationService->sendToUser(
                $tenantId,
                'admin',
                $superAdmin->id,
                $title,
                $content,
                'system',
                null,
                $type
            );

            // Send push notification to superadmin
            $this->sendAdminPushNotification($superAdmin->id, $tenantId, $title, $content, $type);

            Log::info('Superadmin notified', [
                'tenant_id' => $tenantId,
                'admin_id' => $superAdmin->id
            ]);
        }
    }

    /**
     * Notify manager (tenant admin/owner)
     */
    private function notifyManager(
        int $tenantId,
        string $title,
        string $content,
        string $type,
        ?int $excludeAdminId = null
    ): void {
        // Get manager/owner of the tenant
        $manager = Admin::where('tenant_id', $tenantId)
                       ->where(function ($query) {
                           $query->where('role', 'manager')
                                 ->orWhere('role', 'owner')
                                 ->orWhere('subadmin', '0');
                       })
                       ->when($excludeAdminId, function ($query, $excludeAdminId) {
                           return $query->where('id', '!=', $excludeAdminId);
                       })
                       ->first();

        if ($manager) {
            $this->notificationService->sendToUser(
                $tenantId,
                'admin',
                $manager->id,
                $title,
                $content,
                'system',
                null,
                $type
            );

            // Send push notification to manager
            $this->sendAdminPushNotification($manager->id, $tenantId, $title, $content, $type);

            Log::info('Manager notified', [
                'tenant_id' => $tenantId,
                'manager_id' => $manager->id
            ]);
        }
    }

    /**
     * Send browser push notification to employee
     */
    private function sendBrowserPushNotification(
        int $employeeId,
        int $tenantId,
        string $title,
        string $content,
        string $type
    ): void {
        try {
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();

            if (!$employee) {
                Log::warning('Employee not found for push notification', [
                    'employee_id' => $employeeId,
                    'tenant_id' => $tenantId
                ]);
                return;
            }

            $subscriptions = $employee->pushSubscriptions;
            
            if ($subscriptions->isEmpty()) {
                Log::info('No push subscriptions for employee', [
                    'employee_id' => $employeeId
                ]);
                return;
            }

            $payload = $this->createPushPayload($title, $content, $type);
            $successCount = 0;

            foreach ($subscriptions as $subscription) {
                if ($this->sendToSubscription($subscription, $payload)) {
                    $successCount++;
                }
            }

            Log::info('Push notifications sent to employee', [
                'employee_id' => $employeeId,
                'subscriptions_count' => $subscriptions->count(),
                'successful_sends' => $successCount
            ]);

        } catch (\Exception $e) {
            Log::error('Browser push notification failed', [
                'employee_id' => $employeeId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Send browser push notification to admin
     */
    private function sendAdminPushNotification(
        int $adminId,
        int $tenantId,
        string $title,
        string $content,
        string $type
    ): void {
        try {
            $admin = Admin::where('id', $adminId)
                         ->where('tenant_id', $tenantId)
                         ->first();

            if (!$admin) {
                Log::warning('Admin not found for push notification', [
                    'admin_id' => $adminId,
                    'tenant_id' => $tenantId
                ]);
                return;
            }

            $subscriptions = $admin->pushSubscriptions;
            
            if ($subscriptions->isEmpty()) {
                Log::info('No push subscriptions for admin', [
                    'admin_id' => $adminId
                ]);
                return;
            }

            $payload = $this->createPushPayload($title, $content, $type);
            $successCount = 0;

            foreach ($subscriptions as $subscription) {
                if ($this->sendToSubscription($subscription, $payload)) {
                    $successCount++;
                }
            }

            Log::info('Push notifications sent to admin', [
                'admin_id' => $adminId,
                'subscriptions_count' => $subscriptions->count(),
                'successful_sends' => $successCount
            ]);

        } catch (\Exception $e) {
            Log::error('Admin push notification failed', [
                'admin_id' => $adminId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    /**
     * Create push notification payload
     */
    private function createPushPayload(string $title, string $content, string $type): array
    {
        // Limit content length for push notifications
        if (strlen($content) > 100) {
            $content = substr($content, 0, 97) . '...';
        }

        $iconMap = [
            'points_awarded' => '🎉',
            'points_approved' => '✅',
            'points_rejected' => '❌',
            'points_cancelled' => '⚠️',
            'target_completed' => '🏆',
            'target_set' => '🎯',
            'default' => '📢'
        ];

        $icon = $iconMap[$type] ?? $iconMap['default'];

        return [
            'title' => $title,
            'body' => $content,
            'icon' => '/favicon.ico',
            'badge' => '/favicon.ico',
            'tag' => 'points-' . $type . '-' . time(),
            'url' => url('/dashboard'),
            'data' => [
                'type' => $type,
                'category' => 'points',
                'url' => url('/dashboard'),
                'timestamp' => time()
            ],
            'requireInteraction' => true,
            'renotify' => true,
            'silent' => false,
            'actions' => [
                [
                    'action' => 'view',
                    'title' => 'View Dashboard',
                    'icon' => '/favicon.ico'
                ]
            ]
        ];
    }

    /**
     * Send notification to a specific push subscription
     */
    private function sendToSubscription(PushSubscription $pushSubscription, array $payload): bool
    {
        try {
            $subscription = Subscription::create($pushSubscription->getSubscriptionData());
            
            $result = $this->webPush->sendOneNotification(
                $subscription,
                json_encode($payload)
            );

            // Remove invalid subscriptions
            if (!$result->isSuccess()) {
                if ($result->getStatusCode() === 410 || $result->getStatusCode() === 404) {
                    Log::info('Removing invalid push subscription', [
                        'subscription_id' => $pushSubscription->id,
                        'status_code' => $result->getStatusCode()
                    ]);
                    $pushSubscription->delete();
                } else {
                    Log::warning('Push notification failed', [
                        'subscription_id' => $pushSubscription->id,
                        'status_code' => $result->getStatusCode(),
                        'reason' => $result->getReason()
                    ]);
                }
                return false;
            }

            Log::debug('Push notification sent successfully', [
                'subscription_id' => $pushSubscription->id
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Push notification exception', [
                'subscription_id' => $pushSubscription->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send bulk notifications to multiple employees
     */
    public function sendBulkNotification(
        array $employeeIds,
        int $tenantId,
        string $title,
        string $content,
        string $type = 'points',
        ?int $adminId = null
    ): array {
        $results = [
            'success' => [],
            'failed' => []
        ];

        foreach ($employeeIds as $employeeId) {
            $sent = $this->sendPointsNotification(
                $employeeId,
                $tenantId,
                $title,
                $content,
                $type,
                $adminId
            );

            if ($sent) {
                $results['success'][] = $employeeId;
            } else {
                $results['failed'][] = $employeeId;
            }
        }

        Log::info('Bulk points notifications sent', [
            'total_employees' => count($employeeIds),
            'successful' => count($results['success']),
            'failed' => count($results['failed']),
            'type' => $type
        ]);

        return $results;
    }
}