<?php

namespace App\Services;

use App\Models\Employee;
use App\Models\Admin;
use App\Models\SuperAdmin;
use App\Models\Points;
use App\Models\Target;
use Illuminate\Support\Facades\Log;

class PointsEmailNotificationService
{
    private MailjetService $mailjetService;

    public function __construct(MailjetService $mailjetService)
    {
        $this->mailjetService = $mailjetService;
    }

    /**
     * Send points request submitted notification
     */
    public function sendPointsRequestSubmittedNotification(
        int $employeeId,
        int $tenantId,
        int $totalPoints,
        array $targetNames,
        string $vinNo
    ): bool {
        try {
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();

            if (!$employee) {
                Log::warning('Employee not found for points request notification', [
                    'employee_id' => $employeeId,
                    'tenant_id' => $tenantId
                ]);
                return false;
            }

            $targetsText = !empty($targetNames) ? implode(', ', $targetNames) : 'Various targets';
            
            // Email to employee
            $this->sendEmployeeConfirmationEmail($employee, $totalPoints, $targetsText, $vinNo);
            
            // Email to all stakeholders
            $this->sendStakeholderNotificationEmail(
                $employee,
                $totalPoints,
                $targetsText,
                $vinNo,
                'Points Request Submitted',
                'A new points request has been submitted and is awaiting review.'
            );

            return true;

        } catch (\Exception $e) {
            Log::error('Points request submitted notification failed', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send points request approved notification
     */
    public function sendPointsRequestApprovedNotification(
        int $employeeId,
        int $tenantId,
        int $points,
        string $targetName,
        ?int $approvedBy = null
    ): bool {
        try {
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();

            if (!$employee) {
                return false;
            }

            $approverName = $this->getApproverName($approvedBy, $tenantId);
            
            // Email to employee
            $this->sendEmployeeApprovalEmail($employee, $points, $targetName, $approverName);
            
            // Email to all stakeholders
            $this->sendStakeholderNotificationEmail(
                $employee,
                $points,
                $targetName,
                '',
                'Points Request Approved',
                'A points request has been approved and points have been awarded.'
            );

            return true;

        } catch (\Exception $e) {
            Log::error('Points request approved notification failed', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send points request rejected notification
     */
    public function sendPointsRequestRejectedNotification(
        int $employeeId,
        int $tenantId,
        int $points,
        string $targetName,
        string $rejectionReason,
        ?int $rejectedBy = null
    ): bool {
        try {
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();

            if (!$employee) {
                return false;
            }

            $rejectorName = $this->getApproverName($rejectedBy, $tenantId);
            
            // Email to employee
            $this->sendEmployeeRejectionEmail($employee, $points, $targetName, $rejectionReason, $rejectorName);
            
            // Email to all stakeholders
            $this->sendStakeholderNotificationEmail(
                $employee,
                $points,
                $targetName,
                '',
                'Points Request Rejected',
                'A points request has been rejected and requires attention.'
            );

            return true;

        } catch (\Exception $e) {
            Log::error('Points request rejected notification failed', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send points request reversed notification
     */
    public function sendPointsRequestReversedNotification(
        int $employeeId,
        int $tenantId,
        int $points,
        string $targetName,
        ?int $reversedBy = null
    ): bool {
        try {
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();

            if (!$employee) {
                return false;
            }

            $reverserName = $this->getApproverName($reversedBy, $tenantId);
            
            // Email to employee
            $this->sendEmployeeReversalEmail($employee, $points, $targetName, $reverserName);
            
            // Email to all stakeholders
            $this->sendStakeholderNotificationEmail(
                $employee,
                $points,
                $targetName,
                '',
                'Points Request Reversed',
                'A previously approved points request has been reversed and is now pending review.'
            );

            return true;

        } catch (\Exception $e) {
            Log::error('Points request reversed notification failed', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send points request reprocessed notification
     */
    public function sendPointsRequestReprocessedNotification(
        int $employeeId,
        int $tenantId,
        int $points,
        string $targetName,
        ?int $reprocessedBy = null
    ): bool {
        try {
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();

            if (!$employee) {
                return false;
            }

            $reprocessorName = $this->getApproverName($reprocessedBy, $tenantId);
            
            // Email to employee
            $this->sendEmployeeReprocessEmail($employee, $points, $targetName, $reprocessorName);
            
            // Email to all stakeholders
            $this->sendStakeholderNotificationEmail(
                $employee,
                $points,
                $targetName,
                '',
                'Points Request Reprocessed',
                'A previously rejected points request has been reprocessed and is now pending review.'
            );

            return true;

        } catch (\Exception $e) {
            Log::error('Points request reprocessed notification failed', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send employee confirmation email
     */
    private function sendEmployeeConfirmationEmail(
        Employee $employee,
        int $totalPoints,
        string $targetsText,
        string $vinNo
    ): void {
        $subject = "Points Request Submitted - {$totalPoints} Points";
        
        $htmlContent = $this->generateEmployeeConfirmationHtml(
            $employee->name,
            $totalPoints,
            $targetsText,
            $vinNo
        );

        $textContent = $this->generateEmployeeConfirmationText(
            $employee->name,
            $totalPoints,
            $targetsText,
            $vinNo
        );

        $this->mailjetService->sendEmail([
            'to' => [
                [
                    'Email' => $employee->email,
                    'Name' => $employee->name
                ]
            ],
            'subject' => $subject,
            'html_part' => $htmlContent,
            'text_part' => $textContent,
            'custom_id' => 'points-request-confirmation'
        ]);
    }

    /**
     * Send employee approval email
     */
    private function sendEmployeeApprovalEmail(
        Employee $employee,
        int $points,
        string $targetName,
        string $approverName
    ): void {
        $subject = "Points Request Approved - {$points} Points Awarded!";
        
        $htmlContent = $this->generateEmployeeApprovalHtml(
            $employee->name,
            $points,
            $targetName,
            $approverName
        );

        $textContent = $this->generateEmployeeApprovalText(
            $employee->name,
            $points,
            $targetName,
            $approverName
        );

        $this->mailjetService->sendEmail([
            'to' => [
                [
                    'Email' => $employee->email,
                    'Name' => $employee->name
                ]
            ],
            'subject' => $subject,
            'html_part' => $htmlContent,
            'text_part' => $textContent,
            'custom_id' => 'points-request-approved'
        ]);
    }

    /**
     * Send employee rejection email
     */
    private function sendEmployeeRejectionEmail(
        Employee $employee,
        int $points,
        string $targetName,
        string $rejectionReason,
        string $rejectorName
    ): void {
        $subject = "Points Request Update - Requires Review";
        
        $htmlContent = $this->generateEmployeeRejectionHtml(
            $employee->name,
            $points,
            $targetName,
            $rejectionReason,
            $rejectorName
        );

        $textContent = $this->generateEmployeeRejectionText(
            $employee->name,
            $points,
            $targetName,
            $rejectionReason,
            $rejectorName
        );

        $this->mailjetService->sendEmail([
            'to' => [
                [
                    'Email' => $employee->email,
                    'Name' => $employee->name
                ]
            ],
            'subject' => $subject,
            'html_part' => $htmlContent,
            'text_part' => $textContent,
            'custom_id' => 'points-request-rejected'
        ]);
    }

    /**
     * Send stakeholder notification email
     */
    private function sendStakeholderNotificationEmail(
        Employee $employee,
        int $points,
        string $targetsText,
        string $vinNo,
        string $action,
        string $description
    ): void {
        $stakeholders = $this->getStakeholders($employee->tenant_id, $employee->category, $employee->department_id);
        
        if (empty($stakeholders)) {
            Log::warning('No stakeholders found for notification', [
                'tenant_id' => $employee->tenant_id,
                'category' => $employee->category,
                'department_id' => $employee->department_id
            ]);
            return;
        }

        $subject = "{$action} - {$employee->name} ({$points} Points)";
        
        $htmlContent = $this->generateStakeholderNotificationHtml(
            $employee,
            $points,
            $targetsText,
            $vinNo,
            $action,
            $description
        );

        $textContent = $this->generateStakeholderNotificationText(
            $employee,
            $points,
            $targetsText,
            $vinNo,
            $action,
            $description
        );

        $this->mailjetService->sendBulkEmail([
            'recipients' => $stakeholders,
            'subject' => $subject,
            'html_part' => $htmlContent,
            'text_part' => $textContent,
            'custom_id' => 'points-stakeholder-notification'
        ]);
    }

    /**
     * Get all stakeholders for notifications based on database schema
     */
    private function getStakeholders(int $tenantId, string $category, int $departmentId): array
    {
        $stakeholders = [];

        // 1. Get ALL SuperAdmins (global admins, no tenant restriction)
        $superAdmins = SuperAdmin::where('status', 'active')->get();
        foreach ($superAdmins as $admin) {
            if ($admin->email) {
                $stakeholders[] = [
                    'email' => $admin->email,
                    'name' => $admin->name,
                    'role' => 'SuperAdmin'
                ];
            }
        }

        // 2. Get Managers in the SAME TENANT and SAME CATEGORY
        $managers = Employee::where('tenant_id', $tenantId)
                           ->where('category', $category)
                           ->where('role', 'Manager')
                           ->where('status', '!=', '0') // Active managers only
                           ->get();
        foreach ($managers as $manager) {
            if ($manager->email) {
                $stakeholders[] = [
                    'email' => $manager->email,
                    'name' => $manager->name,
                    'role' => 'Manager'
                ];
            }
        }

        // 3. Get PointsAdmins (Admins table) in the SAME TENANT and SAME CATEGORY
        // Note: Based on schema, PointsAdmins are stored in 'admins' table
        // and tenant_id references admins.id, so we need to find admins with matching category
        $pointsAdmins = Employee::where('tenant_id', $tenantId)
                               ->where('category', $category)
                               ->where('role', 'Admin')
                               ->where('status', '!=', '0') // Active admins only
                               ->get();
        foreach ($pointsAdmins as $admin) {
            if ($admin->email) {
                $stakeholders[] = [
                    'email' => $admin->email,
                    'name' => $admin->name,
                    'role' => 'PointsAdmin'
                ];
            }
        }

        // 4. Get the tenant admin (the main admin for this tenant)
        $tenantAdmin = \App\Models\Admin::find($tenantId);
        if ($tenantAdmin && $tenantAdmin->email) {
            $stakeholders[] = [
                'email' => $tenantAdmin->email,
                'name' => $tenantAdmin->name,
                'role' => 'TenantAdmin'
            ];
        }

        // Remove duplicates based on email
        $uniqueStakeholders = [];
        $seenEmails = [];
        foreach ($stakeholders as $stakeholder) {
            if (!in_array($stakeholder['email'], $seenEmails)) {
                $uniqueStakeholders[] = $stakeholder;
                $seenEmails[] = $stakeholder['email'];
            }
        }

        return $uniqueStakeholders;
    }

    /**
     * Get approver name by ID
     */
    private function getApproverName(?int $approverId, int $tenantId): string
    {
        if (!$approverId) {
            return 'System Administrator';
        }

        // Try SuperAdmin first
        $superAdmin = SuperAdmin::where('id', $approverId)->first();
        if ($superAdmin) {
            return $superAdmin->name;
        }

        // Try Admin (PointsAdmin)
        $admin = Admin::where('id', $approverId)->first();
        if ($admin) {
            return $admin->name;
        }

        // Try Employee (Manager)
        $employee = Employee::where('employee_id', $approverId)->first();
        if ($employee) {
            return $employee->name;
        }

        return 'System Administrator';
    }

    // HTML Template Methods
    private function generateEmployeeConfirmationHtml(string $name, int $points, string $targets, string $vinNo): string
    {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='utf-8'>
            <title>Points Request Submitted</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #007bff; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f8f9fa; }
                .highlight { background: #e3f2fd; padding: 15px; border-left: 4px solid #2196f3; margin: 15px 0; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>🎯 Points Request Submitted</h1>
                </div>
                <div class='content'>
                    <h2>Hello {$name}!</h2>
                    <p>Your points request has been successfully submitted and is now under review.</p>
                    
                    <div class='highlight'>
                        <h3>Request Details:</h3>
                        <p><strong>Total Points:</strong> {$points}</p>
                        <p><strong>Targets:</strong> {$targets}</p>
                        <p><strong>VIN Number:</strong> {$vinNo}</p>
                        <p><strong>Status:</strong> Pending Review</p>
                    </div>
                    
                    <p>Your request will be reviewed by the appropriate administrators. You will receive another email once the review is complete.</p>
                    
                    <p>Thank you for your contribution!</p>
                </div>
                <div class='footer'>
                    <p>This is an automated message from the Carisma Car Points System.</p>
                </div>
            </div>
        </body>
        </html>";
    }

    private function generateEmployeeApprovalHtml(string $name, int $points, string $target, string $approver): string
    {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='utf-8'>
            <title>Points Request Approved</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #28a745; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f8f9fa; }
                .highlight { background: #d4edda; padding: 15px; border-left: 4px solid #28a745; margin: 15px 0; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>✅ Points Request Approved!</h1>
                </div>
                <div class='content'>
                    <h2>Congratulations {$name}!</h2>
                    <p>Great news! Your points request has been approved and your points have been awarded.</p>
                    
                    <div class='highlight'>
                        <h3>Award Details:</h3>
                        <p><strong>Points Awarded:</strong> {$points}</p>
                        <p><strong>Target:</strong> {$target}</p>
                        <p><strong>Approved By:</strong> {$approver}</p>
                        <p><strong>Status:</strong> ✅ Approved</p>
                    </div>
                    
                    <p>Your points have been added to your account and will be reflected in the leaderboard.</p>
                    
                    <p>Keep up the excellent work!</p>
                </div>
                <div class='footer'>
                    <p>This is an automated message from the Carisma Car Points System.</p>
                </div>
            </div>
        </body>
        </html>";
    }

    private function generateEmployeeRejectionHtml(string $name, int $points, string $target, string $reason, string $rejector): string
    {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='utf-8'>
            <title>Points Request Update</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #dc3545; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f8f9fa; }
                .highlight { background: #f8d7da; padding: 15px; border-left: 4px solid #dc3545; margin: 15px 0; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>⚠️ Points Request Update</h1>
                </div>
                <div class='content'>
                    <h2>Hello {$name},</h2>
                    <p>Your points request requires additional review before it can be approved.</p>
                    
                    <div class='highlight'>
                        <h3>Request Details:</h3>
                        <p><strong>Points Requested:</strong> {$points}</p>
                        <p><strong>Target:</strong> {$target}</p>
                        <p><strong>Reviewed By:</strong> {$rejector}</p>
                        <p><strong>Reason:</strong> {$reason}</p>
                        <p><strong>Status:</strong> ⚠️ Requires Review</p>
                    </div>
                    
                    <p>Please review the feedback above and resubmit your request with the necessary corrections.</p>
                    
                    <p>If you have any questions, please contact your manager or administrator.</p>
                </div>
                <div class='footer'>
                    <p>This is an automated message from the Carisma Car Points System.</p>
                </div>
            </div>
        </body>
        </html>";
    }

    private function generateStakeholderNotificationHtml(Employee $employee, int $points, string $targets, string $vinNo, string $action, string $description): string
    {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='utf-8'>
            <title>{$action}</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #6c757d; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f8f9fa; }
                .highlight { background: #e9ecef; padding: 15px; border-left: 4px solid #6c757d; margin: 15px 0; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>📊 {$action}</h1>
                </div>
                <div class='content'>
                    <h2>Administrative Notification</h2>
                    <p>{$description}</p>
                    
                    <div class='highlight'>
                        <h3>Employee Details:</h3>
                        <p><strong>Name:</strong> {$employee->name}</p>
                        <p><strong>Email:</strong> {$employee->email}</p>
                        <p><strong>Category:</strong> {$employee->category}</p>
                        <p><strong>Department:</strong> " . ($employee->department->name ?? 'N/A') . "</p>
                        <p><strong>Points:</strong> {$points}</p>
                        <p><strong>Targets:</strong> {$targets}</p>
                        " . ($vinNo ? "<p><strong>VIN:</strong> {$vinNo}</p>" : "") . "
                    </div>
                    
                    <p>Please log into the system to review and take any necessary action.</p>
                </div>
                <div class='footer'>
                    <p>This is an automated message from the Carisma Car Points System.</p>
                </div>
            </div>
        </body>
        </html>";
    }

    // Text Template Methods
    private function generateEmployeeConfirmationText(string $name, int $points, string $targets, string $vinNo): string
    {
        return "Hello {$name}!\n\nYour points request has been successfully submitted and is now under review.\n\nRequest Details:\n- Total Points: {$points}\n- Targets: {$targets}\n- VIN Number: {$vinNo}\n- Status: Pending Review\n\nYour request will be reviewed by the appropriate administrators. You will receive another email once the review is complete.\n\nThank you for your contribution!\n\nThis is an automated message from the Carisma Car Points System.";
    }

    private function generateEmployeeApprovalText(string $name, int $points, string $target, string $approver): string
    {
        return "Congratulations {$name}!\n\nGreat news! Your points request has been approved and your points have been awarded.\n\nAward Details:\n- Points Awarded: {$points}\n- Target: {$target}\n- Approved By: {$approver}\n- Status: Approved\n\nYour points have been added to your account and will be reflected in the leaderboard.\n\nKeep up the excellent work!\n\nThis is an automated message from the Carisma Car Points System.";
    }

    private function generateEmployeeRejectionText(string $name, int $points, string $target, string $reason, string $rejector): string
    {
        return "Hello {$name},\n\nYour points request requires additional review before it can be approved.\n\nRequest Details:\n- Points Requested: {$points}\n- Target: {$target}\n- Reviewed By: {$rejector}\n- Reason: {$reason}\n- Status: Requires Review\n\nPlease review the feedback above and resubmit your request with the necessary corrections.\n\nIf you have any questions, please contact your manager or administrator.\n\nThis is an automated message from the Carisma Car Points System.";
    }

    private function generateStakeholderNotificationText(Employee $employee, int $points, string $targets, string $vinNo, string $action, string $description): string
    {
        return "Administrative Notification\n\n{$action}\n\n{$description}\n\nEmployee Details:\n- Name: {$employee->name}\n- Email: {$employee->email}\n- Category: {$employee->category}\n- Department: " . ($employee->department->name ?? 'N/A') . "\n- Points: {$points}\n- Targets: {$targets}\n" . ($vinNo ? "- VIN: {$vinNo}\n" : "") . "\n\nPlease log into the system to review and take any necessary action.\n\nThis is an automated message from the Carisma Car Points System.";
    }

    /**
     * Send employee reversal email
     */
    private function sendEmployeeReversalEmail(
        Employee $employee,
        int $points,
        string $targetName,
        string $reverserName
    ): void {
        $subject = "Points Request Reversed - {$points} Points";
        
        $htmlContent = $this->generateEmployeeReversalHtml(
            $employee->name,
            $points,
            $targetName,
            $reverserName
        );

        $textContent = $this->generateEmployeeReversalText(
            $employee->name,
            $points,
            $targetName,
            $reverserName
        );

        $this->mailjetService->sendEmail([
            'to' => [
                [
                    'Email' => $employee->email,
                    'Name' => $employee->name
                ]
            ],
            'subject' => $subject,
            'html_part' => $htmlContent,
            'text_part' => $textContent,
            'custom_id' => 'points-request-reversed'
        ]);
    }

    /**
     * Send employee reprocess email
     */
    private function sendEmployeeReprocessEmail(
        Employee $employee,
        int $points,
        string $targetName,
        string $reprocessorName
    ): void {
        $subject = "Points Request Reprocessed - {$points} Points";
        
        $htmlContent = $this->generateEmployeeReprocessHtml(
            $employee->name,
            $points,
            $targetName,
            $reprocessorName
        );

        $textContent = $this->generateEmployeeReprocessText(
            $employee->name,
            $points,
            $targetName,
            $reprocessorName
        );

        $this->mailjetService->sendEmail([
            'to' => [
                [
                    'Email' => $employee->email,
                    'Name' => $employee->name
                ]
            ],
            'subject' => $subject,
            'html_part' => $htmlContent,
            'text_part' => $textContent,
            'custom_id' => 'points-request-reprocessed'
        ]);
    }

    // HTML Template Methods for new notifications
    private function generateEmployeeReversalHtml(string $name, int $points, string $target, string $reverser): string
    {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='utf-8'>
            <title>Points Request Reversed</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #ffc107; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f8f9fa; }
                .highlight { background: #fff3cd; padding: 15px; border-left: 4px solid #ffc107; margin: 15px 0; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>🔄 Points Request Reversed</h1>
                </div>
                <div class='content'>
                    <h2>Hello {$name},</h2>
                    <p>Your previously approved points request has been reversed and is now pending review.</p>
                    
                    <div class='highlight'>
                        <h3>Request Details:</h3>
                        <p><strong>Points:</strong> {$points}</p>
                        <p><strong>Target:</strong> {$target}</p>
                        <p><strong>Reversed By:</strong> {$reverser}</p>
                        <p><strong>Status:</strong> 🔄 Pending Review</p>
                    </div>
                    
                    <p>Your request will be reviewed again by the appropriate administrators. You will receive another email once the review is complete.</p>
                    
                    <p>If you have any questions, please contact your manager or administrator.</p>
                </div>
                <div class='footer'>
                    <p>This is an automated message from the Carisma Car Points System.</p>
                </div>
            </div>
        </body>
        </html>";
    }

    private function generateEmployeeReprocessHtml(string $name, int $points, string $target, string $reprocessor): string
    {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='utf-8'>
            <title>Points Request Reprocessed</title>
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #17a2b8; color: white; padding: 20px; text-align: center; }
                .content { padding: 20px; background: #f8f9fa; }
                .highlight { background: #d1ecf1; padding: 15px; border-left: 4px solid #17a2b8; margin: 15px 0; }
                .footer { text-align: center; padding: 20px; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class='container'>
                <div class='header'>
                    <h1>🔄 Points Request Reprocessed</h1>
                </div>
                <div class='content'>
                    <h2>Hello {$name},</h2>
                    <p>Your previously rejected points request has been reprocessed and is now pending review.</p>
                    
                    <div class='highlight'>
                        <h3>Request Details:</h3>
                        <p><strong>Points:</strong> {$points}</p>
                        <p><strong>Target:</strong> {$target}</p>
                        <p><strong>Reprocessed By:</strong> {$reprocessor}</p>
                        <p><strong>Status:</strong> 🔄 Pending Review</p>
                    </div>
                    
                    <p>Your request will be reviewed again by the appropriate administrators. You will receive another email once the review is complete.</p>
                    
                    <p>If you have any questions, please contact your manager or administrator.</p>
                </div>
                <div class='footer'>
                    <p>This is an automated message from the Carisma Car Points System.</p>
                </div>
            </div>
        </body>
        </html>";
    }

    // Text Template Methods for new notifications
    private function generateEmployeeReversalText(string $name, int $points, string $target, string $reverser): string
    {
        return "Hello {$name},\n\nYour previously approved points request has been reversed and is now pending review.\n\nRequest Details:\n- Points: {$points}\n- Target: {$target}\n- Reversed By: {$reverser}\n- Status: Pending Review\n\nYour request will be reviewed again by the appropriate administrators. You will receive another email once the review is complete.\n\nIf you have any questions, please contact your manager or administrator.\n\nThis is an automated message from the Carisma Car Points System.";
    }

    private function generateEmployeeReprocessText(string $name, int $points, string $target, string $reprocessor): string
    {
        return "Hello {$name},\n\nYour previously rejected points request has been reprocessed and is now pending review.\n\nRequest Details:\n- Points: {$points}\n- Target: {$target}\n- Reprocessed By: {$reprocessor}\n- Status: Pending Review\n\nYour request will be reviewed again by the appropriate administrators. You will receive another email once the review is complete.\n\nIf you have any questions, please contact your manager or administrator.\n\nThis is an automated message from the Carisma Car Points System.";
    }
}
