<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\Employee;
use App\Models\Admin;
use App\Models\Message;
use App\Models\Conversation;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Send a chat message notification
     */
    public function sendChatNotification(Message $message)
    {
        $conversation = $message->conversation;
        
        // Get all participants except the sender
        $participants = $conversation->participants()
            ->where(function ($query) use ($message) {
                if ($message->sender_type === 'admin') {
                    $query->where(function ($q) use ($message) {
                        $q->whereNull('admin_id')
                          ->orWhere('admin_id', '!=', $message->sender_admin_id);
                    });
                } else {
                    $query->where(function ($q) use ($message) {
                        $q->whereNull('employee_id')
                          ->orWhere('employee_id', '!=', $message->sender_id);
                    });
                }
            })
            ->get();

        $senderName = $this->getSenderName($message->sender_type, $message->sender_admin_id ?: $message->sender_id);
        $title = $conversation->type === 'group' 
            ? "New message in {$conversation->name}" 
            : "New message from {$senderName}";

        // Create notification for each participant
        foreach ($participants as $participant) {
            $receiverType = $participant->admin_id ? 'admin' : 'employee';
            $receiverId = $participant->admin_id ?: $participant->employee_id;

            $this->createNotification([
                'tenant_id' => $conversation->tenant_id,
                'sender_type' => $message->sender_type,
                'sender_id' => $message->sender_admin_id ?: $message->sender_id,
                'receiver_type' => $receiverType,
                'receiver_id' => $receiverId,
                'title' => $title,
                'content' => $this->truncateText($message->content, 100),
                'type' => 'chat',
                'related_id' => $conversation->id,
            ]);
        }
    }

    /**
     * Send announcement to specific users
     */
    public function sendAnnouncement($tenantId, $title, $content, $receivers = [])
    {
        foreach ($receivers as $receiver) {
            $this->createNotification([
                'tenant_id' => $tenantId,
                'sender_type' => 'system',
                'sender_id' => null,
                'receiver_type' => $receiver['type'],
                'receiver_id' => $receiver['id'],
                'title' => $title,
                'content' => $content,
                'type' => 'announcement',
                'related_id' => null,
            ]);
        }
    }

    /**
     * Send notification to all employees in tenant
     */
    public function sendToAllEmployees($tenantId, $title, $content, $senderType = 'system', $senderId = null)
    {
        $employees = Employee::where('tenant_id', $tenantId)->get();
        
        foreach ($employees as $employee) {
            $this->createNotification([
                'tenant_id' => $tenantId,
                'sender_type' => $senderType,
                'sender_id' => $senderId,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'title' => $title,
                'content' => $content,
                'type' => 'announcement',
                'related_id' => null,
            ]);
        }
    }

    /**
     * Send notification to all admins in tenant
     */
    public function sendToAllAdmins($tenantId, $title, $content, $senderType = 'system', $senderId = null)
    {
        $admins = Admin::where('tenant_id', $tenantId)->get();
        
        foreach ($admins as $admin) {
            $this->createNotification([
                'tenant_id' => $tenantId,
                'sender_type' => $senderType,
                'sender_id' => $senderId,
                'receiver_type' => 'admin',
                'receiver_id' => $admin->id,
                'title' => $title,
                'content' => $content,
                'type' => 'announcement',
                'related_id' => null,
            ]);
        }
    }

    /**
     * Send notification to specific user
     */
    public function sendToUser($tenantId, $receiverType, $receiverId, $title, $content, $senderType = 'system', $senderId = null, $type = 'general', $relatedId = null)
    {
        return $this->createNotification([
            'tenant_id' => $tenantId,
            'sender_type' => $senderType,
            'sender_id' => $senderId,
            'receiver_type' => $receiverType,
            'receiver_id' => $receiverId,
            'title' => $title,
            'content' => $content,
            'type' => $type,
            'related_id' => $relatedId,
        ]);
    }

    /**
     * Get user notifications
     */
    public function getUserNotifications($tenantId, $receiverType, $receiverId, $limit = 20, $onlyUnread = false)
    {
        $query = Notification::forTenant($tenantId)
            ->where('receiver_id', $receiverId)
            ->orderBy('created_at', 'desc');
        if ($onlyUnread) {
            $query->unread();
        }

        return $query->limit($limit)->get();
    }

    /**
     * Get unread count for user
     */
    public function getUnreadCount($tenantId, $receiverType, $receiverId)
    {
        return Notification::forTenant($tenantId)
            ->forReceiver($receiverType, $receiverId)
            ->unread()
            ->count();
    }

    /**
     * Mark notification as read
     */
    public function markAsRead($notificationId, $tenantId, $receiverType, $receiverId)
    {
        $notification = Notification::forTenant($tenantId)
            ->forReceiver($receiverType, $receiverId)
            ->find($notificationId);

        if ($notification) {
            return $notification->markAsRead();
        }

        return false;
    }

    /**
     * Mark all notifications as read for user
     */
    public function markAllAsRead($tenantId, $receiverType, $receiverId)
    {
        return Notification::forTenant($tenantId)
            ->forReceiver($receiverType, $receiverId)
            ->unread()
            ->update([
                'is_read' => true,
                'read_at' => now(),
            ]);
    }

    /**
     * Mark chat notifications as read when user opens conversation
     */
    public function markChatNotificationsAsRead($tenantId, $receiverType, $receiverId, $conversationId)
    {
        return Notification::forTenant($tenantId)
            ->forReceiver($receiverType, $receiverId)
            ->where('type', 'chat')
            ->where('related_id', $conversationId)
            ->unread()
            ->update([
                'is_read' => true,
                'read_at' => now(),
            ]);
    }

    /**
     * Delete notification
     */
    public function deleteNotification($notificationId, $tenantId, $receiverType, $receiverId)
    {
        return Notification::forTenant($tenantId)
            ->forReceiver($receiverType, $receiverId)
            ->where('id', $notificationId)
            ->delete();
    }

    /**
     * Delete all notifications for user
     */
    public function deleteAllNotifications($tenantId, $receiverType, $receiverId)
    {
        return Notification::forTenant($tenantId)
            ->forReceiver($receiverType, $receiverId)
            ->delete();
    }

    /**
     * Create notification record
     */
    private function createNotification($data)
    {
        try {
            return Notification::create($data);
        } catch (\Exception $e) {
            Log::error('Failed to create notification', [
                'data' => $data,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Get sender name
     */
    private function getSenderName($senderType, $senderId)
    {
        if ($senderType === 'admin') {
            $admin = Admin::find($senderId);
            return $admin ? $admin->email : 'Admin';
        }

        if ($senderType === 'employee') {
            $employee = Employee::where('employee_id', $senderId)->first();
            return $employee ? $employee->name : 'Employee';
        }

        return 'System';
    }

    /**
     * Truncate text for notification content
     */
    private function truncateText($text, $length = 100)
    {
        if (strlen($text) <= $length) {
            return $text;
        }

        return substr($text, 0, $length) . '...';
    }
}