<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class MailjetService
{
    private string $apiKey;
    private string $secretKey;
    private string $version;
    private string $fromEmail;
    private string $fromName;

    public function __construct()
    {
        $this->apiKey = config('mailjet.api_key');
        $this->secretKey = config('mailjet.secret_key');
        $this->version = config('mailjet.version');
        $this->fromEmail = config('mailjet.from_email');
        $this->fromName = config('mailjet.from_name');
    }

    /**
     * Send email using Mailjet API
     */
    public function sendEmail(array $data): bool
    {
        try {
            $url = "https://api.mailjet.com/{$this->version}/send";
            
            $payload = [
                'Messages' => [
                    [
                        'From' => [
                            'Email' => $this->fromEmail,
                            'Name' => $this->fromName
                        ],
                        'To' => $data['to'],
                        'Subject' => $data['subject'],
                        'TextPart' => $data['text_part'] ?? '',
                        'HTMLPart' => $data['html_part'] ?? '',
                        'CustomID' => $data['custom_id'] ?? 'points-notification',
                        'TrackOpens' => config('mailjet.defaults.track_opens', true) ? 'enabled' : 'disabled',
                        'TrackClicks' => config('mailjet.defaults.track_clicks', true) ? 'enabled' : 'disabled',
                    ]
                ]
            ];

            // Add template if provided
            if (isset($data['template_id'])) {
                $payload['Messages'][0]['TemplateID'] = $data['template_id'];
                $payload['Messages'][0]['TemplateLanguage'] = true;
                $payload['Messages'][0]['Variables'] = $data['variables'] ?? [];
            }

            $response = Http::withBasicAuth($this->apiKey, $this->secretKey)
                ->withHeaders([
                    'Content-Type' => 'application/json',
                ])
                ->post($url, $payload);

            if ($response->successful()) {
                Log::info('Mailjet email sent successfully', [
                    'to' => $data['to'],
                    'subject' => $data['subject'],
                    'response' => $response->json()
                ]);
                return true;
            } else {
                Log::error('Mailjet email failed', [
                    'to' => $data['to'],
                    'subject' => $data['subject'],
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                return false;
            }

        } catch (\Exception $e) {
            Log::error('Mailjet service error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send email to multiple recipients
     */
    public function sendBulkEmail(array $data): bool
    {
        try {
            $url = "https://api.mailjet.com/{$this->version}/send";
            
            $payload = [
                'Messages' => []
            ];

            foreach ($data['recipients'] as $recipient) {
                $message = [
                    'From' => [
                        'Email' => $this->fromEmail,
                        'Name' => $this->fromName
                    ],
                    'To' => [
                        [
                            'Email' => $recipient['email'],
                            'Name' => $recipient['name'] ?? $recipient['email']
                        ]
                    ],
                    'Subject' => $data['subject'],
                    'TextPart' => $data['text_part'] ?? '',
                    'HTMLPart' => $data['html_part'] ?? '',
                    'CustomID' => $data['custom_id'] ?? 'points-notification-bulk',
                    'TrackOpens' => config('mailjet.defaults.track_opens', true) ? 'enabled' : 'disabled',
                    'TrackClicks' => config('mailjet.defaults.track_clicks', true) ? 'enabled' : 'disabled',
                ];

                // Add template if provided
                if (isset($data['template_id'])) {
                    $message['TemplateID'] = $data['template_id'];
                    $message['TemplateLanguage'] = true;
                    $message['Variables'] = $recipient['variables'] ?? [];
                }

                $payload['Messages'][] = $message;
            }

            $response = Http::withBasicAuth($this->apiKey, $this->secretKey)
                ->withHeaders([
                    'Content-Type' => 'application/json',
                ])
                ->post($url, $payload);

            if ($response->successful()) {
                Log::info('Mailjet bulk email sent successfully', [
                    'recipients_count' => count($data['recipients']),
                    'subject' => $data['subject'],
                    'response' => $response->json()
                ]);
                return true;
            } else {
                Log::error('Mailjet bulk email failed', [
                    'recipients_count' => count($data['recipients']),
                    'subject' => $data['subject'],
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                return false;
            }

        } catch (\Exception $e) {
            Log::error('Mailjet bulk service error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Test Mailjet connection
     */
    public function testConnection(): bool
    {
        try {
            $url = "https://api.mailjet.com/{$this->version}/user";
            
            $response = Http::withBasicAuth($this->apiKey, $this->secretKey)
                ->timeout(30)
                ->get($url);

            if ($response->successful()) {
                Log::info('Mailjet connection test successful', [
                    'response' => $response->json()
                ]);
                return true;
            } else {
                Log::error('Mailjet connection test failed', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                return false;
            }

        } catch (\Exception $e) {
            Log::error('Mailjet connection test error', [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
}
