<?php

namespace App\Services;

use App\Models\EmployeeLeaderBoard;
use App\Models\DepartmentLeaderBoard;
use App\Models\CategoryLeaderBoard;
use App\Models\Employee;
use App\Models\Department;
use App\Models\Prize;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;

class LeaderboardService
{
    protected $tenantId;
    const CACHE_TTL = 300; // 5 minutes
    
    public function __construct($tenantId)
    {
        $this->tenantId = $tenantId;
    }
    
    /**
     * Get leaderboard data with advanced filtering
     */
    public function getLeaderboardData($filters = [])
    {
        $cacheKey = $this->generateCacheKey($filters);
        
        return Cache::remember($cacheKey, self::CACHE_TTL, function() use ($filters) {
            return $this->fetchLeaderboardData($filters);
        });
    }
    
    /**
     * Get leaderboard data filtered by specific category
     */
    public function getLeaderboardDataByCategory($filters = [])
    {
        $cacheKey = $this->generateCacheKey($filters, 'category');
        
        return Cache::remember($cacheKey, self::CACHE_TTL, function() use ($filters) {
            return $this->fetchLeaderboardDataByCategory($filters);
        });
    }
    
    /**
     * Get BDC-specific leaderboard
     */
    public function getBDCLeaderboard($timePeriod = 'current_month')
    {
        $dateRange = $this->getDateRange($timePeriod);
        
        return EmployeeLeaderBoard::join('employees', function($join) {
                $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                     ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id');
            })
            ->where('employee_leader_boards.tenant_id', $this->tenantId)
            ->where('employees.category', 'BDC')
            ->whereBetween('employee_leader_boards.updated_at', $dateRange)
            ->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee_name',
                'employees.category',
                'employees.department_id',
                'employees.image'
            ]);
    }
    
    /**
     * Get Service Advisor leaderboard
     */
    public function getServiceAdvisorLeaderboard($timePeriod = 'current_month')
    {
        $dateRange = $this->getDateRange($timePeriod);
        
        return EmployeeLeaderBoard::join('employees', function($join) {
                $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                     ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id');
            })
            ->where('employee_leader_boards.tenant_id', $this->tenantId)
            ->where('employees.category', 'Service Advisor')
            ->whereBetween('employee_leader_boards.updated_at', $dateRange)
            ->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee_name',
                'employees.category',
                'employees.department_id',
                'employees.image'
            ]);
    }
    
    /**
     * Get Department leaderboard
     */
    public function getDepartmentLeaderboard($timePeriod = 'current_month')
    {
        $dateRange = $this->getDateRange($timePeriod);
        
        return DepartmentLeaderBoard::join('departments', function($join) {
                $join->on('department_leader_boards.department_id', '=', 'departments.id')
                     ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id');
            })
            ->join('employees', function($join) {
                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
                     ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id')
                     ->where('employees.role', 'Manager');
            })
            ->where('department_leader_boards.tenant_id', $this->tenantId)
            ->whereBetween('department_leader_boards.updated_at', $dateRange)
            ->groupBy('department_leader_boards.id', 'department_leader_boards.rank', 'department_leader_boards.earned_points', 'departments.name')
            ->orderBy('department_leader_boards.rank', 'asc')
            ->get([
                'department_leader_boards.*',
                'departments.name as department_name',
                DB::raw('COUNT(DISTINCT employees.employee_id) as count'),
                DB::raw('MIN(employees.name) as manager_name'),
                DB::raw('MIN(employees.image) as manager_image')
            ]);
    }
    
    /**
     * Get Employee leaderboard
     */
    public function getEmployeeLeaderboard($timePeriod = 'current_month')
    {
        $dateRange = $this->getDateRange($timePeriod);
        
        return EmployeeLeaderBoard::join('employees', function($join) {
                $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                     ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id');
            })
            ->where('employee_leader_boards.tenant_id', $this->tenantId)
            ->whereBetween('employee_leader_boards.updated_at', $dateRange)
            ->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee_name',
                'employees.category',
                'employees.department_id',
                'employees.image'
            ]);
    }
    
    /**
     * Get Category leaderboard
     */
    public function getCategoryLeaderboard($timePeriod = 'current_month')
    {
        $dateRange = $this->getDateRange($timePeriod);
        
        return CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->where('category_leader_board.tenant_id', $this->tenantId)
            ->whereBetween('category_leader_board.updated_at', $dateRange)
            ->orderBy('category_leader_board.rank', 'asc')
            ->get([
                'category_leader_board.*',
                'departments.name as department_name'
            ]);
    }
    
    /**
     * Get top 3 performers for any category
     */
    public function getTopPerformers($category, $timePeriod = 'current_month', $limit = 3)
    {
        $data = $this->getLeaderboardData(['category' => $category, 'time_period' => $timePeriod]);
        return $data->take($limit);
    }
    
    /**
     * Get complete ranking for any category
     */
    public function getCompleteRanking($category, $timePeriod = 'current_month')
    {
        $data = $this->getLeaderboardData(['category' => $category, 'time_period' => $timePeriod]);
        return $data->skip(3); // Skip top 3, return the rest
    }
    
    /**
     * Get available months with data
     */
    public function getAvailableMonths()
    {
        return collect([
            EmployeeLeaderBoard::where('tenant_id', $this->tenantId)->select('month_year')->distinct()->pluck('month_year'),
            DepartmentLeaderBoard::where('tenant_id', $this->tenantId)->select('month_year')->distinct()->pluck('month_year'),
            CategoryLeaderBoard::where('tenant_id', $this->tenantId)->select('month_year')->distinct()->pluck('month_year')
        ])->flatten()->unique()->sort()->reverse()->values();
    }
    
    /**
     * Get available categories
     */
    public function getAvailableCategories()
    {
        return Employee::where('tenant_id', $this->tenantId)
            ->select('category')
            ->distinct()
            ->pluck('category')
            ->filter()
            ->values();
    }
    
    /**
     * Get available departments
     */
    public function getAvailableDepartments()
    {
        return Department::where('tenant_id', $this->tenantId)
            ->select('id', 'name')
            ->get();
    }
    
    private function fetchLeaderboardData($filters)
    {
        // Implementation based on filter type
        switch($filters['category'] ?? 'employees') {
            case 'bdc':
                return $this->getBDCLeaderboard($filters['time_period'] ?? 'current_month');
            case 'service_advisors':
                return $this->getServiceAdvisorLeaderboard($filters['time_period'] ?? 'current_month');
            case 'departments':
                return $this->getDepartmentLeaderboard($filters['time_period'] ?? 'current_month');
            case 'categories':
                return $this->getCategoryLeaderboard($filters['time_period'] ?? 'current_month');
            default:
                return $this->getEmployeeLeaderboard($filters['time_period'] ?? 'current_month');
        }
    }
    
    private function fetchLeaderboardDataByCategory($filters)
    {
        $category = $filters['category'] ?? 'all';
        $timePeriod = $filters['time_period'] ?? 'current_month';
        
        if ($category === 'all') {
            return $this->getCategoryLeaderboard($timePeriod);
        }
        
        // Get data for specific category
        $dateRange = $this->getDateRange($timePeriod);
        
        \Log::info('Fetching leaderboard data for category: ' . $category, [
            'tenant_id' => $this->tenantId,
            'dateRange' => $dateRange,
            'category' => $category
        ]);
        
        $query = EmployeeLeaderBoard::join('employees', function($join) {
                $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                     ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id');
            })
            ->where('employee_leader_boards.tenant_id', $this->tenantId)
            ->where('employees.category', '=', $category)
            ->whereBetween('employee_leader_boards.updated_at', $dateRange)
            ->select(
                'employee_leader_boards.*',
                'employees.name as employee_name',
                'employees.category',
                'employees.image'
            )
            ->orderBy('employee_leader_boards.earned_points', 'desc');
            
        \Log::info('SQL Query: ' . $query->toSql());
        \Log::info('Query Bindings: ' . json_encode($query->getBindings()));
        
        return $query->get()
            ->map(function($item) {
                return (object) [
                    'employee_id' => $item->employee_id,
                    'employee_name' => $item->employee_name,
                    'category' => $item->category,
                    'earned_points' => $item->earned_points,
                    'image' => $item->image,
                    'rank' => null // Will be calculated
                ];
            })
            ->values();
    }
    
    private function generateCacheKey($filters, $type = '')
    {
        $typeSuffix = $type ? '_' . $type : '';
        return 'leaderboard' . $typeSuffix . '_' . $this->tenantId . '_' . md5(serialize($filters));
    }
    
    private function getDateRange($timePeriod)
    {
        $now = Carbon::now();
        
        switch($timePeriod) {
            case 'current_month':
                return [$now->copy()->startOfMonth(), $now->copy()->endOfMonth()];
            case 'last_month':
                return [$now->copy()->subMonth()->startOfMonth(), $now->copy()->subMonth()->endOfMonth()];
            case 'quarter':
                return [$now->copy()->startOfQuarter(), $now->copy()->endOfQuarter()];
            case 'year':
                return [$now->copy()->startOfYear(), $now->copy()->endOfYear()];
            default:
                return [$now->copy()->startOfMonth(), $now->copy()->endOfMonth()];
        }
    }
    
    /**
     * Get prizes for leaderboard positions
     */
    public function getPrizesForLeaderboard($category = 'all', $timePeriod = 'current_month')
    {
        $cacheKey = 'prizes_' . $this->tenantId . '_' . $category . '_' . $timePeriod;
        
        return Cache::remember($cacheKey, self::CACHE_TTL, function() use ($category, $timePeriod) {
            $query = Prize::forTenant($this->tenantId)
                ->active()
                ->currentlyValid()
                ->orderBy('position');

            if ($category !== 'all') {
                $query->forCategory($category);
            }

            if ($timePeriod !== 'all') {
                $query->currentPeriod($timePeriod);
            }

            return $query->get()->groupBy('position');
        });
    }

    /**
     * Get prize for specific position
     */
    public function getPrizeForPosition($position, $category = 'all', $timePeriod = 'current_month')
    {
        $prizes = $this->getPrizesForLeaderboard($category, $timePeriod);
        return $prizes->get($position) ? $prizes->get($position)->first() : null;
    }

    /**
     * Get leaderboard data with prizes
     */
    public function getLeaderboardDataWithPrizes($filters = [])
    {
        $leaderboardData = $this->getLeaderboardData($filters);
        $prizes = $this->getPrizesForLeaderboard(
            $filters['category'] ?? 'all',
            $filters['time_period'] ?? 'current_month'
        );

        return [
            'leaderboard' => $leaderboardData,
            'prizes' => $prizes
        ];
    }

    /**
     * Clear cache for this tenant
     */
    public function clearCache()
    {
        $pattern = 'leaderboard_' . $this->tenantId . '_*';
        $keys = Cache::getRedis()->keys($pattern);
        if (!empty($keys)) {
            Cache::getRedis()->del($keys);
        }

        // Also clear prize cache
        $prizePattern = 'prizes_' . $this->tenantId . '_*';
        $prizeKeys = Cache::getRedis()->keys($prizePattern);
        if (!empty($prizeKeys)) {
            Cache::getRedis()->del($prizeKeys);
        }
    }
}
