<?php

namespace App\Services;

use App\Models\Admin;
use App\Models\Employee;
use App\Models\PushSubscription;
use Minishlink\WebPush\WebPush;
use Minishlink\WebPush\Subscription;
use Illuminate\Support\Facades\Log;

class EmployeePointsNotificationService
{
    private NotificationService $notificationService;
    private WebPush $webPush;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        $this->webPush = new WebPush([
            'VAPID' => [
                'subject' => config('app.vapid_subject'),
                'publicKey' => config('app.vapid_public_key'),
                'privateKey' => config('app.vapid_private_key'),
            ]
        ]);
    }

    /**
     * Notify admins about new points request from employee
     * Now filters admins by employee's category
     */
    public function notifyAdminsAboutPointsRequest(
        int $employeeId,
        int $tenantId,
        int $totalPoints,
        array $targetNames,
        string $vinNo,
        string $employeeCategory = null
    ): bool {
        try {
            // Get employee details
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();
            if (!$employee) {
                Log::warning('Employee not found for points request notification', [
                    'employee_id' => $employeeId,
                    'tenant_id' => $tenantId
                ]);
                return false;
            }

            $employeeName = $employee->name;
            $category = $employeeCategory ?? $employee->category;
            $targetsText = !empty($targetNames) ? ' for ' . implode(', ', $targetNames) : '';
            
            $title = 'New Points Request 📝';
            $content = "{$employeeName} submitted a request for {$totalPoints} points{$targetsText}. VIN: {$vinNo}";

            // Get admins with same tenant_id and category as employee
            $admins = Employee::where('tenant_id', $tenantId)
                             ->where('role', 'admin')
                             ->where('category', $category)
                             ->get();
            if ($admins->isEmpty()) {
                Log::warning('No admins found for category', [
                    'tenant_id' => $tenantId,
                    'category' => $category,
                    'employee_id' => $employeeId
                ]);
                return false;
            }

            $notifiedCount = 0;

            foreach ($admins as $admin) {
                // Send database notification
                $this->notificationService->sendToUser(
                    $tenantId,
                    'admin',
                    $admin->employee_id, // Using employee_id as admin identifier
                    $title,
                    $content,
                    'employee',
                    $employeeId,
                    'points_request'
                );
                
                // Send browser push notification
                if ($this->sendAdminPushNotification($admin, $title, $content, 'points_request')) {
                    $notifiedCount++;
                }
            }

            Log::info('Category-filtered admin notifications sent for points request', [
                'employee_id' => $employeeId,
                'employee_name' => $employeeName,
                'employee_category' => $category,
                'tenant_id' => $tenantId,
                'total_points' => $totalPoints,
                'admins_total' => $admins->count(),
                'admins_notified' => $notifiedCount
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to notify admins about points request', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Notify admins about bulk points request from employee
     * Specifically for bulk requests with category filtering
     */
    public function notifyAdminsAboutBulkPointsRequest(
        int $employeeId,
        int $tenantId,
        int $totalPoints,
        array $targetNames,
        string $bulkVinNo,
        int $requestCount
    ): bool {
        try {
            // Get employee details
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();
            dd($employee);
            if (!$employee) {
                Log::warning('Employee not found for bulk points request notification', [
                    'employee_id' => $employeeId,
                    'tenant_id' => $tenantId
                ]);
                return false;
            }

            $employeeName = $employee->name;
            $category = $employee->category;
            $targetsText = !empty($targetNames) ? ' for ' . implode(', ', array_unique($targetNames)) : '';
            
            $title = 'New Bulk Points Request 📝📋';
            $content = "{$employeeName} submitted {$requestCount} bulk requests totaling {$totalPoints} points{$targetsText}. Batch ID: {$bulkVinNo}";

            // Get admins with same tenant_id and category as employee
            $admins = Employee::where('tenant_id', $tenantId)
                             ->where('role', 'admin')
                             ->where('category', $category)
                             ->get();

            if ($admins->isEmpty()) {
                Log::warning('No category-matched admins found for bulk request', [
                    'tenant_id' => $tenantId,
                    'category' => $category,
                    'employee_id' => $employeeId
                ]);
                return false;
            }

            $notifiedCount = 0;

            foreach ($admins as $admin) {
                // Send database notification
                $this->notificationService->sendToUser(
                    $tenantId,
                    'admin',
                    $admin->employee_id,
                    $title,
                    $content,
                    'employee',
                    $employeeId,
                    'bulk_points_request'
                );

                // Send browser push notification
                if ($this->sendAdminPushNotification($admin, $title, $content, 'bulk_points_request')) {
                    $notifiedCount++;
                }
            }

            Log::info('Bulk points request notifications sent to category admins', [
                'employee_id' => $employeeId,
                'employee_name' => $employeeName,
                'employee_category' => $category,
                'tenant_id' => $tenantId,
                'total_points' => $totalPoints,
                'request_count' => $requestCount,
                'admins_total' => $admins->count(),
                'admins_notified' => $notifiedCount
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to notify admins about bulk points request', [
                'employee_id' => $employeeId,
                'tenant_id' => $tenantId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Send confirmation notification to employee about request submission
     */
    public function sendEmployeeRequestConfirmation(
        int $employeeId,
        int $tenantId,
        int $totalPoints,
        array $targetNames
    ): bool {
        try {
            $targetsText = !empty($targetNames) ? ' for ' . implode(', ', $targetNames) : '';
            
            $title = 'Request Submitted ✅';
            $content = "Your request for {$totalPoints} points{$targetsText} has been submitted and is pending admin approval.";

            // Send database notification to employee
            $this->notificationService->sendToUser(
                $tenantId,
                'employee',
                $employeeId,
                $title,
                $content,
                'system',
                null,
                'points_submitted'
            );

            Log::info('Employee confirmation sent', [
                'employee_id' => $employeeId,
                'total_points' => $totalPoints,
                'targets_count' => count($targetNames)
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send employee confirmation', [
                'employee_id' => $employeeId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Notify admins about points request cancellation
     * Now filters by category
     */
    public function notifyAdminsAboutCancellation(
        int $employeeId,
        int $tenantId,
        int $points,
        string $targetName,
        string $vinNo
    ): bool {
        try {
            $employee = Employee::where('employee_id', $employeeId)
                               ->where('tenant_id', $tenantId)
                               ->first();
            
            if (!$employee) {
                return false;
            }

            $employeeName = $employee->name;
            $category = $employee->category;
            $title = 'Points Request Cancelled ❌';
            $content = "{$employeeName} cancelled their request for {$points} points for {$targetName} (VIN: {$vinNo})";

            // Get admins with same tenant_id and category as employee
            $admins = Employee::where('tenant_id', $tenantId)
                             ->where('role', 'admin')
                             ->where('category', $category)
                             ->get();
            
            foreach ($admins as $admin) {
                $this->notificationService->sendToUser(
                    $tenantId,
                    'admin',
                    $admin->employee_id,
                    $title,
                    $content,
                    'employee',
                    $employeeId,
                    'points_cancelled'
                );

                $this->sendAdminPushNotification($admin, $title, $content, 'points_cancelled');
            }

            Log::info('Category-filtered admin notifications sent for points cancellation', [
                'employee_id' => $employeeId,
                'employee_name' => $employeeName,
                'employee_category' => $category,
                'points' => $points,
                'target' => $targetName,
                'admins_notified' => $admins->count()
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to notify admins about cancellation', [
                'employee_id' => $employeeId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send reminder notification to employee about pending requests
     */
    public function sendPendingRequestReminder(
        int $employeeId,
        int $tenantId,
        int $pendingCount,
        int $totalPendingPoints
    ): bool {
        try {
            $title = 'Pending Points Requests 📋';
            $content = "You have {$pendingCount} pending points requests totaling {$totalPendingPoints} points awaiting admin approval.";

            $this->notificationService->sendToUser(
                $tenantId,
                'employee',
                $employeeId,
                $title,
                $content,
                'system',
                null,
                'points_reminder'
            );

            return true;

        } catch (\Exception $e) {
            Log::error('Failed to send pending request reminder', [
                'employee_id' => $employeeId,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send browser push notification to admin
     * Updated to work with Employee model instead of Admin model
     */
    private function sendAdminPushNotification($admin, string $title, string $content, string $type): bool
    {
        try {
            // Get push subscriptions for admin (assuming you have a relationship or method)
            $subscriptions = $admin->pushSubscriptions ?? collect();
            
            if ($subscriptions->isEmpty()) {
                Log::debug('No push subscriptions for admin', ['admin_employee_id' => $admin->employee_id]);
                return false;
            }

            $payload = $this->createAdminPushPayload($title, $content, $type);
            $successCount = 0;

            foreach ($subscriptions as $subscription) {
                if ($this->sendToSubscription($subscription, $payload)) {
                    $successCount++;
                }
            }

            Log::debug('Push notifications sent to admin', [
                'admin_employee_id' => $admin->employee_id,
                'subscriptions_count' => $subscriptions->count(),
                'successful_sends' => $successCount
            ]);

            return $successCount > 0;

        } catch (\Exception $e) {
            Log::error('Admin push notification failed', [
                'admin_employee_id' => $admin->employee_id ?? 'unknown',
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Create push notification payload for admins
     */
    private function createAdminPushPayload(string $title, string $content, string $type): array
    {
        // Limit content length for push notifications
        if (strlen($content) > 100) {
            $content = substr($content, 0, 97) . '...';
        }

        $actionMap = [
            'points_request' => [
                'url' => url('/admin/points/requests'),
                'action_title' => 'Review Request'
            ],
            'bulk_points_request' => [
                'url' => url('/admin/points/requests'),
                'action_title' => 'Review Bulk Request'
            ],
            'points_cancelled' => [
                'url' => url('/admin/points/cancelled'),
                'action_title' => 'View Cancelled'
            ],
            'default' => [
                'url' => url('/admin/dashboard'),
                'action_title' => 'View Dashboard'
            ]
        ];

        $actionData = $actionMap[$type] ?? $actionMap['default'];

        return [
            'title' => $title,
            'body' => $content,
            'icon' => '/favicon.ico',
            'badge' => '/favicon.ico',
            'tag' => 'employee-points-' . $type . '-' . time(),
            'url' => $actionData['url'],
            'data' => [
                'type' => $type,
                'category' => 'employee_points',
                'url' => $actionData['url'],
                'timestamp' => time()
            ],
            'requireInteraction' => true,
            'renotify' => true,
            'silent' => false,
            'actions' => [
                [
                    'action' => 'view',
                    'title' => $actionData['action_title'],
                    'icon' => '/favicon.ico'
                ]
            ]
        ];
    }

    /**
     * Send notification to a specific push subscription
     */
    private function sendToSubscription(PushSubscription $pushSubscription, array $payload): bool
    {
        try {
            $subscription = Subscription::create($pushSubscription->getSubscriptionData());
            
            $result = $this->webPush->sendOneNotification(
                $subscription,
                json_encode($payload)
            );

            // Remove invalid subscriptions
            if (!$result->isSuccess()) {
                if ($result->getStatusCode() === 410 || $result->getStatusCode() === 404) {
                    Log::info('Removing invalid admin push subscription', [
                        'subscription_id' => $pushSubscription->id,
                        'status_code' => $result->getStatusCode()
                    ]);
                    $pushSubscription->delete();
                } else {
                    Log::warning('Admin push notification failed', [
                        'subscription_id' => $pushSubscription->id,
                        'status_code' => $result->getStatusCode(),
                        'reason' => $result->getReason()
                    ]);
                }
                return false;
            }

            Log::debug('Admin push notification sent successfully', [
                'subscription_id' => $pushSubscription->id
            ]);

            return true;

        } catch (\Exception $e) {
            Log::error('Admin push notification exception', [
                'subscription_id' => $pushSubscription->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Bulk notify admins about multiple employee requests
     * Updated to filter by category
     */
    public function bulkNotifyAdminsAboutRequests(
        int $tenantId,
        array $requestData,
        string $batchId = null
    ): array {
        $results = [
            'success' => 0,
            'failed' => 0,
            'details' => []
        ];

        $batchId = $batchId ?? 'BATCH_' . time();

        foreach ($requestData as $request) {
            $success = $this->notifyAdminsAboutPointsRequest(
                $request['employee_id'],
                $tenantId,
                $request['total_points'],
                $request['target_names'],
                $request['vin_no'],
                $request['employee_category'] ?? null
            );

            if ($success) {
                $results['success']++;
                $results['details'][] = [
                    'employee_id' => $request['employee_id'],
                    'status' => 'success'
                ];
            } else {
                $results['failed']++;
                $results['details'][] = [
                    'employee_id' => $request['employee_id'],
                    'status' => 'failed'
                ];
            }
        }

        Log::info('Bulk admin notifications completed', [
            'batch_id' => $batchId,
            'tenant_id' => $tenantId,
            'total_requests' => count($requestData),
            'successful' => $results['success'],
            'failed' => $results['failed']
        ]);

        return $results;
    }
}