<?php
// app/Services/ChatPushNotificationService.php

namespace App\Services;

use App\Models\PushSubscription;
use App\Models\Employee;
use App\Models\Admin;
use App\Models\Conversation;
use App\Models\Message;
use App\Models\ConversationParticipant;
use Minishlink\WebPush\WebPush;
use Minishlink\WebPush\Subscription;
use Illuminate\Support\Facades\Log;

class ChatPushNotificationService
{
    private WebPush $webPush;

    public function __construct()
    {
        $this->webPush = new WebPush([
            'VAPID' => [
                'subject' => config('app.vapid_subject'),
                'publicKey' => config('app.vapid_public_key'),
                'privateKey' => config('app.vapid_private_key'),
            ]
        ]);
    }

    /**
     * Send push notification for a new chat message
     */
    public function sendNewMessageNotification(Message $message): void
    {
        // dd($message->sender_id);
        $conversation = $message->conversation;
        $sender = $this->getMessageSender($message);
        // dd($sender);
        if (!$sender) {
            Log::warning('Unable to determine message sender', ['message_id' => $message->id]);
            return;
        }
        // Get all participants in the conversation EXCEPT the sender
        $participants = ConversationParticipant::where('conversation_id', $conversation->id)
        // ->where('tenant_id', $conversation->tenant_id)
            ->where(function($query) use ($message) {
                // Exclude the sender from receiving notifications
                if ($message->sender_type == 'admin') {
                    $query->where(function($q) use ($message) {
                        $q->where('admin_id', '!=', $message->sender_admin_id)
                          ->orWhereNull('admin_id');
                    });
                } else {
                    $query->where(function($q) use ($message) {
                        $q->where('employee_id', '!=', $message->sender_id)
                          ->orWhereNull('employee_id');
                    });
                }
            })
            ->get();

        Log::info('Found participants to notify', [
            'total_participants' => $participants->count(),
            'conversation_id' => $conversation->id
        ]);
        $notificationsSent = 0;
        foreach ($participants as $participant) {
            if ($participant->employee_id) {
                $sent = $this->sendToEmployee($participant->employee_id, $message, $sender, $conversation);
                if ($sent) $notificationsSent++;
            } elseif ($participant->admin_id) {
                $sent = $this->sendToAdmin($participant->admin_id, $message, $sender, $conversation);
                if ($sent) $notificationsSent++;
            }
        }
        Log::info('Push notifications sent', [
            'message_id' => $message->id,
            'notifications_sent' => $notificationsSent,
            'total_participants' => $participants->count()
        ]);
    }

    /**
     * Send notification to a specific employee
     */
    private function sendToEmployee(int $employeeId, Message $message, $sender, Conversation $conversation): bool
    {
        $employee = Employee::where('employee_id', $employeeId)
                           ->where('tenant_id', $conversation->tenant_id)
                           ->first();

        if (!$employee) {
            Log::warning('Employee not found for notification', ['employee_id' => $employeeId]);
            return false;
        }

        $subscriptions = $employee->pushSubscriptions;
        
        if ($subscriptions->isEmpty()) {
            Log::info('No push subscriptions for employee', ['employee_id' => $employeeId]);
            return false;
        }

        $payload = $this->createNotificationPayload($message, $sender, $conversation);
        $successCount = 0;

        foreach ($subscriptions as $subscription) {
            if ($this->sendToSubscription($subscription, $payload)) {
                $successCount++;
            }
        }

        Log::info('Sent notifications to employee', [
            'employee_id' => $employeeId,
            'subscriptions_count' => $subscriptions->count(),
            'successful_sends' => $successCount
        ]);

        return $successCount > 0;
    }

    /**
     * Send notification to a specific admin
     */
    private function sendToAdmin(int $adminId, Message $message, $sender, Conversation $conversation): bool
    {
        $admin = Admin::where('id', $adminId)
                     ->where('tenant_id', $conversation->tenant_id)
                     ->first();

        if (!$admin) {
            Log::warning('Admin not found for notification', ['admin_id' => $adminId]);
            return false;
        }

        $subscriptions = $admin->pushSubscriptions;
        
        if ($subscriptions->isEmpty()) {
            Log::info('No push subscriptions for admin', ['admin_id' => $adminId]);
            return false;
        }

        $payload = $this->createNotificationPayload($message, $sender, $conversation);
        $successCount = 0;

        foreach ($subscriptions as $subscription) {
            if ($this->sendToSubscription($subscription, $payload)) {
                $successCount++;
            }
        }

        Log::info('Sent notifications to admin', [
            'admin_id' => $adminId,
            'subscriptions_count' => $subscriptions->count(),
            'successful_sends' => $successCount
        ]);

        return $successCount > 0;
    }

    /**
     * Create notification payload
     */
    private function createNotificationPayload(Message $message, $sender, Conversation $conversation): array
    {
        $senderName = $this->getSenderDisplayName($sender);
        $conversationName = $this->getConversationDisplayName($conversation, $sender);
        
        $title = $conversation->type === 'group' 
            ? $conversationName 
            : $senderName;

        $body = $conversation->type === 'group'
            ? "{$senderName}: {$message->content}"
            : $message->content;

        // Limit body length
        if (strlen($body) > 100) {
            $body = substr($body, 0, 97) . '...';
        }

        return [
            'title' => $title,
            'body' => $body,
            'icon' => '/favicon.ico', // Update this path as needed
            'badge' => '/favicon.ico', // Update this path as needed
            'tag' => 'chat-' . $conversation->id,
            'url' => url("/chat/conversation/{$conversation->id}"),
            'data' => [
                'conversation_id' => $conversation->id,
                'message_id' => $message->id,
                'type' => 'chat_message',
                'url' => url("/chat/conversation/{$conversation->id}")
            ],
            'requireInteraction' => false,
            'renotify' => true,
            'silent' => false
        ];
    }

    /**
     * Send to a specific push subscription
     */
    private function sendToSubscription(PushSubscription $pushSubscription, array $payload): bool
    {
        try {
            $subscription = Subscription::create($pushSubscription->getSubscriptionData());
            
            $result = $this->webPush->sendOneNotification(
                $subscription,
                json_encode($payload)
            );

            // Remove invalid subscriptions
            if (!$result->isSuccess()) {
                if ($result->getStatusCode() === 410 || $result->getStatusCode() === 404) {
                    Log::info('Removing invalid push subscription', [
                        'subscription_id' => $pushSubscription->id,
                        'status_code' => $result->getStatusCode()
                    ]);
                    $pushSubscription->delete();
                } else {
                    Log::warning('Push notification failed', [
                        'subscription_id' => $pushSubscription->id,
                        'status_code' => $result->getStatusCode(),
                        'reason' => $result->getReason()
                    ]);
                }
                return false;
            }

            Log::debug('Push notification sent successfully', [
                'subscription_id' => $pushSubscription->id
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Push notification exception', [
                'subscription_id' => $pushSubscription->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Get message sender
     */
    private function getMessageSender(Message $message)
    {
        if ($message->sender_type === 'admin') {
            return $message->senderAdmin;
        }
        return $message->sender;
    }

    /**
     * Get sender display name
     */
    private function getSenderDisplayName($sender): string
    {
        if ($sender instanceof Admin) {
            return $sender->display_name ?? $sender->email;
        }
        
        if ($sender instanceof Employee) {
            return $sender->name;
        }

        return 'Unknown User';
    }

    /**
     * Get conversation display name
     */
    private function getConversationDisplayName(Conversation $conversation, $sender): string
    {
        if ($conversation->type === 'group' && $conversation->name) {
            return $conversation->name;
        }

        if ($conversation->type === 'group') {
            return 'Group Chat';
        }

        // For personal chats, you might want to get the other participant's name
        return 'Chat';
    }
}