<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Tenant extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'domain',
        'active',
        'plan',
        'phone',
        'address',
        'logo',
        'admin_id',
        'stripe_customer_id',
        'stripe_subscription_id',
        'trial_ends_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'active' => 'boolean',
        'trial_ends_at' => 'datetime',
    ];

    /**
     * Scope a query to only include active tenants.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('active', true);
    }

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'domain';
    }

    /**
     * Get the admins for the tenant.
     */
    public function admins(): HasMany
    {
        return $this->hasMany(Admin::class);
    }

    /**
     * Get the departments for the tenant.
     */
    public function departments(): HasMany
    {
        return $this->hasMany(Department::class);
    }

    /**
     * Get the employees for the tenant.
     */
    public function employees(): HasMany
    {
        return $this->hasMany(Employee::class);
    }

    /**
     * Get the products for the tenant.
     */
    public function products(): HasMany
    {
        return $this->hasMany(Product::class);
    }

    /**
     * Get the targets for the tenant.
     */
    public function targets(): HasMany
    {
        return $this->hasMany(Target::class);
    }

    /**
     * Get the points for the tenant.
     */
    public function points(): HasMany
    {
        return $this->hasMany(Points::class);
    }

    /**
     * Get the department leaderboards for the tenant.
     */
    public function departmentLeaderboards(): HasMany
    {
        return $this->hasMany(DepartmentLeaderBoard::class);
    }

    /**
     * Get the employee leaderboards for the tenant.
     */
    public function employeeLeaderboards(): HasMany
    {
        return $this->hasMany(EmployeeLeaderBoard::class);
    }

    /**
     * Get the category leaderboards for the tenant.
     */
    public function categoryLeaderboards(): HasMany
    {
        return $this->hasMany(CategoryLeaderBoard::class, 'tenant_id');
    }

    /**
     * Check if the tenant is on a specific plan
     *
     * @param string $planName
     * @return bool
     */
    public function onPlan(string $planName): bool
    {
        return $this->plan === $planName;
    }

    /**
     * Check if the tenant is on a trial period
     *
     * @return bool
     */
    public function onTrial(): bool
    {
        return $this->trial_ends_at && $this->trial_ends_at->isFuture();
    }

    /**
     * Get tenant URL
     *
     * @param string $path
     * @return string
     */
    public function url(string $path = ''): string
    {
        $scheme = request()->isSecure() ? 'https://' : 'http://';
        $domain = $this->domain . '.' . config('app.url_base', 'yourdomain.com');
        
        return $scheme . $domain . '/' . ltrim($path, '/');
    }
}