<?php
// app/Models/PushSubscription.php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Traits\BelongsToTenant;

class PushSubscription extends Model
{
    use BelongsToTenant;

    protected $fillable = [
        'tenant_id',
        'employee_id',
        'admin_id',
        'endpoint',
        'public_key',
        'auth_token',
        'content_encoding'
    ];

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Employee::class, 'employee_id', 'employee_id');
    }

    public function admin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'admin_id');
    }

    /**
     * Get the subscription data in the format expected by web-push
     */
    public function getSubscriptionData(): array
    {
        return [
            'endpoint' => $this->endpoint,
            'keys' => [
                'p256dh' => $this->public_key,
                'auth' => $this->auth_token,
            ],
            'contentEncoding' => $this->content_encoding,
        ];
    }

    /**
     * Get the user (employee or admin) who owns this subscription
     */
    public function getOwnerAttribute()
    {
        return $this->employee ?: $this->admin;
    }

    /**
     * Get the user type (employee or admin)
     */
    public function getUserTypeAttribute()
    {
        return $this->employee_id ? 'employee' : 'admin';
    }
}