<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class Prize extends Model
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'position',
        'title',
        'description',
        'prize_amount',
        'prize_type',
        'currency',
        'category',
        'time_period',
        'is_active',
        'start_date',
        'end_date',
        'created_by_type',
        'created_by_id'
    ];

    protected $casts = [
        'prize_amount' => 'decimal:2',
        'is_active' => 'boolean',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
    ];

    /**
     * Get the tenant that owns the prize
     */
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    /**
     * Scope for active prizes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope for current time period
     */
    public function scopeCurrentPeriod($query, $timePeriod = 'current_month')
    {
        return $query->where('time_period', $timePeriod);
    }

    /**
     * Scope for specific position
     */
    public function scopeForPosition($query, $position)
    {
        return $query->where('position', $position);
    }

    /**
     * Scope for specific category
     */
    public function scopeForCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Scope for tenant
     */
    public function scopeForTenant($query, $tenantId)
    {
        return $query->where('tenant_id', $tenantId);
    }

    /**
     * Scope for currently valid prizes (within date range)
     */
    public function scopeCurrentlyValid($query)
    {
        $now = Carbon::now();
        return $query->where(function($q) use ($now) {
            $q->whereNull('start_date')
              ->orWhere('start_date', '<=', $now);
        })->where(function($q) use ($now) {
            $q->whereNull('end_date')
              ->orWhere('end_date', '>=', $now);
        });
    }

    /**
     * Get formatted prize amount
     */
    public function getFormattedAmountAttribute()
    {
        if (!$this->prize_amount) {
            return null;
        }

        $symbols = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'CAD' => 'C$',
            'AUD' => 'A$'
        ];

        $symbol = $symbols[$this->currency] ?? $this->currency . ' ';
        return $symbol . number_format($this->prize_amount, 2);
    }

    /**
     * Get position name
     */
    public function getPositionNameAttribute()
    {
        $positions = [
            1 => '1st Place',
            2 => '2nd Place',
            3 => '3rd Place'
        ];

        return $positions[$this->position] ?? "Position {$this->position}";
    }

    /**
     * Get prize type display name
     */
    public function getPrizeTypeDisplayAttribute()
    {
        $types = [
            'money' => 'Cash Prize',
            'gift' => 'Gift',
            'recognition' => 'Recognition',
            'bonus' => 'Bonus',
            'voucher' => 'Voucher',
            'trophy' => 'Trophy'
        ];

        return $types[$this->prize_type] ?? ucfirst($this->prize_type);
    }

    /**
     * Check if prize is currently active and valid
     */
    public function isCurrentlyValid()
    {
        if (!$this->is_active) {
            return false;
        }

        $now = Carbon::now();
        
        if ($this->start_date && $this->start_date->gt($now)) {
            return false;
        }

        if ($this->end_date && $this->end_date->lt($now)) {
            return false;
        }

        return true;
    }

    /**
     * Get creator information
     */
    public function getCreatorAttribute()
    {
        if (!$this->created_by_type || !$this->created_by_id) {
            return null;
        }

        switch ($this->created_by_type) {
            case 'admin':
                return Admin::find($this->created_by_id);
            case 'manager':
                return Employee::where('role', 'Manager')->find($this->created_by_id);
            case 'superadmin':
                return Tenant::find($this->created_by_id);
            default:
                return null;
        }
    }
}