<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Notification extends Model
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'sender_type',
        'sender_id',
        'receiver_type',
        'receiver_id',
        'title',
        'content',
        'is_read',
        'read_at',
        'type',
        'related_id',
    ];

    protected $casts = [
        'is_read' => 'boolean',
        'read_at' => 'datetime',
    ];

    /**
     * Get the sender name
     */
    public function getSenderNameAttribute()
    {
        if ($this->sender_type === 'system') {
            return 'System';
        }
        
        if ($this->sender_type === 'admin') {
            $admin = Admin::find($this->sender_id);
            return $admin ? $admin->email : 'Admin';
        }
        
        if ($this->sender_type === 'employee') {
            $employee = Employee::where('employee_id', $this->sender_id)->first();
            return $employee ? $employee->name : 'Employee';
        }
        
        return 'Unknown';
    }

    /**
     * Get the receiver name
     */
    public function getReceiverNameAttribute()
    {
        if ($this->receiver_type === 'admin') {
            $admin = Admin::find($this->receiver_id);
            return $admin ? $admin->email : 'Admin';
        }
        
        if ($this->receiver_type === 'employee') {
            $employee = Employee::where('employee_id', $this->receiver_id)->first();
            return $employee ? $employee->name : 'Employee';
        }
        
        return 'Unknown';
    }

    /**
     * Mark notification as read
     */
    public function markAsRead()
    {
        return $this->update([
            'is_read' => true,
            'read_at' => now(),
        ]);
    }

    /**
     * Scope for unread notifications
     */
    public function scopeUnread($query)
    {
        return $query->where('is_read', false);
    }

    /**
     * Scope for specific receiver
     */
    public function scopeForReceiver($query, $type, $id)
    {
        return $query->where('receiver_type', $type)->where('receiver_id', $id);
    }

    /**
     * Scope for specific tenant
     */
    public function scopeForTenant($query, $tenantId)
    {
        return $query->where('tenant_id', $tenantId);
    }

    /**
     * Scope for specific type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Get time ago
     */
    public function getTimeAgoAttribute()
    {
        return $this->created_at->diffForHumans();
    }
}