<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class MessageRead extends Model
{
    use HasFactory;

    protected $fillable = [
        'message_id', 
        'reader_type',
        'employee_id', 
        'admin_id',
        'read_at'
    ];

    public $timestamps = false;

    protected $casts = [
        'read_at' => 'datetime',
    ];

    public function message()
    {
        return $this->belongsTo(Message::class);
    }

    public function employee()
    {
        return $this->belongsTo(Employee::class, 'employee_id', 'employee_id');
    }

    public function admin()
    {
        return $this->belongsTo(Admin::class, 'admin_id');
    }

    // Get the reader model (either employee or admin)
    public function getReaderAttribute()
    {
        if ($this->reader_type === 'admin') {
            return $this->admin;
        }
        return $this->employee;
    }

    // Get reader name
    public function getReaderNameAttribute()
    {
        if ($this->reader_type === 'admin') {
            return $this->admin ? $this->admin->display_name : 'Admin';
        }
        return $this->employee ? $this->employee->name : 'Unknown';
    }

    // Create a read record for a user
    public static function markAsRead($messageId, $userType, $userId)
    {
        $data = [
            'message_id' => $messageId,
            'reader_type' => $userType,
            'read_at' => now()
        ];

        if ($userType === 'admin') {
            $data['admin_id'] = $userId;
            $data['employee_id'] = null;
        } else {
            $data['employee_id'] = $userId;
            $data['admin_id'] = null;
        }

        return static::firstOrCreate(
            [
                'message_id' => $messageId,
                'reader_type' => $userType,
                $userType === 'admin' ? 'admin_id' : 'employee_id' => $userId
            ],
            $data
        );
    }
}