<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Message extends Model
{
    use HasFactory;

    protected $fillable = [
        'conversation_id', 'sender_id', 'sender_admin_id', 'sender_type', 'content', 
        'type', 'attachments', 'is_edited', 'edited_at'
    ];

    protected $casts = [
        'attachments' => 'array',
        'is_edited' => 'boolean',
        'edited_at' => 'datetime',
    ];

    public function conversation()
    {
        return $this->belongsTo(Conversation::class);
    }

    public function sender()
    {
        return $this->belongsTo(Employee::class, 'sender_id', 'employee_id');
    }
    
    public function senderAdmin()
    {
        return $this->belongsTo(Admin::class, 'sender_admin_id');
    }
    
    public function reads()
    {
        return $this->hasMany(MessageRead::class);
    }

    public function readByEmployees()
    {
        return $this->belongsToMany(Employee::class, 'message_reads', 'message_id', 'employee_id')
                    ->withPivot('read_at');
    }

    public function isReadBy($employeeId)
    {
        return $this->reads()->where('employee_id', $employeeId)->exists();
    }

    public function markAsReadBy($employeeId)
    {
        if (!$this->isReadBy($employeeId) && $this->sender_id != $employeeId) {
            MessageRead::create([
                'message_id' => $this->id,
                'employee_id' => $employeeId,
            ]);
        }
    }
    
    // Get sender name regardless of type
    public function getSenderNameAttribute()
    {
        if ($this->sender_admin_id) {
            return $this->senderAdmin ? $this->senderAdmin->display_name : 'Admin';
        }
        return $this->sender ? $this->sender->name : 'Unknown';
    }
    
    // Get full sender display name with type indicator
    public function getFullSenderNameAttribute()
    {
        if ($this->sender_admin_id) {
            return $this->senderAdmin ? 'Admin: ' . $this->senderAdmin->display_name : 'Admin';
        }
        return $this->sender ? $this->sender->name : 'Unknown';
    }
    
    // Check if message is from admin
    public function isFromAdmin()
    {
        return !empty($this->sender_admin_id);
    }
    
    // Check if message is from employee
    public function isFromEmployee()
    {
        return !empty($this->sender_id);
    }
}