<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CustomEmoji extends Model
{
    use HasFactory;

    protected $table = 'custom_emojis';

    protected $fillable = [
        'name',
        'type',
        'emoji_text', // For text emojis
        'file_path', // For GIF images
        'file_name',
        'mime_type',
        'file_size',
        'tenant_id',
        'created_by',
        'is_active'
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    // Relationship with Tenant
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    // Get creator (can be SuperAdmin, Admin, or Employee)
    public function getCreator()
    {
        // Try to find the creator in different user tables
        $superAdmin = \App\Models\SuperAdmin::find($this->created_by);
        if ($superAdmin) {
            return $superAdmin;
        }
        
        $admin = \App\Models\Admin::find($this->created_by);
        if ($admin) {
            return $admin;
        }
        
        $employee = \App\Models\Employee::find($this->created_by);
        if ($employee) {
            return $employee;
        }
        
        return null;
    }
    
    // Get creator name
    public function getCreatorNameAttribute()
    {
        $creator = $this->getCreator();
        return $creator ? $creator->name : 'Unknown';
    }

    // Scope for active emojis
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope for emojis by type
    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    // Scope for tenant
    public function scopeForTenant($query, $tenantId)
    {
        return $query->where('tenant_id', $tenantId);
    }

    // Get display content for emoji/GIF
    public function getDisplayContentAttribute()
    {
        if ($this->type === 'emoji') {
            return $this->emoji_text; // Return text emoji
        } else {
            return asset($this->file_path); // Return GIF URL (direct from public folder)
        }
    }

    // Get full URL for the emoji/GIF (for backward compatibility)
    public function getUrlAttribute()
    {
        if ($this->type === 'emoji') {
            return $this->emoji_text; // Return text emoji
        } else {
            return asset($this->file_path); // Return GIF URL (direct from public folder)
        }
    }
}