<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\BelongsToTenant;

class Conversation extends Model
{
    use HasFactory, BelongsToTenant;

    protected $fillable = [
        'tenant_id', 'type', 'name', 'description', 
        'avatar', 'created_by', 'created_by_type', 'created_by_id', 'last_message_at'
    ];

    protected $casts = [
        'last_message_at' => 'datetime',
    ];

    public function participants()
    {
        return $this->hasMany(ConversationParticipant::class);
    }

    public function messages()
    {
        return $this->hasMany(Message::class);
    }

    public function lastMessage()
    {
        return $this->hasOne(Message::class)->latest();
    }

    public function creator()
    {
        return $this->belongsTo(Employee::class, 'created_by', 'employee_id');
    }

    public function isGroupChat()
    {
        return $this->type === 'group';
    }

    public function isPersonalChat()
    {
        return $this->type === 'personal';
    }

    public function participantEmployees()
    {
        return $this->belongsToMany(Employee::class, 'conversation_participants', 'conversation_id', 'employee_id')
                    ->withPivot('role', 'joined_at', 'last_read_at', 'is_muted')
                    ->withTimestamps();
    }

    public function participantAdmins()
    {
        return $this->belongsToMany(Admin::class, 'conversation_participants', 'conversation_id', 'admin_id')
            ->withPivot('role', 'last_read_at')
            ->withTimestamps();
    }

    // Get all participants (both employees and admins)
    public function getAllParticipants()
    {
        $employees = $this->participantEmployees->map(function ($employee) {
            $employee->participant_type = 'employee';
            return $employee;
        });

        $admins = $this->participantAdmins->map(function ($admin) {
            $admin->participant_type = 'admin';
            return $admin;
        });

        return $employees->concat($admins);
    }

    public function getUnreadCountForEmployee($employeeId)
    {
        $participant = $this->participants()->where('employee_id', $employeeId)->first();
        
        if (!$participant) {
            return 0;
        }

        return $this->messages()
                    ->where('created_at', '>', $participant->last_read_at ?? $participant->joined_at)
                    ->where('sender_id', '!=', $employeeId)
                    ->where(function($q) use ($employeeId) {
                        $q->where('sender_type', 'employee')
                          ->orWhere('sender_type', 'admin');
                    })
                    ->count();
    }

    // Get unread count for a specific user
    public function getUnreadCountForUser($userType, $userId)
    {
        $participant = $this->participants()
            ->where($userType === 'admin' ? 'admin_id' : 'employee_id', $userId)
            ->first();

        if (!$participant) {
            return 0;
        }

        return $this->messages()
            ->where('created_at', '>', $participant->last_read_at ?? '1970-01-01')
            ->where(function($q) use ($userType, $userId) {
                // Don't count messages sent by the user themselves
                if ($userType === 'admin') {
                    $q->where('sender_admin_id', '!=', $userId)
                      ->orWhereNull('sender_admin_id');
                } else {
                    $q->where('sender_id', '!=', $userId)
                      ->orWhereNull('sender_id');
                }
            })
            ->count();
    }

    // Get conversation display name for personal chats
    public function getDisplayNameForUser($userType, $userId)
    {
        if ($this->type === 'group') {
            return $this->name;
        }

        // For personal chats, show the other participant's name
        $otherParticipant = $this->participants()
            ->where(function($q) use ($userType, $userId) {
                if ($userType === 'admin') {
                    $q->where('admin_id', '!=', $userId)
                      ->orWhereNotNull('employee_id');
                } else {
                    $q->where('employee_id', '!=', $userId)
                      ->orWhereNotNull('admin_id');
                }
            })
            ->first();

        if ($otherParticipant) {
            if ($otherParticipant->admin_id) {
                return $otherParticipant->admin ? 'Admin: ' . $otherParticipant->admin->email : 'Admin';
            } else {
                return $otherParticipant->employee ? $otherParticipant->employee->name : 'Employee';
            }
        }

        return 'Personal Chat';
    }
}