<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use App\Traits\BelongsToTenant;

class BusinessCard extends Model
{
    use HasFactory, BelongsToTenant;

    protected $fillable = [
        'tenant_id',
        'user_id',
        'user_type',
        'name',
        'title',
        'company_name',
        'phone',
        'email',
        'website',
        'bio',
        'specialties',
        'brands_worked_with',
        'certifications',
        'years_experience',
        'languages_spoken',
        'services_offered',
        'location',
        'preferred_contact_method',
        'profile_photo',
        'company_logo',
        'background_image',
        'social_links',
        'whatsapp',
        'linkedin',
        'facebook',
        'instagram',
        'twitter',
        'template_id',
        'custom_colors',
        'font_style',
        'unique_slug',
        'is_active',
        'is_public',
        'view_count',
        'last_viewed_at',
        'qr_code',
        'additional_notes',
        'business_hours'
    ];

    protected $casts = [
        'specialties' => 'array',
        'brands_worked_with' => 'array',
        'certifications' => 'array',
        'social_links' => 'array',
        'custom_colors' => 'array',
        'business_hours' => 'array',
        'is_active' => 'boolean',
        'is_public' => 'boolean',
        'view_count' => 'integer',
        'years_experience' => 'integer',
        'last_viewed_at' => 'datetime'
    ];

    // Boot method to generate unique slug
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($businessCard) {
            if (empty($businessCard->unique_slug)) {
                $businessCard->unique_slug = $businessCard->generateUniqueSlug();
            }
        });
    }

    // Generate unique slug
    public function generateUniqueSlug()
    {
        $baseSlug = Str::slug($this->name . '-' . $this->company_name);
        $slug = $baseSlug;
        $counter = 1;

        while (static::where('unique_slug', $slug)->exists()) {
            $slug = $baseSlug . '-' . $counter;
            $counter++;
        }

        return $slug;
    }

    // Relationships
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    public function user()
    {
        if ($this->user_type === 'employee') {
            return $this->belongsTo(Employee::class, 'user_id', 'employee_id');
        } else {
            return $this->belongsTo(Admin::class, 'user_id', 'id');
        }
    }

    public function employee()
    {
        return $this->belongsTo(Employee::class, 'user_id', 'employee_id');
    }

    public function admin()
    {
        return $this->belongsTo(Admin::class, 'user_id', 'id');
    }

    // Accessors
    public function getPublicUrlAttribute()
    {
        return route('business-card.public', $this->unique_slug);
    }

    public function getProfilePhotoUrlAttribute()
    {
        if ($this->profile_photo) {
            return asset($this->profile_photo);
        }
        return asset('images/default-avatar.png');
    }

    public function getCompanyLogoUrlAttribute()
    {
        if ($this->company_logo) {
            return asset($this->company_logo);
        }
        return null;
    }

    public function getBackgroundImageUrlAttribute()
    {
        if ($this->background_image) {
            return asset($this->background_image);
        }
        return null;
    }

    // Get template name
    public function getTemplateNameAttribute()
    {
        $templates = [
            1 => 'Modern Dealership',
            2 => 'Luxury Professional',
            3 => 'Service Specialist',
            4 => 'Sales Champion',
            5 => 'Executive Elite'
        ];

        return $templates[$this->template_id] ?? 'Modern Dealership';
    }

    // Get primary color
    public function getPrimaryColorAttribute()
    {
        return $this->custom_colors['primary'] ?? '#007bff';
    }

    // Get secondary color
    public function getSecondaryColorAttribute()
    {
        return $this->custom_colors['secondary'] ?? '#6c757d';
    }

    // Increment view count
    public function incrementViewCount()
    {
        $this->increment('view_count');
        $this->update(['last_viewed_at' => now()]);
    }

    // Check if user can edit this card
    public function canEdit($user, $userType)
    {
        return $this->user_id == $user->id && $this->user_type == $userType;
    }

    // Get formatted specialties
    public function getFormattedSpecialtiesAttribute()
    {
        if (empty($this->specialties)) {
            return '';
        }
        return implode(' • ', $this->specialties);
    }

    // Get formatted brands
    public function getFormattedBrandsAttribute()
    {
        if (empty($this->brands_worked_with)) {
            return '';
        }
        return implode(', ', $this->brands_worked_with);
    }

    // Get formatted certifications
    public function getFormattedCertificationsAttribute()
    {
        if (empty($this->certifications)) {
            return '';
        }
        return implode(' • ', $this->certifications);
    }

    // Scope for active cards
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope for public cards
    public function scopePublic($query)
    {
        return $query->where('is_public', true);
    }

    // Get business hours for today
    public function getTodayHoursAttribute()
    {
        if (empty($this->business_hours)) {
            return null;
        }

        $today = strtolower(now()->format('l')); // e.g., 'monday'
        return $this->business_hours[$today] ?? null;
    }

    // Check if open now
    public function getIsOpenNowAttribute()
    {
        $todayHours = $this->today_hours;
        if (!$todayHours || $todayHours === 'Closed') {
            return false;
        }

        if (strpos($todayHours, '-') === false) {
            return true; // 24/7 or special format
        }

        list($open, $close) = explode('-', $todayHours);
        $now = now()->format('H:i');
        
        return $now >= trim($open) && $now <= trim($close);
    }

    // Get contact methods
    public function getContactMethodsAttribute()
    {
        $methods = [];
        
        if ($this->phone) {
            $methods[] = [
                'type' => 'phone',
                'value' => $this->phone,
                'label' => 'Phone',
                'icon' => 'fas fa-phone'
            ];
        }
        
        if ($this->email) {
            $methods[] = [
                'type' => 'email',
                'value' => $this->email,
                'label' => 'Email',
                'icon' => 'fas fa-envelope'
            ];
        }
        
        if ($this->whatsapp) {
            $methods[] = [
                'type' => 'whatsapp',
                'value' => $this->whatsapp,
                'label' => 'WhatsApp',
                'icon' => 'fab fa-whatsapp'
            ];
        }

        return $methods;
    }
}