<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Auth\User as Authenticatable;
use App\Traits\BelongsToTenant;

class Admin extends Authenticatable
{
    use HasFactory;
    use BelongsToTenant;

    protected $guarded = [];

    protected $fillable = ['email', 'password', 'tenant_id', 'subadmin', 'name'];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }

    // In your Admin model, add:
    public function employee()
    {
        return $this->belongsTo(Employee::class, 'employee_id', 'employee_id');
    }
// Add this relationship to your existing Admin model
public function pushSubscriptions()
{
    return $this->hasMany(PushSubscription::class, 'admin_id');
}
    // Add relationship to conversations
    public function conversations()
    {
        return $this->belongsToMany(Conversation::class, 'conversation_participants', 'admin_id', 'conversation_id')
            ->withPivot('role', 'last_read_at')
            ->withTimestamps();
    }

    public function sentMessages()
    {
        return $this->hasMany(Message::class, 'sender_admin_id');
    }

    // Get display name for admin
    public function getNameAttribute()
    {
        // If admin table has a 'name' column, return it
        if (!empty($this->attributes['name'])) {
            return $this->attributes['name'];
        }

        // Otherwise, return a formatted version of email
        return 'Admin: ' . $this->email;
    }

    // Get a shorter display name without "Admin:" prefix
    public function getDisplayNameAttribute()
    {
        if (!empty($this->attributes['name'])) {
            return $this->attributes['name'];
        }

        // Extract name from email (before @)
        $emailParts = explode('@', $this->email);
        $namePart = $emailParts[0];

        // Convert dots, underscores, and dashes to spaces and capitalize
        $name = str_replace(['.', '_', '-'], ' ', $namePart);
        return ucwords($name);
    }

    // Get initials for avatar
    public function getInitialsAttribute()
    {
        $name = $this->display_name;
        $words = explode(' ', $name);
        $initials = '';

        foreach ($words as $word) {
            if (!empty($word)) {
                $initials .= strtoupper($word[0]);
            }
        }

        return substr($initials, 0, 2) ?: 'A';
    }

    // Relationship with emoji assignments
    public function emojiAssignments()
    {
        return $this->morphMany(EmojiAssignment::class, 'assignable');
    }
}