<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\YoutubeVideo;
use App\Services\YouTubeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class YoutubeVideoController extends Controller
{
    public function index()
    {
        $videos = YoutubeVideo::with('addedBy')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('highSuperAdmin.youtubevideos.index', compact('videos'));
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'youtube_url' => 'required|url',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'Error' => true,
                'errors' => $validator->errors(),
                'Message' => 'Validation failed'
            ], 422);
        }

        $youtubeUrl = $request->youtube_url;
        
        // Extract video ID
        $videoId = YouTubeService::extractVideoId($youtubeUrl);
        
        if (!$videoId) {
            return response()->json([
                'Error' => true,
                'Message' => 'Invalid YouTube URL'
            ], 400);
        }

        // Check if video already exists
        $existingVideo = YoutubeVideo::where('video_id', $videoId)->first();
        if ($existingVideo) {
            return response()->json([
                'Error' => true,
                'Message' => 'This video has already been added'
            ], 400);
        }

        // Get video information
        $videoInfo = YouTubeService::getVideoInfo($videoId);
        
        if (!$videoInfo) {
            return response()->json([
                'Error' => true,
                'Message' => 'Unable to fetch video information'
            ], 400);
        }

        // Create video record
        $video = YoutubeVideo::create([
            'title' => $videoInfo['title'],
            'youtube_url' => $youtubeUrl,
            'video_id' => $videoId,
            'thumbnail_url' => $videoInfo['thumbnail_url'],
            'description' => $videoInfo['description'],
            'duration' => $videoInfo['duration'],
            'channel_name' => $videoInfo['channel_name'],
            'added_by' => Auth::id(),
            'is_active' => $request->boolean('is_active', true)
        ]);

        return response()->json([
            'Error' => false,
            'Message' => 'Video added successfully!',
            'video' => $video
        ]);
    }

    public function update(Request $request, YoutubeVideo $youtubeVideo)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'Error' => true,
                'errors' => $validator->errors(),
                'Message' => 'Validation failed'
            ], 422);
        }

        $youtubeVideo->update([
            'title' => $request->title,
            'description' => $request->description,
            'is_active' => $request->boolean('is_active')
        ]);

        return response()->json([
            'Error' => false,
            'Message' => 'Video updated successfully!',
            'video' => $youtubeVideo
        ]);
    }

    public function destroy(YoutubeVideo $youtubeVideo)
    {
        $youtubeVideo->delete();

        return response()->json([
            'Error' => false,
            'Message' => 'Video deleted successfully!'
        ]);
    }
}