<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;

class VinDecoderController extends Controller
{
    public function index()
    {
        return view('Employee.vin-decoder');
    }

    public function decode(Request $request)
    {
        // Validate the VIN number
        $validator = Validator::make($request->all(), [
            'vin_number' => 'required|string|min:17|max:17|regex:/^[A-HJ-NPR-Z0-9]{17}$/',
        ], [
            'vin_number.required' => 'VIN number is required.',
            'vin_number.min' => 'VIN number must be exactly 17 characters.',
            'vin_number.max' => 'VIN number must be exactly 17 characters.',
            'vin_number.regex' => 'Invalid VIN format. VIN should contain only letters and numbers (excluding I, O, Q).',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $vinNumber = strtoupper(trim($request->vin_number));

        try {
            // Call the NHTSA VIN Decode API
            $response = Http::timeout(30)->get("https://vpic.nhtsa.dot.gov/api/vehicles/decodevin/{$vinNumber}?format=json");

            if ($response->successful()) {
                $data = $response->json();
                
                // Extract Make, Model, and Model Year for recalls API
                $make = null;
                $model = null;
                $modelYear = null;
                
                if (isset($data['Results']) && is_array($data['Results'])) {
                    foreach ($data['Results'] as $result) {
                        switch ($result['Variable']) {
                            case 'Make':
                                $make = $result['Value'];
                                break;
                            case 'Model':
                                $model = $result['Value'];
                                break;
                            case 'Model Year':
                                $modelYear = $result['Value'];
                                break;
                        }
                    }
                }
                
                $recallData = null;
                
                // Call the NHTSA Recalls API if we have the required data
                if ($make && $model && $modelYear) {
                    try {
                        $recallResponse = Http::timeout(30)->get("https://api.nhtsa.gov/recalls/recallsByVehicle", [
                            'make' => $make,
                            'model' => $model,
                            'modelYear' => $modelYear
                        ]);
                        
                        if ($recallResponse->successful()) {
                            $recallData = $recallResponse->json();
                            // dd($recallData);
                        }
                    } catch (\Exception $e) {
                        // If recall API fails, continue with VIN data only
                        // Log the error but don't break the main functionality
                        \Log::warning('Recall API failed: ' . $e->getMessage());
                    }
                }
                
                return view('Employee.vin-decoder', [
                    'vinData' => $data,
                    'recallData' => $recallData,
                    'vinNumber' => $vinNumber,
                    'make' => $make,
                    'model' => $model,
                    'modelYear' => $modelYear,
                    'success' => true
                ]);
            } else {
                return redirect()->back()
                    ->with('error', 'Failed to decode VIN. Please try again.')
                    ->withInput();
            }

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Error connecting to VIN decoder service. Please try again later.')
                ->withInput();
        }
    }
}