<?php

namespace App\Http\Controllers;

use App\Services\UniversalNotificationService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class UniversalNotificationController extends Controller
{
    private UniversalNotificationService $notificationService;

    public function __construct(UniversalNotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Get current user info (reusable method)
     */
    private function getCurrentUser(Request $request): array
    {
        $user = $request->attributes->get('authenticated_user');
        $tenantId = $request->attributes->get('tenant_id');
        
        if (get_class($user) === 'App\Models\Admin' && $user['role'] === 'Admin') {
            return [
                'tenant_id' => $tenantId,
                'type' => 'admin',
                'id' => $user->id,
                'name' => $user->name ?? 'Admin User'
            ];
        }
        
        if (isset($user->employee_id)) {
            return [
                'tenant_id' => $tenantId,
                'type' => 'employee',
                'id' => $user->employee_id,
                'name' => $user->name ?? 'Employee'
            ];
        }
        
        abort(403, 'Unable to determine user identity.');
    }

    // ================================
    // BASIC NOTIFICATION METHODS
    // ================================

    /**
     * Send notification to specific user
     */
    public function sendToUser(Request $request): JsonResponse
    {
        $request->validate([
            'receiver_type' => 'required|in:admin,employee',
            'receiver_id' => 'required|integer',
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:1000',
            'type' => 'nullable|string|max:50',
            'priority' => 'nullable|in:low,normal,high,urgent',
            'action_url' => 'nullable|url',
            'action_text' => 'nullable|string|max:50',
        ]);

        $user = $this->getCurrentUser($request);

        $sent = $this->notificationService->sendToUser([
            'tenant_id' => $user['tenant_id'],
            'receiver_type' => $request->receiver_type,
            'receiver_id' => $request->receiver_id,
            'sender_type' => $user['type'],
            'sender_id' => $user['id'],
            'sender_name' => $user['name'],
            'title' => $request->title,
            'content' => $request->content,
            'type' => $request->type ?? 'general',
            'priority' => $request->priority ?? 'normal',
            'action_url' => $request->action_url,
            'action_text' => $request->action_text,
        ]);

        return response()->json([
            'success' => $sent,
            'message' => $sent ? 'Notification sent successfully' : 'Failed to send notification'
        ]);
    }

    /**
     * Send announcement to all users
     */
    public function sendAnnouncement(Request $request): JsonResponse
    {
        $request->validate([
            'receiver_type' => 'required|in:admin,employee,all',
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:1000',
            'priority' => 'nullable|in:low,normal,high,urgent',
            'category' => 'nullable|string',
            'department_id' => 'nullable|integer',
            'expires_at' => 'nullable|date|after:now',
        ]);

        $user = $this->getCurrentUser($request);
        $sentCount = 0;

        if ($request->receiver_type === 'all') {
            // Send to both admins and employees
            $sentCount += $this->notificationService->sendAnnouncement([
                'tenant_id' => $user['tenant_id'],
                'receiver_type' => 'admin',
                'sender_id' => $user['id'],
                'sender_name' => $user['name'],
                'title' => $request->title,
                'content' => $request->content,
                'priority' => $request->priority ?? 'high',
                'expires_at' => $request->expires_at,
            ]);

            $sentCount += $this->notificationService->sendAnnouncement([
                'tenant_id' => $user['tenant_id'],
                'receiver_type' => 'employee',
                'sender_id' => $user['id'],
                'sender_name' => $user['name'],
                'title' => $request->title,
                'content' => $request->content,
                'priority' => $request->priority ?? 'high',
                'expires_at' => $request->expires_at,
                'filters' => [
                    'category' => $request->category,
                    'department_id' => $request->department_id,
                ]
            ]);
        } else {
            $sentCount = $this->notificationService->sendAnnouncement([
                'tenant_id' => $user['tenant_id'],
                'receiver_type' => $request->receiver_type,
                'sender_id' => $user['id'],
                'sender_name' => $user['name'],
                'title' => $request->title,
                'content' => $request->content,
                'priority' => $request->priority ?? 'high',
                'expires_at' => $request->expires_at,
                'filters' => [
                    'category' => $request->category,
                    'department_id' => $request->department_id,
                ]
            ]);
        }

        return response()->json([
            'success' => $sentCount > 0,
            'message' => "Announcement sent to {$sentCount} users",
            'sent_count' => $sentCount
        ]);
    }

    // ================================
    // EMOJI REACTIONS
    // ================================

    /**
     * Send emoji reaction
     */
    public function sendEmoji(Request $request): JsonResponse
    {
        $request->validate([
            'receiver_type' => 'required|in:admin,employee',
            'receiver_id' => 'required|integer',
            'emoji' => 'required|string|max:10',
            'message' => 'nullable|string|max:255',
            'context' => 'nullable|string|max:100', // e.g., 'chat', 'post', 'achievement'
        ]);

        $user = $this->getCurrentUser($request);

        $sent = $this->notificationService->sendEmoji([
            'tenant_id' => $user['tenant_id'],
            'receiver_type' => $request->receiver_type,
            'receiver_id' => $request->receiver_id,
            'sender_type' => $user['type'],
            'sender_id' => $user['id'],
            'sender_name' => $user['name'],
            'emoji' => $request->emoji,
            'message' => $request->message,
            'context' => $request->context ?? 'general',
        ]);

        return response()->json([
            'success' => $sent,
            'message' => $sent ? 'Emoji sent successfully' : 'Failed to send emoji'
        ]);
    }

    // ================================
    // CHAT NOTIFICATIONS
    // ================================

    /**
     * Send chat notification
     */
    public function sendChatNotification(Request $request): JsonResponse
    {
        $request->validate([
            'receiver_type' => 'required|in:admin,employee',
            'receiver_id' => 'required|integer',
            'message' => 'required|string|max:1000',
            'chat_id' => 'nullable|integer',
            'message_id' => 'nullable|integer',
            'is_group' => 'nullable|boolean',
            'chat_url' => 'nullable|url',
        ]);

        $user = $this->getCurrentUser($request);

        $sent = $this->notificationService->sendChatNotification([
            'tenant_id' => $user['tenant_id'],
            'receiver_type' => $request->receiver_type,
            'receiver_id' => $request->receiver_id,
            'sender_type' => $user['type'],
            'sender_id' => $user['id'],
            'sender_name' => $user['name'],
            'message' => $request->message,
            'chat_id' => $request->chat_id,
            'message_id' => $request->message_id,
            'is_group' => $request->is_group ?? false,
            'chat_url' => $request->chat_url,
        ]);

        return response()->json([
            'success' => $sent,
            'message' => $sent ? 'Chat notification sent' : 'Failed to send chat notification'
        ]);
    }

    // ================================
    // POINTS NOTIFICATIONS
    // ================================

    /**
     * Send points notification
     */
    public function sendPointsNotification(Request $request): JsonResponse
    {
        $request->validate([
            'receiver_type' => 'required|in:admin,employee',
            'receiver_id' => 'required|integer',
            'points_action' => 'required|in:awarded,approved,rejected,pending,cancelled,target_completed,target_set',
            'content' => 'required|string|max:1000',
            'points_amount' => 'nullable|integer',
            'target_name' => 'nullable|string',
            'vin_no' => 'nullable|string',
            'priority' => 'nullable|in:low,normal,high,urgent',
        ]);

        $user = $this->getCurrentUser($request);

        $sent = $this->notificationService->sendPointsNotification([
            'tenant_id' => $user['tenant_id'],
            'receiver_type' => $request->receiver_type,
            'receiver_id' => $request->receiver_id,
            'sender_type' => $user['type'],
            'sender_id' => $user['id'],
            'sender_name' => $user['name'],
            'points_action' => $request->points_action,
            'content' => $request->content,
            'points_amount' => $request->points_amount,
            'target_name' => $request->target_name,
            'vin_no' => $request->vin_no,
            'priority' => $request->priority ?? 'normal',
        ]);

        return response()->json([
            'success' => $sent,
            'message' => $sent ? 'Points notification sent' : 'Failed to send points notification'
        ]);
    }

    // ================================
    // REMINDER NOTIFICATIONS
    // ================================

    /**
     * Send reminder notification
     */
    public function sendReminder(Request $request): JsonResponse
    {
        $request->validate([
            'receiver_type' => 'required|in:admin,employee',
            'receiver_id' => 'required|integer',
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:1000',
            'priority' => 'nullable|in:low,normal,high,urgent',
            'action_url' => 'nullable|url',
            'action_text' => 'nullable|string|max:50',
            'reminder_data' => 'nullable|array',
        ]);

        $user = $this->getCurrentUser($request);

        $sent = $this->notificationService->sendReminder([
            'tenant_id' => $user['tenant_id'],
            'receiver_type' => $request->receiver_type,
            'receiver_id' => $request->receiver_id,
            'title' => $request->title,
            'content' => $request->content,
            'priority' => $request->priority ?? 'normal',
            'action_url' => $request->action_url,
            'action_text' => $request->action_text,
            'reminder_data' => $request->reminder_data ?? [],
        ]);

        return response()->json([
            'success' => $sent,
            'message' => $sent ? 'Reminder sent successfully' : 'Failed to send reminder'
        ]);
    }

    // ================================
    // NOTIFICATION MANAGEMENT
    // ================================

    /**
     * Get user notifications
     */
    public function getNotifications(Request $request): JsonResponse
    {
        $request->validate([
            'limit' => 'nullable|integer|min:1|max:100',
            'type' => 'nullable|string',
            'unread_only' => 'nullable|boolean',
            'priority' => 'nullable|in:low,normal,high,urgent',
            'from_date' => 'nullable|date',
        ]);

        $user = $this->getCurrentUser($request);

        $filters = array_filter([
            'type' => $request->type,
            'unread_only' => $request->boolean('unread_only', false),
            'priority' => $request->priority,
            'from_date' => $request->from_date,
        ]);

        $notifications = $this->notificationService->getUserNotifications(
            $user['tenant_id'],
            $user['type'],
            $user['id'],
            $request->get('limit', 20),
            $filters
        );

        $unreadCount = $this->notificationService->getUnreadCount(
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => true,
            'notifications' => $notifications,
            'unread_count' => $unreadCount,
        ]);
    }

    /**
     * Mark notification as read
     */
    public function markAsRead(Request $request, $id): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        $success = $this->notificationService->markAsRead(
            $id,
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => $success,
            'message' => $success ? 'Notification marked as read' : 'Notification not found',
        ], $success ? 200 : 404);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        $count = $this->notificationService->markAllAsRead(
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => true,
            'message' => "Marked {$count} notifications as read",
            'count' => $count,
        ]);
    }

    /**
     * Delete notification
     */
    public function deleteNotification(Request $request, $id): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        $success = $this->notificationService->deleteNotification(
            $id,
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => $success,
            'message' => $success ? 'Notification deleted' : 'Notification not found',
        ], $success ? 200 : 404);
    }

    /**
     * Get unread count
     */
    public function getUnreadCount(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        $count = $this->notificationService->getUnreadCount(
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => true,
            'count' => $count,
        ]);
    }

    /**
     * Get notification statistics (for admins)
     */
    public function getStats(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        // Only admins can view stats
        if ($user['type'] !== 'admin') {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $request->validate([
            'user_type' => 'nullable|in:admin,employee',
            'from_date' => 'nullable|date',
        ]);

        $filters = array_filter([
            'user_type' => $request->user_type,
            'from_date' => $request->from_date,
        ]);

        $stats = $this->notificationService->getNotificationStats(
            $user['tenant_id'],
            $filters
        );

        return response()->json([
            'success' => true,
            'stats' => $stats,
        ]);
    }

    /**
     * Cleanup old notifications (for admins)
     */
    public function cleanup(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        // Only admins can cleanup
        if ($user['type'] !== 'admin') {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $request->validate([
            'days_old' => 'nullable|integer|min:1|max:365',
        ]);

        $deletedCount = $this->notificationService->cleanupOldNotifications(
            $request->get('days_old', 30)
        );

        return response()->json([
            'success' => true,
            'message' => "Cleaned up {$deletedCount} old notifications",
            'deleted_count' => $deletedCount,
        ]);
    }

    // ================================
    // BULK OPERATIONS
    // ================================

    /**
     * Send bulk notifications to multiple users
     */
    public function sendBulkNotifications(Request $request): JsonResponse
    {
        $request->validate([
            'receivers' => 'required|array|min:1|max:100',
            'receivers.*.type' => 'required|in:admin,employee',
            'receivers.*.id' => 'required|integer',
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:1000',
            'type' => 'nullable|string|max:50',
            'priority' => 'nullable|in:low,normal,high,urgent',
        ]);

        $user = $this->getCurrentUser($request);

        $results = $this->notificationService->sendToMultipleUsers([
            'tenant_id' => $user['tenant_id'],
            'receivers' => $request->receivers,
            'sender_type' => $user['type'],
            'sender_id' => $user['id'],
            'sender_name' => $user['name'],
            'title' => $request->title,
            'content' => $request->content,
            'type' => $request->type ?? 'general',
            'priority' => $request->priority ?? 'normal',
        ]);

        return response()->json([
            'success' => $results['successful'] > 0,
            'message' => "Sent to {$results['successful']} users, failed for {$results['failed']} users",
            'results' => $results,
        ]);
    }

    /**
     * Send notification by category/department
     */
    public function sendByCategory(Request $request): JsonResponse
    {
        $request->validate([
            'receiver_type' => 'required|in:admin,employee',
            'category' => 'nullable|string',
            'department_id' => 'nullable|integer',
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:1000',
            'type' => 'nullable|string|max:50',
            'priority' => 'nullable|in:low,normal,high,urgent',
        ]);

        $user = $this->getCurrentUser($request);

        $sentCount = $this->notificationService->sendByCategory([
            'tenant_id' => $user['tenant_id'],
            'receiver_type' => $request->receiver_type,
            'category' => $request->category,
            'department_id' => $request->department_id,
            'sender_type' => $user['type'],
            'sender_id' => $user['id'],
            'sender_name' => $user['name'],
            'title' => $request->title,
            'content' => $request->content,
            'type' => $request->type ?? 'general',
            'priority' => $request->priority ?? 'normal',
        ]);

        return response()->json([
            'success' => $sentCount > 0,
            'message' => "Notification sent to {$sentCount} users",
            'sent_count' => $sentCount,
        ]);
    }
}