<?php

namespace App\Http\Controllers;

use App\Models\Tenant;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Stripe\Stripe;
use Stripe\Checkout\Session;

class TenantSignupController extends Controller
{
    public function showSignupForm()
    {
        return view('tenant.signup');
    }

  public function register(Request $request)
{
    $request->validate([
        'company_name' => 'required|string|max:255',
        'subdomain' => 'required|string|max:50|unique:tenants,domain|regex:/^[a-z0-9]+$/',
        'admin_name' => 'required|string|max:255',
        'admin_email' => 'required|email|max:255',
        'admin_password' => 'required|string|min:8|confirmed',
        'plan' => 'required|string|in:basic,premium,enterprise',
        'phone' => 'nullable|string|max:20',
        'address' => 'nullable|string|max:255',
        'promo_code' => 'nullable|string|max:50',
    ]);
    
    $validPromoCodes = ['CARISMA2025', 'FREEMONTH', 'LAUNCH'];
    $hasValidPromo = $request->promo_code && in_array(strtoupper($request->promo_code), $validPromoCodes);
    
    // dd($request->plan);
    // Create tenant (initially inactive)
    $tenant = Tenant::create([
        'name' => $request->company_name,
        'domain' => $request->subdomain,
        'plan' => $request->plan,
        'active' => false,
        'phone' => $request->phone,
        'address' => $request->address,
        'promo_code' => $hasValidPromo ? strtoupper($request->promo_code) : null,
    ]);
    // Create tenant admin
    $admin = Admin::create([
        'name' => $request->admin_name,
        'email' => $request->admin_email,
        'password' => bcrypt($request->admin_password),
        'tenant_id' => $tenant->id,
    ]);

    $tenant->update(['admin_id' => $admin->id]);

    // Store tenant ID in session
    session(['temp_tenant_id' => $tenant->id]);

    // Create Stripe Checkout Session
    Stripe::setApiKey(config('services.stripe.secret'));

    // Set Test price based on selected plan
    $prices = [
            'basic' => 'price_1RJGrHAIEkqtoj6HqM7yv75C',
            'premium' => 'price_1RJGs5AIEkqtoj6H02hTEp4e',
            'enterprise' => 'price_1RJGsEAIEkqtoj6HAQRlqv04',
        ];
        
        // Set Live price based on selected plan
    // $prices = [
    //     'basic' => 'price_1RivGiP7VooaeP3TtUChcPai',
    //     'premium' => 'price_1RivEbP7VooaeP3T1Fv0Tb0X',
    //     'enterprise' => 'price_1RivCjP7VooaeP3T6PEGk9XD',
    // ];

    // dd($prices[$request->plan]);
    // Build session data
    $sessionData = [
        'payment_method_types' => ['card'],
        'line_items' => [[
            'price' => $prices[$request->plan],
            'quantity' => 1,
        ]],
        'mode' => 'subscription',
        'success_url' => route('tenant.payment.success', ['tenant_id' => $tenant->id]) . '?session_id={CHECKOUT_SESSION_ID}',
        'cancel_url' => route('tenant.payment.cancel', ['tenant_id' => $tenant->id]),
        'customer_email' => $request->admin_email,
        'client_reference_id' => $tenant->id,
        'metadata' => [
            'tenant_id' => $tenant->id,
            'plan' => $request->plan,
            'promo_code' => $hasValidPromo ? $request->promo_code : null,
        ],
    ];

    // Add free trial if valid promo code and not basic plan
    if ($hasValidPromo && $request->plan !== 'basic') {
        $sessionData['subscription_data'] = [
            'trial_period_days' => 30, // 30 days free trial
        ];
    }

    $session = Session::create($sessionData);

    return redirect($session->url);
}

    public function paymentSuccess(Request $request, $tenant_id)
    {
        // Verify session ID with Stripe
        Stripe::setApiKey(config('services.stripe.secret'));
        $session = Session::retrieve($request->session_id);

        // Ensure this is a valid checkout session for this tenant
        if ($session->client_reference_id != $tenant_id) {
            abort(403, 'Invalid payment session');
        }

        // Activate the tenant
        $tenant = Tenant::findOrFail($tenant_id);
        $tenant->update([
            'active' => true,
            'stripe_customer_id' => $session->customer,
            'stripe_subscription_id' => $session->subscription,
        ]);

        // Initialize tenant with default data
        $this->initializeTenant($tenant);

        // Show different success messages based on promo code
        $message = $tenant->promo_code
            ? 'Your dealership has been registered successfully with a FREE first month! Please log in.'
            : 'Your dealership has been registered successfully! Please log in.';

        // Redirect to the new tenant subdomain
        return redirect("http://{$tenant->domain}.platform.carismacar.org/super-admin")
            ->with('success', $message);
    }

    public function paymentCancel(Request $request, $tenant_id)
    {
        // Tenant decided not to complete payment
        // You may want to delete the tenant or keep it for follow-up

        return redirect()->route('tenant.signup')
            ->with('warning', 'Your registration was not completed because the payment was cancelled.');
    }

    private function initializeTenant($tenant)
    {
        // Create default departments
        $defaultDepartments = [
            ['name' => 'Service', 'role' => 'Service', 'categories' => ['Service Advisor', 'Express Tech', 'Repair Tech', 'BDC']],
            ['name' => 'Sales', 'role' => 'Sales Member', 'categories' => ['Sales Agent']],
            ['name' => 'Parts', 'role' => 'Parts', 'categories' => ['Back Counter']],
            ['name' => 'Finance', 'role' => 'Finance Agent', 'categories' => ['Finance']]
        ];

        foreach ($defaultDepartments as $dept) {
            \App\Models\Department::create([
                'name' => $dept['name'],
                'no_of_emp' => 0,
                'role' => $dept['role'],
                'categories' => json_encode($dept['categories']),
                'slug' => Str::slug($dept['name']),
                'tenant_id' => $tenant->id,
            ]);
        }

        // Create default products
        $defaultProducts = [
            ['name' => 'Fuel Injection Clean', 'number' => '0001', 'status' => '2'],
            ['name' => 'Brake Fluid Exchange', 'number' => '0002', 'status' => '2'],
            ['name' => 'Trans Fluid Exchange', 'number' => '0003', 'status' => '2']
        ];

        foreach ($defaultProducts as $index => $product) {
            \App\Models\Product::create([
                'name' => $product['name'],
                'code' => str_pad($index + 1, 4, '0', STR_PAD_LEFT),
                'number' => $product['number'],
                'status' => $product['status'],
                'tenant_id' => $tenant->id,
            ]);
        }
    }
}