<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\SuperAdmin;
use App\Models\Tenant;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;

class SuperAdminController extends Controller
{
    /**
     * Show the superadmin login form
     */
    public function showLoginForm()
    {
        return view('highSuperAdmin.login');
    }

    /**
     * Handle superadmin login
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|min:6',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $superadmin = SuperAdmin::where('email', $request->email)->first();

        if ($superadmin && Hash::check($request->password, $superadmin->password)) {
            if ($superadmin->status !== 'active') {
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Your account is inactive. Please contact support.'
                    ], 403);
                }
                return redirect()->back()
                    ->with('error', 'Your account is inactive. Please contact support.');
            }

            // Manual authentication since we're not using config guards
            Session::put('superadmin_id', $superadmin->id);
            Session::put('superadmin_name', $superadmin->name);
            Session::put('superadmin_email', $superadmin->email);
            Session::put('is_superadmin', true);

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Welcome back, ' . $superadmin->name,
                    'redirect' => route('superadmin.dashboard')
                ]);
            }

            return redirect()->route('superadmin.dashboard')
                ->with('success', 'Welcome back, ' . $superadmin->name);
        }

        if ($request->ajax()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid credentials.'
            ], 401);
        }

        return redirect()->back()
            ->with('error', 'Invalid credentials.')
            ->withInput($request->only('email'));
    }

    /**
     * Handle superadmin logout
     */
    public function logout()
    {
        Session::forget(['superadmin_id', 'superadmin_name', 'superadmin_email', 'is_superadmin']);
        Session::flush();

        return redirect()->route('superadmin.login')
            ->with('success', 'You have been logged out successfully.');
    }

    /**
     * Get authenticated superadmin
     */
    private function getAuthenticatedSuperAdmin()
    {
        $superadminId = Session::get('superadmin_id');

        if (!$superadminId) {
            return null;
        }

        return SuperAdmin::find($superadminId);
    }

    /**
     * Check if superadmin is authenticated
     */
    private function checkAuth()
    {
        if (!Session::get('is_superadmin') || !$this->getAuthenticatedSuperAdmin()) {
            abort(401, 'Unauthorized');
        }
    }

    /**
     * Show superadmin dashboard
     */
    public function dashboard()
    {
        $this->checkAuth();

        $superadmin = $this->getAuthenticatedSuperAdmin();

        $data = [
            'totalTenants' => Tenant::count(),
            'activeTenants' => Tenant::where('active', true)->count(),
            'totalAdmins' => Admin::count(),
            'recentTenants' => Tenant::latest()->take(5)->get(),
        ];

        return view('highSuperAdmin.dashboard', compact('data', 'superadmin'));
    }

    /**
     * Show profile page
     */
    public function profile()
    {
        $this->checkAuth();
        $superadmin = $this->getAuthenticatedSuperAdmin();

        return view('highSuperAdmin.profile', compact('superadmin'));
    }

    // =================
    // TENANT MANAGEMENT
    // =================

    /**
     * Get all tenants
     */
    public function getTenants()
    {
        $this->checkAuth();

        $tenants = Tenant::with([
            'admins' => function ($query) {
                $query->select('id', 'email', 'tenant_id', 'created_at');
            }
        ])
            ->withCount(['admins', 'employees', 'departments'])
            ->paginate(10);

        if (request()->wantsJson()) {
            return response()->json($tenants);
        }

        return view('highSuperAdmin.tenants', compact('tenants'));
    }

    /**
     * Get specific tenant details
     */
    public function getTenantDetails($id)
    {
        $this->checkAuth();

        $tenant = Tenant::with([
            'admins' => function ($query) {
                $query->select('id', 'email', 'tenant_id', 'created_at');
            },
            'departments' => function ($query) {
                $query->select('id', 'name', 'tenant_id');
            },
            'employees' => function ($query) {
                $query->select('employee_id', 'name', 'employee_id', 'tenant_id');
            },
            'products' => function ($query) {
                $query->select('id', 'name', 'tenant_id');
            }
        ])
            ->withCount(['admins', 'employees', 'departments', 'products', 'targets'])
            ->findOrFail($id);

        // Always return JSON when called via AJAX
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json($tenant);
        }

        return view('highSuperAdmin.tenant-details', compact('tenant'));
    }

    /**
     * Create a new tenant
     */
    public function createTenant(Request $request)
    {
        $this->checkAuth();

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'domain' => 'required|string|max:255|unique:tenants,domain',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'plan' => 'nullable|in:basic,standard,premium',
            'active' => 'nullable|boolean|in:0,1',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $tenant = Tenant::create([
            'name' => $request->name,
            'domain' => $request->domain,
            'phone' => $request->phone,
            'address' => $request->address,
            'plan' => $request->plan ?? 'basic',
            'active' => $request->has('active') ? (bool) $request->active : true,
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Tenant created successfully',
                'tenant' => $tenant
            ]);
        }

        return redirect()->route('superadmin.tenants')
            ->with('success', 'Tenant created successfully');
    }

    /**
     * Update tenant details (including active status)
     */
    public function updateTenant(Request $request, $id)
    {
        $this->checkAuth();

        $tenant = Tenant::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'domain' => 'required|string|max:255|unique:tenants,domain,' . $id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'plan' => 'nullable|in:basic,standard,premium',
            'active' => 'nullable|boolean|in:0,1',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $tenant->update([
            'name' => $request->name,
            'domain' => $request->domain,
            'phone' => $request->phone,
            'address' => $request->address,
            'plan' => $request->plan ?? $tenant->plan,
            'active' => $request->has('active') ? (bool) $request->active : $tenant->active,
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Tenant updated successfully',
                'tenant' => $tenant->fresh()
            ]);
        }

        return redirect()->route('superadmin.tenants')
            ->with('success', 'Tenant updated successfully');
    }

    /**
     * Toggle tenant status
     */
    public function toggleTenantStatus(Request $request, $id)
    {
        $this->checkAuth();

        $tenant = Tenant::findOrFail($id);

        // Toggle the active status (0 to 1 or 1 to 0)
        $tenant->active = !$tenant->active;
        $tenant->save();

        $status = $tenant->active ? 'activated' : 'deactivated';

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => "Tenant has been {$status} successfully",
                'active' => $tenant->active,
                'status_text' => $tenant->active ? 'Active' : 'Inactive'
            ]);
        }

        return redirect()->back()
            ->with('success', "Tenant has been {$status} successfully");
    }

    /**
     * Delete tenant (CASCADE DELETE all related data)
     */
    public function deleteTenant(Request $request, $id)
    {
        $this->checkAuth();

        $tenant = Tenant::findOrFail($id);
        $tenantName = $tenant->name;

        try {
            // This will cascade delete all related data if foreign keys are set up properly
            // Otherwise, manually delete related records first

            // Delete related admins first
            $tenant->admins()->delete();

            // Delete other related records if they exist
            if (method_exists($tenant, 'employees')) {
                $tenant->employees()->delete();
            }
            if (method_exists($tenant, 'departments')) {
                $tenant->departments()->delete();
            }
            if (method_exists($tenant, 'products')) {
                $tenant->products()->delete();
            }
            if (method_exists($tenant, 'targets')) {
                $tenant->targets()->delete();
            }

            // Finally delete the tenant
            $tenant->delete();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => "Tenant '{$tenantName}' and all related data deleted successfully"
                ]);
            }

            return redirect()->route('superadmin.tenants')
                ->with('success', "Tenant '{$tenantName}' and all related data deleted successfully");

        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error deleting tenant: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'Error deleting tenant: ' . $e->getMessage());
        }
    }

    // ================
    // ADMIN MANAGEMENT
    // ================

    /**
     * Get all admins across all tenants
     */
    public function getAdmins()
    {
        $this->checkAuth();

        $admins = Admin::where('subadmin', 0)
            ->with([
                'tenant' => function ($query) {
                    $query->select('id', 'name', 'domain', 'active');
                }
            ])
            ->paginate(10);

        // Get all tenants for the add admin dropdown
        $tenants = Tenant::where('active', true)
            ->select('id', 'name', 'domain')
            ->orderBy('name')
            ->get();

        if (request()->wantsJson()) {
            return response()->json([
                'admins' => $admins,
                'tenants' => $tenants
            ]);
        }

        return view('highSuperAdmin.admins', compact('admins', 'tenants'));
    }

    /**
     * Get specific admin details
     */
    public function getAdminDetails($id)
    {
        $this->checkAuth();

        $admin = Admin::with([
            'tenant' => function ($query) {
                $query->select('id', 'name', 'domain', 'active', 'plan');
            }
        ])->findOrFail($id);

        // Calculate additional data based on the tenant's relationships
        $additionalData = [
            'login_count' => $admin->login_count ?? 0,
            'total_employees' => 0,
            'total_departments' => 0,
            'points_approved' => 0,
            'last_activity' => $admin->last_activity ?? 'No recent activity',
        ];

        // If admin has a tenant, get the tenant's statistics
        if ($admin->tenant) {
            $additionalData['total_employees'] = method_exists($admin->tenant, 'employees') ? $admin->tenant->employees()->count() : 0;
            $additionalData['total_departments'] = method_exists($admin->tenant, 'departments') ? $admin->tenant->departments()->count() : 0;
        }

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'admin' => $admin,
                'additional_data' => $additionalData
            ]);
        }

        return view('highSuperAdmin.admin-details', compact('admin', 'additionalData'));
    }

    /**
     * Create a new admin for a tenant
     */
    public function createAdmin(Request $request)
    {
        $this->checkAuth();

        $validator = Validator::make($request->all(), [
            'tenant_id' => 'required|exists:tenants,id',
            'email' => 'required|email|unique:admins,email',
            'password' => 'required|min:6',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $admin = Admin::create([
            'tenant_id' => $request->tenant_id,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'subadmin' => 0, // Main admin, not sub-admin
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Admin created successfully',
                'admin' => $admin->load('tenant')
            ]);
        }

        return redirect()->route('superadmin.admins')
            ->with('success', 'Admin created successfully');
    }

    /**
     * Update admin details
     */
    public function updateAdmin(Request $request, $id)
    {
        $this->checkAuth();

        $admin = Admin::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'email' => 'required|email|unique:admins,email,' . $id,
            'tenant_id' => 'required|exists:tenants,id',
            'password' => 'nullable|min:6',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $updateData = [
            'email' => $request->email,
            'tenant_id' => $request->tenant_id,
        ];

        // Only update password if provided
        if ($request->filled('password')) {
            $updateData['password'] = Hash::make($request->password);
        }

        $admin->update($updateData);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Admin updated successfully',
                'admin' => $admin->fresh()->load('tenant')
            ]);
        }

        return redirect()->route('superadmin.admins')
            ->with('success', 'Admin updated successfully');
    }

    /**
     * Reset admin password
     */
    public function resetAdminPassword(Request $request, $id)
    {
        $this->checkAuth();

        $admin = Admin::findOrFail($id);

        // Generate a random password
        $newPassword = Str::random(12);
        $admin->password = Hash::make($newPassword);
        $admin->save();

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Password reset successfully',
                'new_password' => $newPassword
            ]);
        }

        return redirect()->back()
            ->with('success', 'Password reset successfully. New password: ' . $newPassword);
    }

    /**
     * Delete admin
     */
    public function deleteAdmin(Request $request, $id)
    {
        $this->checkAuth();

        $admin = Admin::findOrFail($id);
        $adminEmail = $admin->email;

        try {
            $admin->delete();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => "Admin '{$adminEmail}' has been deleted successfully"
                ]);
            }

            return redirect()->route('superadmin.admins')
                ->with('success', "Admin '{$adminEmail}' has been deleted successfully");

        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error deleting admin: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'Error deleting admin: ' . $e->getMessage());
        }
    }

    // ====================
    // PROFILE MANAGEMENT
    // ====================

    /**
     * Update superadmin profile
     */
    public function updateProfile(Request $request)
    {
        $this->checkAuth();

        $superadmin = $this->getAuthenticatedSuperAdmin();

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:super_admins,email,' . $superadmin->id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'password' => 'nullable|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $superadmin->name = $request->name;
        $superadmin->email = $request->email;
        $superadmin->phone = $request->phone;
        $superadmin->address = $request->address;

        if ($request->filled('password')) {
            $superadmin->password = Hash::make($request->password);
        }

        $superadmin->save();

        // Update session data
        Session::put('superadmin_name', $superadmin->name);
        Session::put('superadmin_email', $superadmin->email);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Profile updated successfully'
            ]);
        }

        return redirect()->back()
            ->with('success', 'Profile updated successfully');
    }

    /**
     * Delete superadmin account (with confirmation)
     */
    public function deleteSuperAdmin(Request $request)
    {
        $this->checkAuth();

        $validator = Validator::make($request->all(), [
            'confirm_email' => 'required|email',
            'confirm_password' => 'required',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors(),
                    'message' => 'Validation failed'
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $superadmin = $this->getAuthenticatedSuperAdmin();

        // Verify email and password
        if ($request->confirm_email !== $superadmin->email || !Hash::check($request->confirm_password, $superadmin->password)) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Email or password confirmation failed'
                ], 422);
            }
            return redirect()->back()
                ->with('error', 'Email or password confirmation failed');
        }

        try {
            // Clear session first
            Session::forget(['superadmin_id', 'superadmin_name', 'superadmin_email', 'is_superadmin']);
            Session::flush();

            // Delete the superadmin account
            $superadmin->delete();

            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Account deleted successfully',
                    'redirect' => route('superadmin.login')
                ]);
            }

            return redirect()->route('superadmin.login')
                ->with('success', 'Your account has been deleted successfully');

        } catch (\Exception $e) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error deleting account: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'Error deleting account: ' . $e->getMessage());
        }
    }
}