<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Tenant;
use Illuminate\Http\Request;

class TenantController extends Controller
{
    public function index()
    {
        $tenants = Tenant::withCount(['admins', 'departments', 'employees'])->get();
        return view('super-admin.tenants.index', compact('tenants'));
    }

    public function show(Tenant $tenant)
    {
        // Load tenant stats
        $stats = [
            'employees' => $tenant->employees()->count(),
            'departments' => $tenant->departments()->count(),
            'points' => $tenant->points()->count(),
            'targets' => $tenant->targets()->count(),
        ];

        return view('super-admin.tenants.show', compact('tenant', 'stats'));
    }

    public function create()
    {
        return view('super-admin.tenants.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'domain' => 'required|string|max:50|regex:/^[a-z0-9]+$/|unique:tenants,domain',
            'active' => 'required|boolean',
            'plan' => 'required|string|in:basic,premium,enterprise',
        ]);

        Tenant::create($request->all());

        return redirect()->route('super.tenants.index')
            ->with('success', 'Tenant created successfully.');
    }

    public function edit(Tenant $tenant)
    {
        return view('super-admin.tenants.edit', compact('tenant'));
    }

    public function update(Request $request, Tenant $tenant)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'domain' => "required|string|max:50|regex:/^[a-z0-9]+$/|unique:tenants,domain,{$tenant->id}",
            'active' => 'required|boolean',
            'plan' => 'required|string|in:basic,premium,enterprise',
        ]);

        $tenant->update($request->all());

        return redirect()->route('super.tenants.index')
            ->with('success', 'Tenant updated successfully.');
    }

    public function destroy(Tenant $tenant)
    {
        $tenant->delete();

        return redirect()->route('super.tenants.index')
            ->with('success', 'Tenant deleted successfully.');
    }
}