<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Department;
use App\Models\Product;
use App\Models\Target;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TargetController extends Controller
{

    protected $owner;

    //H - B : Constructor
    public function __construct()
    {
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->owner = session('owner');
            return $next($request);
        });
    }

    // H - B: Targets Listing Page
    public function listing()
    {
        $targets = Target::where('tenant_id', $this->owner['id'])->whereNot('status', '1')->get();
        $departments = Department::where('tenant_id', $this->owner['id'])->get();
        $products = Product::where('tenant_id', $this->owner['id'])->get();
        return view('SuperAdmin.target.listing',
            compact('targets', 'departments', 'products'));
    }

    // H-B: Targets with auto generation
//    public function addAndEditTarget(Request $request)
//    {
//        $validated = $request->validate([
//            'department_id' => 'required',
//            'category' => 'required',
//            'quantity' => 'required|numeric',
//            'last_date' => 'required',
//            'status' => 'required',
//        ]);
//
//        try {
//            $formattedDate = Carbon::createFromFormat('m/d/Y', $request->last_date)->format('Y-m-d');
//        } catch (\Exception $e) {
//            return response()->json([
//                'Error' => true,
//                'Message' => 'Invalid date format. Please use mm/dd/yyyy.',
//            ]);
//        }
//
//        $data = [
//            'department_id' => $request->department_id,
//            'category' => $request->category,
//            'product_id' => $request->product_id ?? null,
//            'custom' => $request->custom ?? null,
//            'quantity' => $request->quantity,
//            'status' => $request->status,
//            'last_date' => $formattedDate,
//            'tenant_id' => $this->owner['id'],
//            'parent_target' => null,
//        ];
//
//        $currentMonthEnd = Carbon::now()->endOfMonth();
//
//        // If last_date is within the current month, create target without parent/child
//        if (Carbon::parse($formattedDate)->lte($currentMonthEnd)) {
//            Target::create($data);
//        } else {
//            $trgt = Target::find($request->id);
//
//            try {
//                if ($trgt) {
//                    $trgt->update($data);
//
//                    $childTargets = Target::where('parent_target', $trgt->id)
//                        ->where('last_date', '>', Carbon::now())
//                        ->get();
//
//                    $remainingQuantity = $data['quantity'];
//                    $startDate = Carbon::now();
//                    $childCount = 0;
//
//                    foreach ($childTargets as $child) {
//                        $endOfMonth = $startDate->copy()->endOfMonth();
//                        $childLastDate = $endOfMonth->lt(Carbon::parse($formattedDate)) ? $endOfMonth : Carbon::parse($formattedDate);
//
//                        $daysInPeriod = $startDate->diffInDays($childLastDate) + 1;
//                        $totalDays = Carbon::now()->diffInDays(Carbon::parse($formattedDate)) + 1;
//
//                        $childQuantity = round(($daysInPeriod / $totalDays) * $data['quantity']);
//                        $remainingQuantity -= $childQuantity;
//
//                        if ($childLastDate->eq(Carbon::parse($formattedDate))) {
//                            $childQuantity += $remainingQuantity;
//                        }
//
//                        $child->update([
//                            'quantity' => $childQuantity,
//                            'last_date' => $childLastDate,
//                            'created_at' => $childCount === 0 ? $startDate : $startDate->copy()->startOfMonth(),
//                        ]);
//
//                        $startDate = Carbon::parse($childLastDate)->copy()->addDay();
//                        $childCount++;
//                    }
//                } else {
//                    $parentTarget = Target::create(array_merge($data, [
//                        'parent_target' => 'parent',
//                    ]));
//
//                    $childTargets = [];
//                    $totalQuantity = $data['quantity'];
//                    $remainingQuantity = $totalQuantity;
//                    $startDate = Carbon::now();
//                    $childCount = 0;
//
//                    while ($startDate->lt(Carbon::parse($formattedDate))) {
//                        $endOfMonth = $startDate->copy()->endOfMonth();
//                        $childLastDate = $endOfMonth->lt(Carbon::parse($formattedDate)) ? $endOfMonth : Carbon::parse($formattedDate);
//
//                        $daysInPeriod = $startDate->diffInDays($childLastDate) + 1;
//                        $totalDays = Carbon::now()->diffInDays(Carbon::parse($formattedDate)) + 1;
//
//                        $childQuantity = round(($daysInPeriod / $totalDays) * $totalQuantity);
//                        $remainingQuantity -= $childQuantity;
//
//                        if ($childLastDate->eq(Carbon::parse($formattedDate))) {
//                            $childQuantity += $remainingQuantity;
//                        }
//
//                        $childTargets[] = Target::create([
//                            'department_id' => $data['department_id'],
//                            'category' => $data['category'],
//                            'product_id' => $data['product_id'],
//                            'custom' => $data['custom'],
//                            'quantity' => $childQuantity,
//                            'status' => $childCount === 0 ? '2' : '1',
//                            'last_date' => $childLastDate,
//                            'tenant_id' => $this->owner['id'],
//                            'parent_target' => $parentTarget->id,
//                            'created_at' => $childCount === 0 ? $startDate : $startDate->copy()->startOfMonth(),
//                        ]);
//
//                        $startDate = Carbon::parse($childLastDate)->copy()->addDay();
//                        $childCount++;
//                    }
//                }
//            } catch (\Throwable $t) {
//                return response()->json([
//                    'Error' => true,
//                    'Message' => $t->getMessage(),
//                ]);
//            }
//        }
//
//        return response()->json([
//            'Error' => false,
//            'Message' => "Target Assigned Successfully!",
//        ]);
//    }

    //H-B : Target with Auto Generation and Conversion
    public function addAndEditTarget(Request $request)
    {
        $validated = $request->validate([
            'department_id' => 'required',
            'category' => 'required',
            'quantity' => 'required|numeric',
            'last_date' => 'required',
            'status' => 'required',
        ]);
        
        // dd($request->all());

        try {
            $formattedDate = Carbon::createFromFormat('m/d/Y', $request->last_date)->format('Y-m-d');
        } catch (\Exception $e) {
            return response()->json([
                'Error' => true,
                'Message' => 'Invalid date format. Please use mm/dd/yyyy.',
            ]);
        }
        
        try {
            if($request->custom != null)
            {
                $match = DB::table('targets')
                ->where('custom', $request->custom)
                ->where('department_id', $request->department_id)
                ->whereBetween('last_date', [
                    Carbon::now()->startOfMonth(),
                    Carbon::now()->endOfMonth()
                ])
                ->whereIn('status', ['2', '0'])
                ->exists(); // Check if any record exists
            if($match)
            {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Target with the same name already exist, Custom Target must be renamed.',
                ]);
            }
            }
        } catch (\Exception $e)
        {
            return response()->json([
                'Error' => true,
                'Message' => 'Invalid target Matching',
            ]);
        }

        $data = [
            'department_id' => $request->department_id,
            'category' => $request->category,
            'product_id' => $request->product_id ?? null,
            'custom' => $request->custom ?? null,
            'quantity' => $request->quantity,
            'status' => $request->status,
            'last_date' => $formattedDate,
            'tenant_id' => $this->owner['id'],
            'parent_target' => null,
        ];

        $currentMonthEnd = Carbon::now()->endOfMonth();
        $target = Target::find($request->id);

        try {
            if ($target) {
                $previousLastDate = Carbon::parse($target->last_date);
                $newLastDate = Carbon::parse($formattedDate);

                // If previously within the current month but now extends beyond, convert to parent
                if ($previousLastDate->lte($currentMonthEnd) && $newLastDate->gt($currentMonthEnd)) {
                    $target->update(['parent_target' => 'parent']);
                    $this->createChildTargets($target, $newLastDate);
                }
                // If parent target's last_date is reduced, delete future children with zero progress
                elseif ($target->parent_target === 'parent' && $newLastDate->lt($previousLastDate)) {
                    $this->deleteFutureChildren($target->id, $newLastDate);
                }

                $target->update($data);
            } else {
                // If last_date is within the current month, create a simple target
                if (Carbon::parse($formattedDate)->lte($currentMonthEnd)) {
                    Target::create($data);
                } else {
                    // Creating a new parent target with child targets
                    $parentTarget = Target::create(array_merge($data, [
                        'parent_target' => 'parent',
                    ]));

                    $this->createChildTargets($parentTarget, Carbon::parse($formattedDate));
                }
            }
        } catch (\Throwable $t) {
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }

        return response()->json([
            'Error' => false,
            'Message' => "Target Assigned Successfully!",
        ]);
    }

    // H-B: Create child targets for a parent target.
    private function createChildTargets($parentTarget, $lastDate)
    {
        $totalQuantity = $parentTarget->quantity;
        $remainingQuantity = $totalQuantity;
        $startDate = Carbon::now();
        $childCount = 0;

        while ($startDate->lt($lastDate)) {
            $endOfMonth = $startDate->copy()->endOfMonth();
            $childLastDate = $endOfMonth->lt($lastDate) ? $endOfMonth : $lastDate;

            $daysInPeriod = $startDate->diffInDays($childLastDate) + 1;
            $totalDays = Carbon::now()->diffInDays($lastDate) + 1;

            $childQuantity = round(($daysInPeriod / $totalDays) * $totalQuantity);
            $remainingQuantity -= $childQuantity;

            if ($childLastDate->eq($lastDate)) {
                $childQuantity += $remainingQuantity;
            }

            Target::create([
                'department_id' => $parentTarget->department_id,
                'category' => $parentTarget->category,
                'product_id' => $parentTarget->product_id,
                'custom' => $parentTarget->custom,
                'quantity' => $childQuantity,
                'status' => $childCount === 0 ? '2' : '1',
                'last_date' => $childLastDate,
                'tenant_id' => $parentTarget->tenant_id,
                'parent_target' => $parentTarget->id,
                'created_at' => $childCount === 0 ? $startDate : $startDate->copy()->startOfMonth(),
            ]);

            $startDate = Carbon::parse($childLastDate)->copy()->addDay();
            $childCount++;
        }
    }
    // H-B: Delete future Child
    private function deleteFutureChildren($parentId, $newLastDate)
    {
        $futureChildren = Target::where('parent_target', $parentId)
            ->where('last_date', '>', $newLastDate)
            ->get();

        foreach ($futureChildren as $child) {
            $point = Points::where('target_id', $child->id)->exists();
            if ($child->progress == '0' && $child->status == '0' && !$point) { // Ensure progress is zero before deletion
                $child->delete();
            }
        }
    }


    // H - B : Delete Target
    public function deleteTarget(Request $request)
    {
        $trgt = Target::find($request->id);
        if(!$trgt)
        {
            return response()->json([
                'Error'=>true,
                'Message'=>"Target Not Found!"
            ]);
        }
        else
        {
            $trgt->update(['status' => '1']);
            return response()->json([
                'Error'=>false,
                'Message'=>"Target Removed Successfully!"
            ]);
        }
    }

    // H - B : Renew Target
    public function renewTarget(Request $request)
    {
        $trgt = Target::find($request->id);
        if(!$trgt)
        {
            return response()->json([
                'Error'=>true,
                'Message'=>"Target Not Found!"
            ]);
        }
        else
        {
            if($trgt->progress == $trgt->quantity && $trgt->completion_date != null)
            {
                $daysDifference = $trgt->created_at->diffInDays($trgt->last_date);
                $newDate = Carbon::now()->addDays($daysDifference + 1)->format('Y-m-d');

                $data = [
                    'department_id' => $trgt->department_id,
                    'category' => $trgt->category,
                    'product_id' => $trgt->product_id,
                    'custom' => $trgt->custom,
                    'quantity' => $trgt->quantity,
                    'status' => '2'
                ];
                $data['last_date'] = $newDate;
                $data['tenant_id'] = $this->owner['id'];

                //Created New Target With Same info
                $attempt = Target::create($data);
                //Making base target inactive
                $inactive = $trgt->update(['status' => '1']);

                if($attempt && $inactive)
                {
                    return response()->json([
                        'Error'=>false,
                        'Message'=>"Target Renewed Successfully!"
                    ]);
                }
                else
                {
                    return response()->json([
                        'Error'=>true,
                        'Message'=>"Target Renewal Failed!"
                    ]);
                }
            }
            else
            {
                return response()->json([
                    'Error'=>true,
                    'Message'=>"For Renewal, Target Must Be Completed First!"
                ]);
            }
        }
    }

}
