<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Department;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class SubAdminsController extends Controller
{
    protected $owner;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->owner = session('owner');
            return $next($request);
        });
    }

    // Subadmin Listing Page
    public function listing()
    {
        // Get all admins/subadmins for the listing
        $admins = Admin::where('tenant_id', $this->owner['id'])
            ->where('subadmin', 1)
            ->get();

        $departments = Department::where('tenant_id', $this->owner['id'])->get();
        return view('SuperAdmin.subadmin.list', compact('admins', 'departments'));
    }

    // Subadmin Add and Edit
    public function addAndEditSubAdmin(Request $request)
    {
        $validate = $request->validate([
            'email' => 'required|email|unique:admins,email,' . ($request->admin_id ?? 'NULL') . ',id,tenant_id,' . $this->owner['id'],
            'password' => $request->admin_id ? 'nullable|min:6' : 'required|min:6',
        ]);

        try {
            if ($request->admin_id) {
                // Update existing subadmin
                $admin = Admin::where('id', $request->admin_id)
                    ->where('tenant_id', $this->owner['id'])
                    ->where('subadmin', 1)
                    ->first();

                if (!$admin) {
                    return response()->json([
                        'Error' => true,
                        'Message' => 'SubAdmin not found!',
                    ]);
                }

                $updateData = [
                    'name' => $request->name,
                    'email' => $request->email,
                ];

                // Only update password if provided
                if ($request->filled('password')) {
                    $updateData['password'] = Hash::make($request->password);
                }

                $admin->update($updateData);

                return response()->json([
                    'Error' => false,
                    'Message' => 'SubAdmin Updated Successfully!',
                ]);

            } else {
                // Create new subadmin
                Admin::create([
                    'name' => $request->name,
                    'email' => $request->email,
                    'password' => Hash::make($request->password),
                    'tenant_id' => $this->owner['id'],
                    'subadmin' => 1, // Mark as subadmin
                ]);

                return response()->json([
                    'Error' => false,
                    'Message' => 'SubAdmin Created Successfully!',
                ]);
            }

        } catch (\Throwable $t) {
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    // SubAdmin Delete
    public function deleteSubAdmin(Request $request)
    {
        $validate = $request->validate([
            'id' => 'required'
        ]);

        try {
            // Find subadmin by id
            $admin = Admin::where('id', $request->id)
                ->where('tenant_id', $this->owner['id'])
                ->where('subadmin', 1)
                ->first();

            if ($admin) {
                // Delete subadmin record
                $admin->delete();

                return response()->json([
                    'Error' => false,
                    'Message' => "SubAdmin deleted Successfully",
                ]);
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => "SubAdmin not Found!"
                ]);
            }
        } catch (\Throwable $t) {
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }
}