<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\CategoryLeaderBoard;
use App\Models\Department;
use App\Models\DepartmentLeaderBoard;
use App\Models\EmployeeLeaderBoard;
use App\Models\Points;
use App\Models\Product;
use App\Models\Target;
use App\Models\Employee;
use App\Services\LeaderboardService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{

    protected $owner;
    protected $leaderboardService;

    //H - B : Constructor
    public function __construct()
    {
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->owner = session('owner');
            if ($this->owner) {
                $this->leaderboardService = new LeaderboardService($this->owner['id']);
            }
            return $next($request);
        });
    }

    // H-B : Dashboard
    public function dashboard()
    {
        $departments = Department::where('tenant_id', $this->owner['id'])->get(['id', 'name'])->toArray();

        $points = Points::select(DB::raw('DATE(updated_at) as date'), 'department_id', DB::raw('SUM(points) as total_points'))
            ->where('tenant_id', $this->owner['id'])
            ->where('status', 'Redeemed')
            ->wherebetween('updated_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])
            ->groupBy(DB::raw('DATE(updated_at)'), 'department_id')
            ->orderBy(DB::raw('DATE(updated_at)'), 'asc')
            ->get();

        $piePoints = Points::select(
            DB::raw("SUM(CASE WHEN status = 'Redeemed' THEN points ELSE 0 END) as redeemed"),
            DB::raw("SUM(CASE WHEN status = 'Pending' THEN points ELSE 0 END) as pending")
        )
            ->where('tenant_id', $this->owner['id'])
            ->first();

        $piePoints->total_points = $piePoints->redeemed + $piePoints->pending;

        // Get assigned custom emojis/GIFs for this superadmin
        $assignedEmojis = $this->getAssignedEmojis($this->owner['id']);

        return view('SuperAdmin.dashboard.dashboard', compact('departments', 'points', 'piePoints', 'assignedEmojis'));
    }

    private function getAssignedEmojis($superAdminId)
    {
        // Get assigned emojis/GIFs for this superadmin directly
        return \App\Models\EmojiAssignment::with('emoji')
            ->where('assignable_id', $superAdminId)
            ->where('assignable_type', 'App\Models\SuperAdmin')
            ->where('tenant_id', $this->owner['id'])
            ->orderBy('created_at', 'desc')
            ->get();
    }

 public function ownerPoints()
    {
        $owner = $this->owner;
        $products = Product::where('tenant_id', $this->owner['tenant_id'])->get();
        $targets = Target::where('tenant_id', $this->owner['tenant_id'])->get();
        $points = Points::select(
            'points.*',
            'employees.name as e_name',
            'targets.product_id as prod',
            'targets.custom as custom'
        )
            ->join('employees', 'points.employee_id', '=', 'employees.employee_id') // Join with employees table
            ->leftJoin('targets', 'points.target_id', '=', 'targets.id')  // Left join to include nullable fields
            ->where('points.tenant_id', $this->owner['tenant_id'])
            ->get();

        $employees = Employee::where('tenant_id', $this->owner['tenant_id'])
            ->where('role', 'Employee')
            ->get(['name', 'status', 'employee_id', 'category', 'department_id']);


        return view('SuperAdmin.RequestedPoints', compact('owner', 'employees', 'points', 'targets', 'products'));
    }
    // H - B : Main Filter On Dashboard
    public function getPointsData(Request $request)
    {
        $filter = $request->input('filter');

        $query = Points::where('tenant_id', $this->owner['id'])
            ->where('status', 'Redeemed')
            ->select(DB::raw('SUM(points) as total_points'));

        switch ($filter) {
            case 'today':
                $query->addSelect(DB::raw('HOUR(updated_at) as time_group')) // Group by hour
                    ->whereDate('updated_at', today())
                    ->groupBy('time_group')
                    ->orderBy('time_group');
                break;
            case 'week':
                $query->addSelect(DB::raw('DATE(updated_at) as time_group')) // Extract only the date part
                    ->whereBetween('updated_at', [now()->subDays(6)->startOfDay(), now()->endOfDay()]) // Last 7 days including today
                    ->groupBy('time_group') // Group by each day
                    ->orderBy('time_group', 'desc'); // Reverse chronological order

                break;
            case 'month':
                $query->addSelect(DB::raw('DATE(updated_at) as time_group')) // Group by date
                    ->whereMonth('updated_at', now()->month)
                    ->groupBy('time_group')
                    ->orderBy('time_group');
                break;
            case 'quarter':
                // Determine the current quarter start and end dates
                $currentMonth = now()->month;
                $currentYear = now()->year;

                if ($currentMonth >= 1 && $currentMonth <= 3) {
                    $startDate = Carbon::create($currentYear, 1, 1);
                    $endDate = Carbon::create($currentYear, 3, 31);
                } elseif ($currentMonth >= 4 && $currentMonth <= 6) {
                    $startDate = Carbon::create($currentYear, 4, 1);
                    $endDate = Carbon::create($currentYear, 6, 30);
                } elseif ($currentMonth >= 7 && $currentMonth <= 9) {
                    $startDate = Carbon::create($currentYear, 7, 1);
                    $endDate = Carbon::create($currentYear, 9, 30);
                } else {
                    $startDate = Carbon::create($currentYear, 10, 1);
                    $endDate = Carbon::create($currentYear, 12, 31);
                }

                $query->addSelect(DB::raw('MONTH(updated_at) as time_group, YEAR(updated_at) as year_group')) // Group by month
                    ->whereBetween('updated_at', [$startDate, $endDate])
                    ->groupBy('year_group', 'time_group')
                    ->orderByRaw('year_group, time_group');
                break;

            case '6_months':
                $query->addSelect(DB::raw('MONTH(updated_at) as time_group, YEAR(updated_at) as year_group')) // Group by month
                    ->whereBetween('updated_at', [now()->subMonths(6), now()])
                    ->groupBy('year_group', 'time_group')
                    ->orderByRaw('year_group, time_group');
                break;
            case 'year':
                $query->addSelect(DB::raw('MONTH(updated_at) as time_group, YEAR(updated_at) as year_group')) // Group by month
                    ->whereYear('updated_at', now()->year)
                    ->groupBy('year_group', 'time_group')
                    ->orderByRaw('year_group, time_group');
                break;
        }

        $data = $query->get();

        // Format the data for "6 Months" and "Year" filters
        if (in_array($filter, ['6_months', 'year'])) {
            $data->transform(function ($item) {
                $item->time_group = date('M Y', mktime(0, 0, 0, $item->time_group, 1, $item->year_group));
                unset($item->year_group); // Remove unnecessary field
                return $item;
            });
        }
        $totalSum = array_sum(array_column($data->toArray(), 'total_points'));

        //        dd($filter, $data->toArray(), $totalSum);

        return response()->json([
            'data' => $data,
            'total_sum' => $totalSum
        ]);
    }

    // H - B : Departments Filter On Dashboard
    public function getDepPointsData(Request $request)
    {
        $filter = $request->query('filter');
        $depID = $request->query('depID');

        $query = Points::where('tenant_id', $this->owner['id'])
            ->where('status', 'Redeemed')
            ->where('department_id', $depID)
            ->select(DB::raw('SUM(points) as total_points'));

        switch ($filter) {
            case 'today':
                $query->addSelect(DB::raw('HOUR(updated_at) as time_group')) // Group by hour
                    ->whereDate('updated_at', today())
                    ->groupBy('time_group')
                    ->orderBy('time_group');
                break;
            case 'week':
                $query->addSelect(DB::raw('DATE(updated_at) as time_group')) // Extract only the date part
                    ->whereBetween('updated_at', [now()->subDays(6)->startOfDay(), now()->endOfDay()]) // Last 7 days including today
                    ->groupBy('time_group') // Group by each day
                    ->orderBy('time_group', 'desc'); // Reverse chronological order

                break;
            case 'month':
                $query->addSelect(DB::raw('DATE(updated_at) as time_group')) // Group by date
                    ->whereMonth('updated_at', now()->month)
                    ->groupBy('time_group')
                    ->orderBy('time_group');
                break;
            case 'quarter':
                // Determine the current quarter start and end dates
                $currentMonth = now()->month;
                $currentYear = now()->year;

                if ($currentMonth >= 1 && $currentMonth <= 3) {
                    $startDate = Carbon::create($currentYear, 1, 1);
                    $endDate = Carbon::create($currentYear, 3, 31);
                } elseif ($currentMonth >= 4 && $currentMonth <= 6) {
                    $startDate = Carbon::create($currentYear, 4, 1);
                    $endDate = Carbon::create($currentYear, 6, 30);
                } elseif ($currentMonth >= 7 && $currentMonth <= 9) {
                    $startDate = Carbon::create($currentYear, 7, 1);
                    $endDate = Carbon::create($currentYear, 9, 30);
                } else {
                    $startDate = Carbon::create($currentYear, 10, 1);
                    $endDate = Carbon::create($currentYear, 12, 31);
                }

                $query->addSelect(DB::raw('MONTH(updated_at) as time_group, YEAR(updated_at) as year_group')) // Group by month
                    ->whereBetween('updated_at', [$startDate, $endDate])
                    ->groupBy('year_group', 'time_group')
                    ->orderByRaw('year_group, time_group');
                break;

            case '6_months':
                $query->addSelect(DB::raw('MONTH(updated_at) as time_group, YEAR(updated_at) as year_group')) // Group by month
                    ->whereBetween('updated_at', [now()->subMonths(6), now()])
                    ->groupBy('year_group', 'time_group')
                    ->orderByRaw('year_group, time_group');
                break;
            case 'year':
                $query->addSelect(DB::raw('MONTH(updated_at) as time_group, YEAR(updated_at) as year_group')) // Group by month
                    ->whereYear('updated_at', now()->year)
                    ->groupBy('year_group', 'time_group')
                    ->orderByRaw('year_group, time_group');
                break;
        }

        $data = $query->get();

        // Format the data for "6 Months" and "Year" filters
        if (in_array($filter, ['6_months', 'year'])) {
            $data->transform(function ($item) {
                $item->time_group = date('M Y', mktime(0, 0, 0, $item->time_group, 1, $item->year_group));
                unset($item->year_group); // Remove unnecessary field
                return $item;
            });
        }
        $totalSum = array_sum(array_column($data->toArray(), 'total_points'));

        return response()->json([
            'data' => $data,
            'total_sum' => $totalSum
        ]);
    }

    // H-B : LeaderBoard
    public function viewLeaderBoard()
    {
        $currentMonth = Carbon::now()->format('Y-m');

        $months = EmployeeLeaderBoard::select('month_year')
            ->distinct()
            ->orderBy('month_year', 'desc')
            ->pluck('month_year');
            
        // Get available categories from category_leaderboard table
        $availableCategories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('departments.name')
            ->distinct()
            ->where('category_leader_board.tenant_id', $this->owner['id'])
            ->pluck('name')
            ->filter()
            ->values();

        //        $employees = EmployeeLeaderBoard::join('employees', function($join) {
//            $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
//                ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id'); // Match tenant_id
//        })
//            ->where('employee_leader_boards.tenant_id', '=', $this->owner['id']) // Ensure tenant_id matches in employee_leader_boards
//            ->whereBetween('employee_leader_boards.updated_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])
//            ->orderBy('employee_leader_boards.rank', 'asc')
//            ->get([
//                'employee_leader_boards.*',
//                'employees.name as employee',
//                'employees.category as category',
//                'employees.image as image'
//            ]);

        $employees = EmployeeLeaderBoard::join('employees', function ($join) {
            $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id'); // Match tenant_id
        })
            ->where('employee_leader_boards.tenant_id', '=', $this->owner['id']) // Ensure tenant_id matches in employee_leader_boards
            ->where('month_year', $currentMonth)
            ->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee',
                'employees.category as category',
                'employees.image as image'
            ]);

        //        $departments = DepartmentLeaderBoard::join('departments', function($join) {
//            $join->on('department_leader_boards.department_id', '=', 'departments.id')
//                ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id'); // Match tenant_id
//        })
//            ->join('employees', function($join) {
//                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
//                    ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id') // Match tenant_id
//                    ->where('employees.role', 'Manager');
//            })
//            ->where('department_leader_boards.tenant_id', '=', $this->owner['id']) // Match tenant_id with $this->owner['id']
//            ->orderBy('department_leader_boards.rank', 'asc')
//            ->get(['department_leader_boards.*',
//                'departments.name as department',
//                'employees.name as manager',
//                'employees.image as emp_image'
//            ]);

        $departments = DepartmentLeaderBoard::join('departments', function ($join) {
            $join->on('department_leader_boards.department_id', '=', 'departments.id')
                ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id'); // Match tenant_id
        })
            ->join('employees', function ($join) {
                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
                    ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id') // Match tenant_id
                    ->where('employees.role', 'Manager');
            })
            ->where('department_leader_boards.tenant_id', '=', $this->owner['id']) // Match tenant_id
            ->where('month_year', $currentMonth)
            ->groupBy('department_leader_boards.id', 'department_leader_boards.rank', 'department_leader_boards.earned_points', 'departments.name') // Group by leaderboard ID to avoid duplicates
            ->orderBy('department_leader_boards.rank', 'asc')
            ->get([
                'department_leader_boards.id',
                'department_leader_boards.rank',
                'department_leader_boards.earned_points',
                'departments.name as department',
                \DB::raw('COUNT(DISTINCT employees.employee_id) as count'), // Count unique managers
                \DB::raw('MIN(employees.name) as manager'), // Get any one manager’s name
                \DB::raw('MIN(employees.image) as emp_image') // Get any one manager’s image
            ]);

        $categories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('category_leader_board.*', 'departments.name as name')
            ->where('month_year', $currentMonth)->get();

        return view('SuperAdmin.scores.leaderBoard', compact('employees', 'categories', 'departments', 'months', 'availableCategories'));
    }

    // H-B: Employee Filter LeaderBoard
    public function viewEmpLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"
        
        // Get category filter from request
        $category = request()->get('category');

        $query = EmployeeLeaderBoard::join('employees', function ($join) {
            $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id'); // Match tenant_id
        })
            ->join('departments', function ($join) {
                $join->on('employees.department_id', '=', 'departments.id')
                    ->on('employees.tenant_id', '=', 'departments.tenant_id');
            })
            ->where('employee_leader_boards.tenant_id', '=', $this->owner['id']) // Ensure tenant_id matches in employee_leader_boards
            ->where('month_year', $currentMonth);
            
        // Apply category filter if provided (filter by department name)
        if ($category && $category !== 'all') {
            $query->where('departments.name', $category);
        }
        
        $employees = $query->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee',
                'employees.category as category',
                'employees.image as image',
                'departments.name as department_name'
            ]);
        return view('partials.employeeLeaderBoard', compact('employees'))->render();
    }

    // H-B: Employee Filter LeaderBoard
    public function viewDepLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"

        $departments = DepartmentLeaderBoard::join('departments', function ($join) {
            $join->on('department_leader_boards.department_id', '=', 'departments.id')
                ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id'); // Match tenant_id
        })
            ->join('employees', function ($join) {
                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
                    ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id') // Match tenant_id
                    ->where('employees.role', 'Manager');
            })
            ->where('department_leader_boards.tenant_id', '=', $this->owner['id']) // Match tenant_id
            ->where('month_year', $currentMonth)
            ->groupBy('department_leader_boards.id', 'department_leader_boards.rank', 'department_leader_boards.earned_points', 'departments.name')
            ->orderBy('department_leader_boards.rank', 'asc')
            ->get([
                'department_leader_boards.id',
                'department_leader_boards.rank',
                'department_leader_boards.earned_points',
                'departments.name as department',
                \DB::raw('COUNT(DISTINCT employees.employee_id) as count'), // Count unique managers
                \DB::raw('MIN(employees.name) as manager'), // Get any one manager’s name
                \DB::raw('MIN(employees.image) as emp_image') // Get any one manager’s image
            ]);
        //        dd($departments);
        return view('partials.departmentLeaderBoard', compact('departments'))->render();
    }

    // H-B: Employee Filter LeaderBoard
    public function viewCatLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"

        $categories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('category_leader_board.*', 'departments.name as name')
            ->where('category_leader_board.tenant_id', '=', $this->owner['id'])
            ->where('month_year', $currentMonth)
            ->get();
        return view('partials.categoryLeaderBoard', compact('categories'))->render();
    }

    // Enhanced Leaderboard Methods
    
    /**
     * Enhanced leaderboard view with all filters
     */
    public function viewEnhancedLeaderboard()
    {
        $currentMonth = Carbon::now()->format('Y-m');
        
        // Get available months
        $months = $this->leaderboardService->getAvailableMonths();
        
        // Get filter options
        $departments = $this->leaderboardService->getAvailableDepartments();
        $categories = $this->leaderboardService->getAvailableCategories();
        
        // Get initial data
        $leaderboardData = $this->leaderboardService->getLeaderboardData([
            'category' => 'employees',
            'time_period' => 'current_month'
        ]);
        
        return view('SuperAdmin.EnhancedLeaderboard', compact(
            'leaderboardData', 
            'months', 
            'departments', 
            'categories'
        ));
    }
    
    /**
     * AJAX endpoint for filtered leaderboard data
     */
    public function getFilteredLeaderboard(Request $request)
    {
        $filters = $request->only(['category', 'time_period', 'department_id', 'custom_date_range', 'specific_category', 'month']);
        
        try {
            // If specific_category is true, filter by the specific category
            if ($request->get('specific_category') && $request->get('category') !== 'all') {
                $data = $this->leaderboardService->getLeaderboardDataByCategory($filters);
            } else {
                $data = $this->leaderboardService->getLeaderboardData($filters);
            }
            
            $prizes = $this->leaderboardService->getPrizesForLeaderboard(
                $filters['category'] ?? 'all',
                $filters['time_period'] ?? 'current_month'
            );
            
            return response()->json([
                'success' => true,
                'data' => $data,
                'top3' => $data->take(3),
                'complete_ranking' => $data->skip(3),
                'total_count' => $data->count(),
                'prizes' => $prizes
            ]);
        } catch (\Exception $e) {
            Log::error('Error in SuperAdmin getFilteredLeaderboard: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load leaderboard data'
            ], 500);
        }
    }
    
    /**
     * Get BDC leaderboard specifically
     */
    public function getBDCLeaderboard(Request $request)
    {
        $timePeriod = $request->get('time_period', 'current_month');
        $data = $this->leaderboardService->getBDCLeaderboard($timePeriod);
        
        return view('partials.enhancedLeaderboard', [
            'data' => $data,
            'category' => 'BDC',
            'top3' => $data->take(3),
            'complete_ranking' => $data->skip(3)
        ])->render();
    }
    
    /**
     * Get Service Advisor leaderboard specifically
     */
    public function getServiceAdvisorLeaderboard(Request $request)
    {
        $timePeriod = $request->get('time_period', 'current_month');
        $data = $this->leaderboardService->getServiceAdvisorLeaderboard($timePeriod);
        
        return view('partials.enhancedLeaderboard', [
            'data' => $data,
            'category' => 'Service Advisor',
            'top3' => $data->take(3),
            'complete_ranking' => $data->skip(3)
        ])->render();
    }
}
