<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Department;
use App\Models\Employee;
use App\Services\UniversalNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class EmployeeController extends Controller
{
    protected $owner;
    private UniversalNotificationService $notificationService;

    //H - B : Constructor
    public function __construct(UniversalNotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->owner = session('owner');
            return $next($request);
        });
    }

    // H - B : View Listing page
    public function listing()
    {
        $departments = Department::where('tenant_id', $this->owner['id'])->get();
        $employees = Employee::with('department')
            ->where('tenant_id', $this->owner['id'])
            ->get();
        
        // Get available emojis/GIFs created by superadmin
        $availableEmojis = \App\Models\CustomEmoji::forTenant($this->owner['id'])
            ->active()
            ->orderBy('type')
            ->orderBy('name')
            ->get();
            
        return view ('SuperAdmin.employee.listing', compact('employees', 'departments', 'availableEmojis'));
    }

    // H - B : Add And Edit Employee
    public function addAndEditEmployee(Request $request)
    {
        $employee = Employee::where('tenant_id', $this->owner['id'])
            ->where('employee_id', $request->employee_id)->first();

        $validationRules = [
            'employee_id' => 'required',
            'name' => 'required',
            'email' => 'required',
            'role' => 'required',
            'emoji' => 'nullable|string|max:10'
        ];
        
        if(!$employee) {
            $validationRules['password'] = 'required';
        }
        
        if($request->role === 'Manager') {
            $validationRules['department'] = 'required | integer';
        } elseif ($request->role === 'Admin') {
            $validationRules['department'] = 'required | integer ';
            $validationRules['category'] = 'required';
        } else {
            $validationRules['department'] = 'nullable';
        }
        
        $customMessages = [
            'department.required' => 'The department field is required.',
            'department.integer' => 'The department must have a valid value.',
        ];
        
        $validate = $request->validate($validationRules, $customMessages);
        $image = null;
        $pas = null;
        
        // Store original emoji for comparison (for notifications)
        $originalEmoji = $employee ? $employee->emoji : null;
        $isNewEmployee = !$employee;

        if($employee) {
            if(isset($request->password)) {
                $pas = Hash::make($request->password);
            } else {
                $pas = $employee->password;
            }

            if(isset($request->image)) {
                // FIX: Add proper validation before deleting old image
                if (!empty($employee->image) && $employee->image !== '/' && file_exists(public_path($employee->image))) {
                    unlink(public_path($employee->image));
                }
                $image = saveFiles($request->image, 'Employees');
            } else {
                $image = $employee->image;
            }
        } else {
            $pas = Hash::make($request->password);
            if(isset($request->image)) {
                $image = saveFiles($request->image, 'Employees');
            } else {
                $image = null;
            }
        }

        $data = $request->only([
            'employee_id', 'name', 'email',
            'status', 'dob', 'phn', 'emoji', 'gender',
            'marital', 'nationality', 'city',
            'state', 'zip', 'address', 'role',
        ]);

        $data['password'] = $pas;
        $data ['image'] = $image;
        $data['department_id'] = $request->department;
        $data['category'] = $request->category;
        $data['tenant_id'] = $this->owner['id'];
        
        // Handle emoji - if no emoji provided, generate one based on name
        $emojiWasAutoGenerated = false;
        if (empty($data['emoji'])) {
            $data['emoji'] = $this->generateEmojiFromName($data['name']);
            $emojiWasAutoGenerated = true;
        }
        
        try {
            if($employee) {
                $old_dep = $employee->department_id;
                $empCount = Department::where('id', $old_dep)->where('tenant_id', $this->owner['id'])->first();
                if($empCount) {
                    $count = $empCount->no_of_emp - 1;
                    $empCount->update(['no_of_emp' => $count]);
                }
                
                $employee->update($data);
                
                if($request->department != null) {
                    $count = Employee::where('tenant_id', $this->owner['id'])
                        ->where('department_id', $request->department)->count();
                    $test = Department::where('id', $request->department)->update(['no_of_emp' => $count]);
                }

                // Send emoji change notification if emoji changed
                if ($originalEmoji !== $data['emoji']) {
                    Log::info('Emoji changed, sending notification', [
                        'employee_id' => $employee->employee_id,
                        'old_emoji' => $originalEmoji,
                        'new_emoji' => $data['emoji'],
                        'tenant_id' => $employee->tenant_id
                    ]);
                    
                    // Send emoji change notification
                    $this->sendEmojiChangeNotification($employee, $originalEmoji, $data['emoji'], $emojiWasAutoGenerated);
                }

                return response()->json([
                    'Error' => false,
                    'Message' => 'Employee Updated successfully!',
                    'NotificationSent' => ($originalEmoji !== $data['emoji']) ? 'Emoji change notification sent' : 'No emoji change',
                ]);
            } else {
                $newEmployee = Employee::create($data);
                
                // Handle emoji assignment if provided
                if ($request->has('emoji_id') && $request->emoji_id) {
                    $this->assignEmojiToEmployee($newEmployee, $request->emoji_id, $request->emoji_message);
                }
                
                Log::info('New employee created, sending welcome notification', [
                    'employee_id' => $newEmployee->employee_id,
                    'emoji' => $data['emoji'],
                    'tenant_id' => $newEmployee->tenant_id
                ]);
                
                // Send welcome notification with emoji for new employee
                $this->sendWelcomeNotification($newEmployee, $data['emoji'], $emojiWasAutoGenerated);

                return response()->json([
                    'Error' => false,
                    'Message' => 'Employee Added successfully!',
                    'NotificationSent' => 'Welcome notification sent with emoji: ' . $data['emoji'],
                ]);
            }

        } catch(\Throwable $t) {
            Log::error('Employee operation failed', [
                'error' => $t->getMessage(),
                'line' => $t->getLine(),
                'trace' => $t->getTraceAsString()
            ]);
            
            return response()->json([
                'Error' => true,
                'Message' => "line no : ".$t->getLine(). ", Msg : ".$t->getMessage(),
            ]);
        }
    }

    /**
     * Assign emoji/GIF to employee
     */
    private function assignEmojiToEmployee($employee, $emojiId, $message = null)
    {
        try {
            // Get the current user (SuperAdmin, Admin, or Employee)
            $assignedBy = $this->getCurrentUser();
            
            if ($assignedBy) {
                \App\Models\EmojiAssignment::create([
                    'assignable_id' => $employee->employee_id,
                    'assignable_type' => 'App\Models\Employee',
                    'assigned_by_id' => $assignedBy['id'],
                    'assigned_by_type' => $assignedBy['type'],
                    'emoji_id' => $emojiId,
                    'message' => $message,
                    'tenant_id' => $this->owner['id']
                ]);

                Log::info('Emoji assigned to employee', [
                    'employee_id' => $employee->employee_id,
                    'emoji_id' => $emojiId,
                    'assigned_by' => $assignedBy
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Failed to assign emoji to employee', [
                'employee_id' => $employee->employee_id,
                'emoji_id' => $emojiId,
                'error' => $e->getMessage()
            ]);
        }
    }

    private function getCurrentUser()
    {
        // Try to get High SuperAdmin first
        if (session('superadmin_id')) {
            return [
                'id' => session('superadmin_id'),
                'type' => 'App\Models\SuperAdmin'
            ];
        }
        
        // Try to get Tenant SuperAdmin
        if (session('owner')) {
            return [
                'id' => session('owner')['id'],
                'type' => 'App\Models\SuperAdmin'
            ];
        }
        
        // Try to get Admin
        if (session('isadmin')) {
            return [
                'id' => session('isadmin')['id'],
                'type' => 'App\Models\Admin'
            ];
        }
        
        // Try to get Employee
        if (session('isemp')) {
            return [
                'id' => session('isemp')['employee_id'],
                'type' => 'App\Models\Employee'
            ];
        }
        
        return null;
    }

    /**
     * Get emoji assignments for a specific employee
     */
    public function getEmojiAssignments($employeeId)
    {
        try {
            Log::info('Fetching emoji assignments', [
                'employee_id' => $employeeId,
                'tenant_id' => $this->owner['id'],
                'owner_data' => $this->owner
            ]);

            // Try with tenant filter first
            $assignments = \App\Models\EmojiAssignment::with('emoji')
                ->where('assignable_id', $employeeId)
                ->where('assignable_type', 'App\Models\Employee')
                ->where('tenant_id', $this->owner['id'])
                ->orderBy('created_at', 'desc')
                ->get();

            // If no assignments found, try without tenant filter to debug
            if ($assignments->count() === 0) {
                $assignmentsWithoutTenant = \App\Models\EmojiAssignment::with('emoji')
                    ->where('assignable_id', $employeeId)
                    ->where('assignable_type', 'App\Models\Employee')
                    ->orderBy('created_at', 'desc')
                    ->get();
                
                Log::info('No assignments with tenant filter, trying without:', [
                    'count' => $assignmentsWithoutTenant->count(),
                    'assignments' => $assignmentsWithoutTenant->toArray()
                ]);
            }

            Log::info('Found assignments', [
                'count' => $assignments->count(),
                'assignments' => $assignments->toArray()
            ]);

            return response()->json([
                'success' => true,
                'assignments' => $assignments
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching emoji assignments', [
                'employee_id' => $employeeId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch emoji assignments'
            ], 500);
        }
    }

    /**
     * Test method to debug emoji assignments
     */
    public function testEmojiAssignments($employeeId)
    {
        try {
            Log::info('=== TEST EMOJI ASSIGNMENTS ===');
            Log::info('Employee ID:', $employeeId);
            Log::info('Owner data:', $this->owner);
            
            // Check if employee exists
            $employee = Employee::where('employee_id', $employeeId)
                ->where('tenant_id', $this->owner['id'])
                ->first();
            
            Log::info('Employee found:', $employee ? $employee->toArray() : 'Not found');
            
            // Check all emoji assignments for this tenant
            $allAssignments = \App\Models\EmojiAssignment::where('tenant_id', $this->owner['id'])->get();
            Log::info('All assignments for tenant:', $allAssignments->toArray());
            
            // Check specific assignments for this employee
            $assignments = \App\Models\EmojiAssignment::with('emoji')
                ->where('assignable_id', $employeeId)
                ->where('assignable_type', 'App\Models\Employee')
                ->where('tenant_id', $this->owner['id'])
                ->get();
            
            Log::info('Specific assignments for employee:', $assignments->toArray());
            
            return response()->json([
                'success' => true,
                'employee_id' => $employeeId,
                'owner' => $this->owner,
                'employee' => $employee,
                'all_assignments' => $allAssignments,
                'specific_assignments' => $assignments
            ]);
            
        } catch (\Exception $e) {
            Log::error('Test error:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Assign emoji to employee (standalone assignment)
     */
    public function assignEmojiToEmployeeStandalone(Request $request)
    {
        try {
            $request->validate([
                'employee_id' => 'required|string',
                'emoji_id' => 'required|integer',
                'message' => 'nullable|string|max:500'
            ]);

            $employee = Employee::where('employee_id', $request->employee_id)
                ->where('tenant_id', $this->owner['id'])
                ->first();

            if (!$employee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee not found'
                ], 404);
            }

            $emoji = \App\Models\CustomEmoji::where('id', $request->emoji_id)
                ->where('tenant_id', $this->owner['id'])
                ->where('is_active', true)
                ->first();

            if (!$emoji) {
                return response()->json([
                    'success' => false,
                    'message' => 'Emoji/GIF not found or inactive'
                ], 404);
            }

            // Get the current user (SuperAdmin, Admin, or Employee)
            $assignedBy = $this->getCurrentUser();
            
            if (!$assignedBy) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unable to identify current user'
                ], 400);
            }

            // Create the assignment
            $assignment = \App\Models\EmojiAssignment::create([
                'assignable_id' => $employee->employee_id,
                'assignable_type' => 'App\Models\Employee',
                'assigned_by_id' => $assignedBy['id'],
                'assigned_by_type' => $assignedBy['type'],
                'emoji_id' => $request->emoji_id,
                'message' => $request->message,
                'tenant_id' => $this->owner['id']
            ]);

            Log::info('Emoji assigned to employee via standalone assignment', [
                'employee_id' => $employee->employee_id,
                'emoji_id' => $request->emoji_id,
                'assigned_by' => $assignedBy,
                'assignment_id' => $assignment->id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assigned successfully to ' . $employee->name,
                'assignment' => $assignment
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to assign emoji to employee', [
                'employee_id' => $request->employee_id,
                'emoji_id' => $request->emoji_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to assign emoji/GIF: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove emoji assignment
     */
    public function removeEmojiAssignment(Request $request)
    {
        try {
            $request->validate([
                'assignment_id' => 'required|integer'
            ]);

            $assignment = \App\Models\EmojiAssignment::where('id', $request->assignment_id)
                ->where('tenant_id', $this->owner['id'])
                ->first();

            if (!$assignment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Assignment not found'
                ], 404);
            }

            $employeeName = 'Unknown';
            if ($assignment->assignable_type === 'App\Models\Employee') {
                $employee = Employee::where('employee_id', $assignment->assignable_id)
                    ->where('tenant_id', $this->owner['id'])
                    ->first();
                if ($employee) {
                    $employeeName = $employee->name;
                }
            }

            $assignment->delete();

            Log::info('Emoji assignment removed', [
                'assignment_id' => $request->assignment_id,
                'employee_name' => $employeeName
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assignment removed successfully from ' . $employeeName
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to remove emoji assignment', [
                'assignment_id' => $request->assignment_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to remove emoji assignment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update existing emoji assignment
     */
    public function updateEmojiAssignment(Request $request)
    {
        try {
            $request->validate([
                'assignment_id' => 'required|integer',
                'emoji_id' => 'required|integer',
                'message' => 'nullable|string|max:500'
            ]);

            $assignment = \App\Models\EmojiAssignment::where('id', $request->assignment_id)
                ->where('tenant_id', $this->owner['id'])
                ->first();

            if (!$assignment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Assignment not found'
                ], 404);
            }

            $emoji = \App\Models\CustomEmoji::where('id', $request->emoji_id)
                ->where('tenant_id', $this->owner['id'])
                ->where('is_active', true)
                ->first();

            if (!$emoji) {
                return response()->json([
                    'success' => false,
                    'message' => 'Emoji/GIF not found or inactive'
                ], 404);
            }

            // Update the assignment
            $assignment->update([
                'emoji_id' => $request->emoji_id,
                'message' => $request->message
            ]);

            Log::info('Emoji assignment updated', [
                'assignment_id' => $assignment->id,
                'new_emoji_id' => $request->emoji_id,
                'message' => $request->message
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assignment updated successfully',
                'assignment' => $assignment->load('emoji')
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to update emoji assignment', [
                'assignment_id' => $request->assignment_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update emoji assignment: ' . $e->getMessage()
            ], 500);
        }
    }

    private function generateEmojiFromName($name)
    {
        $emojis = ['😊', '😎', '🤓', '😄', '🙂', '😇', '🤗', '😃', '😁', '🤩', '🥳', '😌', '🤠', '👨‍💻', '👩‍💻', '👨‍💼', '👩‍💼'];
        
        // Use the first character of the name to determine emoji
        $nameLength = strlen($name);
        $index = $nameLength % count($emojis);
        
        return $emojis[$index];
    }

    // Send welcome notification for new employee
    private function sendWelcomeNotification($employee, $emoji, $wasAutoGenerated)
    {
        try {
            $emojiSource = $wasAutoGenerated ? 'automatically assigned' : 'set';
            $content = $wasAutoGenerated 
                ? "Welcome to the team! We've assigned you a special emoji {$emoji} based on your name. This emoji represents you across the platform!"
                : "Welcome to the team! Your profile emoji {$emoji} has been set. This emoji represents you across the platform!";

            // Send welcome notification
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->owner['id'],
                'title' => "Welcome! Your emoji is ready {$emoji}",
                'content' => $content,
                'type' => 'welcome_emoji',
                'related_id' => $employee->employee_id
            ]);

            // Send celebratory emoji notification
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->owner['id'],
                'emoji' => $emoji . '🎉',
                'message' => 'Welcome to the team! Here\'s your special emoji!',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Welcome notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);

            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send welcome notification', [
                'employee_id' => $employee->employee_id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    // Send emoji change notification for existing employee
    private function sendEmojiChangeNotification($employee, $oldEmoji, $newEmoji, $wasAutoGenerated)
    {
        try {
            $changeType = $wasAutoGenerated ? 'automatically updated' : 'updated';
            $content = $wasAutoGenerated 
                ? "Hi {$employee->name}! Your profile emoji has been automatically updated from {$oldEmoji} to {$newEmoji} based on your profile changes."
                : "Hi {$employee->name}! Your profile emoji has been updated from {$oldEmoji} to {$newEmoji}. Your new emoji represents you across the platform!";

            // Send change notification
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->owner['id'],
                'title' => "Your emoji has been {$changeType}! {$newEmoji}",
                'content' => $content,
                'type' => 'emoji_update',
                'related_id' => $employee->employee_id
            ]);

            // Send new emoji as reaction
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->owner['id'],
                'emoji' => $newEmoji . '✨',
                'message' => 'Your new emoji is ready!',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Emoji change notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'old_emoji' => $oldEmoji,
                'new_emoji' => $newEmoji,
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);
            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send emoji change notification', [
                'employee_id' => $employee->employee_id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    // H - B : Delete Employee
    public function deleteEmployee(Request $request)
    {
        $employee = Employee::where('employee_id',$request->id)->first();

        try {
            if ($employee) {
                // Send farewell notification before deletion
                $notificationSent = $this->sendFarewellNotification($employee);

                // FIX: Add proper validation before deleting image file
                if (!empty($employee->image) && $employee->image !== '/' && file_exists(public_path($employee->image))) {
                    unlink(public_path($employee->image));
                }
                
                $depart = $employee->department_id;
                $employee->delete();
                $count = Employee::where('tenant_id', $this->owner['id'])
                    ->where('department_id', $depart)->count();
                $test = Department::where('id', $request->department)->update(['no_of_emp' => $count]);
                
                return response()->json([
                    'Error' => false,
                    'Message' => 'Employee record deleted successfully',
                    'NotificationSent' => $notificationSent ? 'Farewell notification sent' : 'Notification failed',
                ]);
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee Record not found!',
                ], 404);
            }
        } catch (\Throwable $t) {
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage()
            ]);
        }
    }

    // Send farewell notification before employee deletion
    private function sendFarewellNotification($employee)
    {
        try {
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->owner['id'],
                'title' => 'Farewell Message 👋',
                'content' => "Hi {$employee->name}, thank you for being part of our team. Your emoji {$employee->emoji} will be remembered. We wish you all the best in your future endeavors!",
                'type' => 'farewell',
                'related_id' => $employee->employee_id
            ]);

            // Send farewell emoji
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->owner['id'],
                'emoji' => '👋💫',
                'message' => 'Thank you for your service. Best wishes!',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Farewell notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);

            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send farewell notification', [
                'employee_id' => $employee->employee_id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    //Employee details view function
    public function employeeDetails(){
        $admin_count = Employee::where('role' , 'admin')->count();
        $manager_count = Employee::where('role' , 'manager')->count();
        return view('SuperAdmin.employee.employee_details' , compact('admin_count' , 'manager_count' ,));
    }
}