<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\CustomEmoji;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class CustomEmojiController extends Controller
{
    protected $owner;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->owner = session('owner');
            return $next($request);
        });
    }

    /**
     * Get the current user ID from session or auth
     */
    private function getCurrentUserId()
    {
        // Try to get from session first (for SuperAdmin)
        if (session()->has('super_admin_id')) {
            return session('super_admin_id');
        }
        
        // Try to get from auth
        if (auth()->check()) {
            return auth()->id();
        }
        
        // Fallback to 1 (Super Administrator)
        return 1;
    }

    /**
     * Display a listing of custom emojis and GIFs
     */
    public function index()
    {
        try {
            \Log::info('Loading emojis for tenant', ['tenant_id' => $this->owner['id']]);
            
            $emojis = CustomEmoji::forTenant($this->owner['id'])
                ->orderBy('created_at', 'desc')
                ->get();

            \Log::info('Loaded emojis', ['count' => $emojis->count()]);

            return view('SuperAdmin.CustomEmojiManagement', compact('emojis'));
        } catch (\Exception $e) {
            \Log::error('Error loading emojis:', ['error' => $e->getMessage()]);
            return view('SuperAdmin.CustomEmojiManagement', ['emojis' => collect()]);
        }
    }

    /**
     * Store a newly uploaded emoji or GIF
     */
    public function store(Request $request)
    {
        try {
            // Debug logging
            \Log::info('=== EMOJI STORE REQUEST START ===');
            \Log::info('Request data', $request->all());
            \Log::info('Owner data', $this->owner);
            \Log::info('Auth user', auth()->user() ? auth()->user()->toArray() : ['status' => 'Not authenticated']);
            \Log::info('Has file', ['has_file' => $request->hasFile('file')]);

            // Basic validation
            if (!$request->name || !$request->type) {
                \Log::error('Missing required fields', ['name' => $request->name, 'type' => $request->type]);
                return response()->json([
                    'success' => false,
                    'message' => 'Name and type are required'
                ], 400);
            }

            // Check tenant
            if (!$this->owner || !isset($this->owner['id'])) {
                \Log::error('No tenant found');
                return response()->json([
                    'success' => false,
                    'message' => 'No tenant found'
                ], 400);
            }

            // Get the current user ID (could be SuperAdmin, Admin, or Employee)
            $currentUserId = $this->getCurrentUserId();
            
            // Prepare base data
            $data = [
                'name' => $request->name,
                'type' => $request->type,
                'tenant_id' => $this->owner['id'],
                'created_by' => $currentUserId,
                'is_active' => true
            ];

            if ($request->type === 'emoji') {
                // Handle emoji text
                if (!$request->emoji_text) {
                    \Log::error('Emoji text is required for emoji type');
                    return response()->json([
                        'success' => false,
                        'message' => 'Emoji text is required'
                    ], 400);
                }

                $data['emoji_text'] = $request->emoji_text;
                $data['file_path'] = null;
                $data['file_name'] = null;
                $data['mime_type'] = 'text/emoji';
                $data['file_size'] = 0;

                \Log::info('Creating emoji with data', $data);

            } elseif ($request->type === 'gif') {
                // Handle GIF file
                if (!$request->hasFile('file')) {
                    \Log::error('File is required for GIF type');
                    return response()->json([
                        'success' => false,
                        'message' => 'File is required for GIF'
                    ], 400);
                }

                        $file = $request->file('file');
                        $fileName = \Str::uuid() . '.' . $file->getClientOriginalExtension();
                        
                        // Create directory if it doesn't exist
                        $uploadPath = public_path('custom-emojis');
                        if (!file_exists($uploadPath)) {
                            mkdir($uploadPath, 0755, true);
                        }
                        
                        // Move file to public/custom-emojis directory
                        $file->move($uploadPath, $fileName);
                        $filePath = 'custom-emojis/' . $fileName;

                $data['emoji_text'] = null;
                $data['file_path'] = $filePath;
                $data['file_name'] = $fileName;
                $data['mime_type'] = $file->getMimeType();
                $data['file_size'] = $file->getSize();

                \Log::info('Creating GIF with data', $data);
            } else {
                \Log::error('Invalid type', ['type' => $request->type]);
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid type. Must be emoji or gif'
                ], 400);
            }

            // Create the emoji
            \Log::info('Attempting to create CustomEmoji...');
            $customEmoji = CustomEmoji::create($data);
            \Log::info('Custom emoji created successfully', ['id' => $customEmoji->id, 'data' => $customEmoji->toArray()]);

            return response()->json([
                'success' => true,
                'message' => ucfirst($request->type) . ' created successfully',
                'emoji' => $customEmoji
            ]);

        } catch (\Exception $e) {
            \Log::error('=== EMOJI STORE ERROR ===');
            \Log::error('Error message:', ['error' => $e->getMessage()]);
            \Log::error('Error trace:', ['trace' => $e->getTraceAsString()]);
            \Log::error('Request data:', $request->all());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create ' . ($request->type ?? 'item'),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified emoji or GIF
     */
    public function update(Request $request, $id)
    {
        $customEmoji = CustomEmoji::forTenant($this->owner['id'])->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $customEmoji->update([
                'name' => $request->name,
                'is_active' => $request->has('is_active') ? $request->is_active : $customEmoji->is_active
            ]);

            return response()->json([
                'success' => true,
                'message' => ucfirst($customEmoji->type) . ' updated successfully',
                'emoji' => $customEmoji
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update ' . $customEmoji->type,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified emoji or GIF
     */
    public function destroy($id)
    {
        try {
            $customEmoji = CustomEmoji::forTenant($this->owner['id'])->findOrFail($id);

            // Delete the file from storage
            if (Storage::disk('public')->exists($customEmoji->file_path)) {
                Storage::disk('public')->delete($customEmoji->file_path);
            }

            // Delete the record
            $customEmoji->delete();

            return response()->json([
                'success' => true,
                'message' => ucfirst($customEmoji->type) . ' deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete emoji/GIF',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle active status
     */
    public function toggleStatus($id)
    {
        try {
            $customEmoji = CustomEmoji::forTenant($this->owner['id'])->findOrFail($id);
            $customEmoji->update(['is_active' => !$customEmoji->is_active]);

            return response()->json([
                'success' => true,
                'message' => ucfirst($customEmoji->type) . ' status updated successfully',
                'is_active' => $customEmoji->is_active
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all active emojis and GIFs for the current tenant
     */
    public function getActiveEmojis()
    {
        $emojis = CustomEmoji::forTenant($this->owner['id'])
            ->active()
            ->orderBy('type')
            ->orderBy('name')
            ->get();

        return response()->json([
            'success' => true,
            'emojis' => $emojis
        ]);
    }

    /**
     * Assign emoji/GIF to a user
     */
    public function assignToUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'emoji_id' => 'required|exists:custom_emojis,id',
            'message' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $assignment = \App\Models\UserEmojiAssignment::create([
                'user_id' => $request->user_id,
                'assigned_by' => auth()->id(),
                'emoji_id' => $request->emoji_id,
                'message' => $request->message,
                'tenant_id' => $this->owner['id']
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assigned successfully',
                'assignment' => $assignment->load(['user', 'emoji', 'assignedBy'])
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to assign emoji/GIF',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get emoji assignments for a user
     */
    public function getUserAssignments($userId)
    {
        try {
            $assignments = \App\Models\UserEmojiAssignment::forTenant($this->owner['id'])
                ->forUser($userId)
                ->with(['emoji', 'assignedBy'])
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'assignments' => $assignments
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get assignments',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}