<?php

namespace App\Http\Controllers;

use App\Models\Prize;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class PrizeController extends Controller
{
    /**
     * Display a listing of prizes for the current tenant
     */
    public function index(Request $request)
    {
        $tenantId = $this->getTenantId();
        $category = $request->get('category', 'all');
        $timePeriod = $request->get('time_period', 'current_month');

        $query = Prize::forTenant($tenantId)
            ->active()
            ->currentlyValid()
            ->orderBy('position')
            ->orderBy('created_at', 'desc');

        if ($category !== 'all') {
            $query->forCategory($category);
        }

        if ($timePeriod !== 'all') {
            $query->currentPeriod($timePeriod);
        }

        $prizes = $query->get();

        return response()->json([
            'success' => true,
            'data' => $prizes,
            'total' => $prizes->count()
        ]);
    }

    /**
     * Store a newly created prize
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'position' => 'required|integer|in:1,2,3',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'prize_amount' => 'nullable|numeric|min:0',
            'prize_type' => 'required|string|in:money,gift,recognition,bonus,voucher,trophy',
            'currency' => 'nullable|string|size:3',
            'category' => 'nullable|string',
            'time_period' => 'required|string|in:current_month,last_month,quarter,year',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $tenantId = $this->getTenantId();
            $userInfo = $this->getCurrentUserInfo();

            // Check if a prize already exists for this position, category, and time period
            $existingPrize = Prize::forTenant($tenantId)
                ->forPosition($request->position)
                ->forCategory($request->category)
                ->currentPeriod($request->time_period)
                ->active()
                ->first();

            if ($existingPrize) {
                return response()->json([
                    'success' => false,
                    'message' => 'A prize already exists for this position, category, and time period. Please edit the existing prize or choose different criteria.'
                ], 409);
            }

            $prize = Prize::create([
                'tenant_id' => $tenantId,
                'position' => $request->position,
                'title' => $request->title,
                'description' => $request->description,
                'prize_amount' => $request->prize_amount,
                'prize_type' => $request->prize_type,
                'currency' => $request->currency ?? 'USD',
                'category' => $request->category,
                'time_period' => $request->time_period,
                'start_date' => $request->start_date ? Carbon::parse($request->start_date) : null,
                'end_date' => $request->end_date ? Carbon::parse($request->end_date) : null,
                'created_by_type' => $userInfo['type'],
                'created_by_id' => $userInfo['id'],
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Prize created successfully',
                'data' => $prize
            ], 201);

        } catch (\Exception $e) {
            Log::error('Error creating prize: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create prize'
            ], 500);
        }
    }

    /**
     * Get current tenant ID from session
     */
    private function getTenantId()
    {
        // Try different session keys based on user type
        $sessions = ['owner', 'ismngr', 'isemp', 'isadmin'];
        
        foreach ($sessions as $sessionKey) {
            $user = session($sessionKey);
            if ($user && isset($user['tenant_id'])) {
                return $user['tenant_id'];
            }
            if ($user && isset($user['id'])) {
                return $user['id']; // For super admin
            }
        }
        
        throw new \Exception('Tenant ID not found in session');
    }

    /**
     * Display the specified prize
     */
    public function show($id)
    {
        $tenantId = $this->getTenantId();
        
        $prize = Prize::forTenant($tenantId)->find($id);
        
        if (!$prize) {
            return response()->json([
                'success' => false,
                'message' => 'Prize not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $prize
        ]);
    }

    /**
     * Update the specified prize
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'position' => 'sometimes|integer|in:1,2,3',
            'title' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'prize_amount' => 'nullable|numeric|min:0',
            'prize_type' => 'sometimes|string|in:money,gift,recognition,bonus,voucher,trophy',
            'currency' => 'nullable|string|size:3',
            'category' => 'nullable|string',
            'time_period' => 'sometimes|string|in:current_month,last_month,quarter,year',
            'is_active' => 'sometimes|boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $tenantId = $this->getTenantId();
            $prize = Prize::forTenant($tenantId)->find($id);

            if (!$prize) {
                return response()->json([
                    'success' => false,
                    'message' => 'Prize not found'
                ], 404);
            }

            $updateData = $request->only([
                'position', 'title', 'description', 'prize_amount', 
                'prize_type', 'currency', 'category', 'time_period', 'is_active'
            ]);

            if ($request->has('start_date')) {
                $updateData['start_date'] = $request->start_date ? Carbon::parse($request->start_date) : null;
            }

            if ($request->has('end_date')) {
                $updateData['end_date'] = $request->end_date ? Carbon::parse($request->end_date) : null;
            }

            $prize->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'Prize updated successfully',
                'data' => $prize->fresh()
            ]);

        } catch (\Exception $e) {
            Log::error('Error updating prize: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update prize'
            ], 500);
        }
    }

    /**
     * Remove the specified prize
     */
    public function destroy($id)
    {
        try {
            $tenantId = $this->getTenantId();
            $prize = Prize::forTenant($tenantId)->find($id);

            if (!$prize) {
                return response()->json([
                    'success' => false,
                    'message' => 'Prize not found'
                ], 404);
            }

            $prize->delete();

            return response()->json([
                'success' => true,
                'message' => 'Prize deleted successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Error deleting prize: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete prize'
            ], 500);
        }
    }

    /**
     * Get prizes for leaderboard display
     */
    public function getLeaderboardPrizes(Request $request)
    {
        $tenantId = $this->getTenantId();
        $category = $request->get('category', 'all');
        $timePeriod = $request->get('time_period', 'current_month');

        $query = Prize::forTenant($tenantId)
            ->active()
            ->currentlyValid()
            ->orderBy('position');

        if ($category !== 'all') {
            $query->forCategory($category);
        }

        if ($timePeriod !== 'all') {
            $query->currentPeriod($timePeriod);
        }

        $prizes = $query->get()->groupBy('position');

        return response()->json([
            'success' => true,
            'data' => $prizes
        ]);
    }

    /**
     * Get current user information
     */
    private function getCurrentUserInfo()
    {
        // Try different session keys based on user type
        $sessions = [
            'owner' => 'superadmin',
            'ismngr' => 'manager',
            'isemp' => 'employee',
            'isadmin' => 'admin'
        ];
        
        foreach ($sessions as $sessionKey => $type) {
            $user = session($sessionKey);
            if ($user) {
                return [
                    'type' => $type,
                    'id' => $user['id'] ?? $user['employee_id'] ?? null
                ];
            }
        }
        
        return [
            'type' => 'unknown',
            'id' => null
        ];
    }
}