<?php

namespace App\Http\Controllers\PointsAdmin;

use App\Http\Controllers\Controller;
use App\Models\Points;
use App\Models\Target;
use App\Models\Employee;
use App\Models\Admin; // Added Admin model
use App\Models\LeaderBoard;
use App\Services\UniversalNotificationService; // Changed from PointsNotificationService
use App\Services\PointsEmailNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PointsController extends Controller
{
    protected $admin;
    private UniversalNotificationService $notificationService; // Changed service
    private PointsEmailNotificationService $emailNotificationService;

    //H - B : Constructor
    public function __construct(
        UniversalNotificationService $notificationService,
        PointsEmailNotificationService $emailNotificationService
    ) {
        $this->notificationService = $notificationService;
        $this->emailNotificationService = $emailNotificationService;
        
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->admin = session('isadmin');
            return $next($request);
        });
    }

    // H-B : Add Points to Employee Directly
    public function pAdminPoints(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required',
            'vin_no' => 'required',
            'target_id' => 'required | array',
        ]);

        $data = [
            'employee_id'=>$request->employee_id,
            'department_id'=> $this->admin['department_id'],
            'category'=> $this->admin['category'],
            'vin_no'=>$request->vin_no,
            'status'=>'Redeemed',
            'detail'=>null,
        ];
        $data['tenant_id'] = $this->admin['tenant_id'];
        $targets = $request->target_id;
        $points = array_count_values($targets);

        try {
            $totalPoints = 0;
            $targetNames = [];
            $completedTargets = [];

            foreach($points as $target => $point)
            {
                $data['target_id'] = $target;
                $data['points'] = $point;
                $done[] = Points::create($data);
                $totalPoints += $point;
                
                $prog = Target::where('id', $target)
                    ->where('tenant_id', $this->admin['tenant_id'])
                    ->whereNot('status', '1')->first();
                
                if ($prog) {
                    $targetNames[] = $prog->name ?? "Target #{$target}";
                    
                    if(($prog->progress + $point) >= $prog->quantity)
                    {
                        $prog->update([
                            'progress' => $prog->quantity,
                            'status' => '0',
                            'completion_date' => now(),
                        ]);
                        $completedTargets[] = $prog->name ?? "Target #{$target}";
                    }
                    else
                    {
                        $prog->update(['progress'=> $prog->progress + $point]);
                    }
                }
            }

            if($done)
            {
                updateLeaderBoard($request->employee_id, $this->admin['department_id'],
                    $this->admin['category'], $this->admin['tenant_id']);

                // Send leaderboard update notifications to all employees
                $this->sendLeaderboardUpdateNotifications(
                    $this->admin['tenant_id'],
                    $this->admin['category'],
                    $this->admin['department_id']
                );

                // Send points awarded notification using UniversalNotificationService
                $this->notificationService->sendPointsNotification([
                    'tenant_id' => $this->admin['tenant_id'],
                    'receiver_type' => 'employee',
                    'receiver_id' => $request->employee_id,
                    'sender_type' => 'admin',
                    'sender_id' => $this->admin['employee_id'] ?? null,
                    'sender_name' => $this->admin['name'] ?? 'Points Admin',
                    'points_action' => 'awarded',
                    'content' => "Congratulations! You have been awarded {$totalPoints} points for completing: " . implode(', ', $targetNames),
                    'points_amount' => $totalPoints,
                    'target_name' => implode(', ', $targetNames),
                    'vin_no' => $request->vin_no,
                    'priority' => 'high'
                ]);

                // Send target completion notifications if any targets were completed
                foreach ($completedTargets as $completedTarget) {
                    $this->notificationService->sendPointsNotification([
                        'tenant_id' => $this->admin['tenant_id'],
                        'receiver_type' => 'employee',
                        'receiver_id' => $request->employee_id,
                        'sender_type' => 'admin',
                        'sender_id' => $this->admin['employee_id'] ?? null,
                        'sender_name' => $this->admin['name'] ?? 'Points Admin',
                        'points_action' => 'target_completed',
                        'content' => "🎯 Congratulations! You have completed the target: {$completedTarget}",
                        'target_name' => $completedTarget,
                        'priority' => 'high'
                    ]);
                }

                // Send email notification for points awarded by PointsAdmin
                $this->emailNotificationService->sendPointsRequestApprovedNotification(
                    $request->employee_id,
                    $this->admin['tenant_id'],
                    $totalPoints,
                    implode(', ', $targetNames),
                    $this->admin['employee_id']
                );

                return response()->json([
                    'Error' => false,
                    'Message' => 'Points Added Successfully!',
                ]);
            }
            else
            {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Error: Attempt Failed!',
                ]);
            }

        } catch(\Throwable $t) {
            Log::error('Points assignment failed', [
                'employee_id' => $request->employee_id,
                'admin_id' => $this->admin['employee_id'] ?? null,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    // H-B : Reject Points
    public function pAdminRejectPoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required',
            'details' => 'required | string | max:250'
        ]);
        
        try {
            $point = Points::find($request->id);

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Points record not found!',
                ]);
            }

            $point->update([
                'status' => 'Rejected',
                'detail' => $request->details,
            ]);

            // Send rejection notification using UniversalNotificationService
            $this->notificationService->sendPointsNotification([
                'tenant_id' => $point->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $point->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->admin['employee_id'] ?? null,
                'sender_name' => $this->admin['name'] ?? 'Points Admin',
                'points_action' => 'rejected',
                'content' => "Your points request has been rejected. Reason: {$request->details}",
                'points_amount' => $point->points,
                'vin_no' => $point->vin_no,
                'priority' => 'normal'
            ]);

            // Send email notification for rejection
            $target = Target::where('id', $point->target_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();
            $targetName = $target ? ($target->name ?? "Target #{$point->target_id}") : "Target #{$point->target_id}";
            
            $this->emailNotificationService->sendPointsRequestRejectedNotification(
                $point->employee_id,
                $this->admin['tenant_id'],
                $point->points,
                $targetName,
                $request->details,
                $this->admin['employee_id']
            );

            return response()->json([
                'Error' => false,
                'Message' => 'Operation Successful!',
            ]);

        } catch (\Throwable $t) {
            Log::error('Points rejection failed', [
                'point_id' => $request->id,
                'admin_id' => $this->admin['employee_id'] ?? null,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    // H-B : Approve Points
    public function pAdminApprovePoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required',
        ]);
        
        try {
            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->admin['tenant_id'])->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Points record not found!',
                ]);
            }

            // Get employee details
            $employee = Employee::find($point->employee_id);
            if (!$employee) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee not found!',
                ]);
            }

            // Update point record with missing department/category if needed
            $pointUpdateData = ['status' => 'Redeemed'];
            if (!$point->department_id || !$point->category) {
                $pointUpdateData['department_id'] = $employee->department_id ?? $this->admin['department_id'];
                $pointUpdateData['category'] = $employee->category ?? $this->admin['category'];
            }
            $point->update($pointUpdateData);

            $target = Target::where('id', $point->target_id)
                ->where('tenant_id', $this->admin['tenant_id'])->first();

            $targetCompleted = false;
            $targetName = $target ? ($target->name ?? "Target #{$point->target_id}") : "Target #{$point->target_id}";

            if ($target) {
                $quantity = $target->quantity;
                $progress = $target->progress;
                $progress += $point->points;

                if ($progress >= $quantity) {
                    $target->update([
                        'progress' => $target->quantity,
                        'status' => '0',
                        'completion_date' => now(),
                    ]);
                    $targetCompleted = true;
                } else {
                    $target->update([
                        'progress' => $progress,
                    ]);
                }
            }

            if ($point && $target) {
                // Use employee data for leaderboard update to ensure non-null values
                updateLeaderBoard(
                    $point->employee_id, 
                    $employee->department_id ?? $this->admin['department_id'],
                    $employee->category ?? $this->admin['category'], 
                    $point->tenant_id
                );

                // Send leaderboard update notifications to all employees
                $this->sendLeaderboardUpdateNotifications(
                    $point->tenant_id,
                    $employee->category ?? $this->admin['category'],
                    $employee->department_id ?? $this->admin['department_id']
                );

                // Send approval notification using UniversalNotificationService
                $this->notificationService->sendPointsNotification([
                    'tenant_id' => $point->tenant_id,
                    'receiver_type' => 'employee',
                    'receiver_id' => $point->employee_id,
                    'sender_type' => 'admin',
                    'sender_id' => $this->admin['employee_id'] ?? null,
                    'sender_name' => $this->admin['name'] ?? 'Points Admin',
                    'points_action' => 'approved',
                    'content' => "Great news! Your points request for {$point->points} points has been approved for {$targetName}",
                    'points_amount' => $point->points,
                    'target_name' => $targetName,
                    'vin_no' => $point->vin_no,
                    'priority' => 'high'
                ]);

                // Send target completion notification if target was completed
                if ($targetCompleted) {
                    $this->notificationService->sendPointsNotification([
                        'tenant_id' => $point->tenant_id,
                        'receiver_type' => 'employee',
                        'receiver_id' => $point->employee_id,
                        'sender_type' => 'admin',
                        'sender_id' => $this->admin['employee_id'] ?? null,
                        'sender_name' => $this->admin['name'] ?? 'Points Admin',
                        'points_action' => 'target_completed',
                        'content' => "🎯 Congratulations! You have completed the target: {$targetName}",
                        'target_name' => $targetName,
                        'priority' => 'high'
                    ]);
                }

                // Send email notification for approval
                $this->emailNotificationService->sendPointsRequestApprovedNotification(
                    $point->employee_id,
                    $this->admin['tenant_id'],
                    $point->points,
                    $targetName,
                    $this->admin['employee_id']
                );

                return response()->json([
                    'Error' => false,
                    'Message' => 'Operation Successful!',
                ]);
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Request Approval Failed!',
                ]);
            }

        } catch (\Throwable $t) {
            Log::error('Points approval failed', [
                'point_id' => $request->id,
                'admin_id' => $this->admin['employee_id'] ?? null,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    /**
     * Get point details for editing
     */
    public function getPoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer|exists:points,id',
        ]);

        try {
            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Point record not found!',
                ]);
            }

            return response()->json([
                'Error' => false,
                'Message' => 'Point details retrieved successfully!',
                'data' => [
                    'id' => $point->id,
                    'employee_id' => $point->employee_id,
                    'vin_no' => $point->vin_no,
                    'target_id' => $point->target_id,
                    'points' => $point->points,
                    'status' => $point->status,
                    'detail' => $point->detail,
                    'department_id' => $point->department_id,
                    'category' => $point->category,
                    'created_at' => $point->created_at,
                    'updated_at' => $point->updated_at,
                ]
            ]);

        } catch (\Throwable $t) {
            Log::error('Get points failed', [
                'point_id' => $request->id,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);
            
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    /**
     * Update points after editing
     */
    public function updatePoints(Request $request)
    {
        $validated = $request->validate([
            'point_id' => 'required|integer|exists:points,id',
            'employee_id' => 'required|integer|exists:employees,employee_id',
            'vin_no' => 'required|string|max:255',
            'target_id' => 'required|integer|exists:targets,id',
            'points' => 'required|integer|min:1',
            'status' => 'required|in:Pending,Redeemed,Rejected',
            'rejection_reason' => 'nullable|string|max:500',
        ]);

        try {
            DB::beginTransaction();

            $point = Points::where('id', $request->point_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Point record not found!',
                ]);
            }

            // Store original values for calculations
            $originalStatus = $point->status;
            $originalPoints = $point->points;
            $originalTargetId = $point->target_id;

            // Get employee details for department_id and category
            $employee = Employee::find($request->employee_id);
            if (!$employee) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee not found!',
                ]);
            }

            // Handle target progress changes
            $this->handleTargetProgressOnEdit(
                $originalTargetId,
                $request->target_id,
                $originalPoints,
                $request->points,
                $originalStatus,
                $request->status
            );

            // Update the point record
            $updateData = [
                'employee_id' => $request->employee_id,
                'vin_no' => $request->vin_no,
                'target_id' => $request->target_id,
                'points' => $request->points,
                'status' => $request->status,
                'department_id' => $employee->department_id ?? $this->admin['department_id'],
                'category' => $employee->category ?? $this->admin['category'],
                'detail' => $request->status === 'Rejected' ? $request->rejection_reason : null,
            ];

            $point->update($updateData);

            // Update leaderboard if needed
            if ($originalStatus !== $request->status || $originalPoints !== $request->points || $point->employee_id !== $request->employee_id) {
                // If employee changed, update both old and new employee leaderboards
                if ($point->employee_id !== $request->employee_id) {
                    // Update original employee leaderboard
                    updateLeaderBoard(
                        $point->employee_id,
                        $point->department_id,
                        $point->category,
                        $this->admin['tenant_id']
                    );
                }
                
                // Update new/current employee leaderboard
                updateLeaderBoard(
                    $request->employee_id,
                    $employee->department_id ?? $this->admin['department_id'],
                    $employee->category ?? $this->admin['category'],
                    $this->admin['tenant_id']
                );
            }

            // Send email notification for points update
            $target = Target::find($request->target_id);
            $targetName = $target ? ($target->name ?? "Target #{$request->target_id}") : "Target #{$request->target_id}";
            
            // Determine the appropriate notification based on status change
            if ($originalStatus !== $request->status) {
                if ($request->status === 'Redeemed') {
                    $this->emailNotificationService->sendPointsRequestApprovedNotification(
                        $request->employee_id,
                        $this->admin['tenant_id'],
                        $request->points,
                        $targetName,
                        $this->admin['employee_id']
                    );
                } elseif ($request->status === 'Rejected') {
                    $this->emailNotificationService->sendPointsRequestRejectedNotification(
                        $request->employee_id,
                        $this->admin['tenant_id'],
                        $request->points,
                        $targetName,
                        $request->rejection_reason ?? 'No reason provided',
                        $this->admin['employee_id']
                    );
                }
            } else {
                // Status didn't change, but points or other details were updated
                // Send a general update notification
                $this->emailNotificationService->sendPointsRequestApprovedNotification(
                    $request->employee_id,
                    $this->admin['tenant_id'],
                    $request->points,
                    $targetName,
                    $this->admin['employee_id']
                );
            }

            DB::commit();

            return response()->json([
                'Error' => false,
                'Message' => 'Point updated successfully!',
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            Log::error('Update points failed', [
                'point_id' => $request->point_id,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Update failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Reverse approved points back to pending
     */
    public function reversePoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer|exists:points,id',
        ]);

        try {
            DB::beginTransaction();

            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->where('status', 'Redeemed')
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Approved point record not found!',
                ]);
            }

            // Get employee details to ensure we have department_id and category
            $employee = Employee::find($point->employee_id);
            if (!$employee) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee not found!',
                ]);
            }

            // Update point record with missing department/category if needed
            $updatePointData = [
                'status' => 'Pending',
                'detail' => null,
            ];

            // Ensure department_id and category are not null
            if (!$point->department_id || !$point->category) {
                $updatePointData['department_id'] = $employee->department_id ?? $this->admin['department_id'];
                $updatePointData['category'] = $employee->category ?? $this->admin['category'];
            }

            $point->update($updatePointData);

            // Reverse target progress
            $target = Target::where('id', $point->target_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();

            if ($target) {
                $newProgress = max(0, $target->progress - $point->points);
                
                $updateData = ['progress' => $newProgress];
                
                // If target was completed, reactivate it
                if ($target->status === '0') {
                    $updateData['status'] = '2';
                    $updateData['completion_date'] = null;
                }
                
                $target->update($updateData);
            }

            // Use employee data for leaderboard update to ensure non-null values
            updateLeaderBoard(
                $point->employee_id,
                $employee->department_id ?? $this->admin['department_id'],
                $employee->category ?? $this->admin['category'],
                $this->admin['tenant_id']
            );

            // Send email notification for reversal
            $targetName = $target ? ($target->name ?? "Target #{$point->target_id}") : "Target #{$point->target_id}";
            $this->emailNotificationService->sendPointsRequestReversedNotification(
                $point->employee_id,
                $this->admin['tenant_id'],
                $point->points,
                $targetName,
                $this->admin['employee_id']
            );

            DB::commit();

            return response()->json([
                'Error' => false,
                'Message' => 'Points reversed successfully! Status changed to pending.',
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            Log::error('Reverse points failed', [
                'point_id' => $request->id,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Reverse failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Reprocess rejected points back to pending
     */
    public function reprocessPoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer|exists:points,id',
        ]);

        try {
            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->where('status', 'Rejected')
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Rejected point record not found!',
                ]);
            }

            // Update point status to pending
            $point->update([
                'status' => 'Pending',
                'detail' => null,
            ]);

            // Send email notification for reprocessing
            $target = Target::where('id', $point->target_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();
            $targetName = $target ? ($target->name ?? "Target #{$point->target_id}") : "Target #{$point->target_id}";
            
            $this->emailNotificationService->sendPointsRequestReprocessedNotification(
                $point->employee_id,
                $this->admin['tenant_id'],
                $point->points,
                $targetName,
                $this->admin['employee_id']
            );

            return response()->json([
                'Error' => false,
                'Message' => 'Points reprocessed successfully! Status changed to pending.',
            ]);

        } catch (\Throwable $t) {
            Log::error('Reprocess points failed', [
                'point_id' => $request->id,
                'error' => $t->getMessage(),
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Reprocess failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Handle target progress changes during point editing
     */
    private function handleTargetProgressOnEdit($originalTargetId, $newTargetId, $originalPoints, $newPoints, $originalStatus, $newStatus)
    {
        // If original status was approved, we need to reverse the progress first
        if ($originalStatus === 'Redeemed') {
            $originalTarget = Target::where('id', $originalTargetId)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();
                
            if ($originalTarget) {
                $newProgress = max(0, $originalTarget->progress - $originalPoints);
                
                $updateData = ['progress' => $newProgress];
                
                // If target was completed, check if it should be reactivated
                if ($originalTarget->status === '0' && $newProgress < $originalTarget->quantity) {
                    $updateData['status'] = '2';
                    $updateData['completion_date'] = null;
                }
                
                $originalTarget->update($updateData);
            }
        }

        // If new status is approved, add the new progress
        if ($newStatus === 'Redeemed') {
            $newTarget = Target::where('id', $newTargetId)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();
                
            if ($newTarget) {
                $newProgress = $newTarget->progress + $newPoints;
                
                $updateData = ['progress' => $newProgress];
                
                // Check if target is now completed
                if ($newProgress >= $newTarget->quantity) {
                    $updateData['progress'] = $newTarget->quantity;
                    $updateData['status'] = '0';
                    $updateData['completion_date'] = now();
                }
                
                $newTarget->update($updateData);
            }
        }
    }

    /**
     * Enhanced bulk approve points (works for both Pending and Rejected points)
     */
    public function bulkApprovePointsEnhanced(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'required|integer|exists:points,id',
        ]);

        try {
            $successCount = 0;
            $failedCount = 0;
            $completedTargets = [];
            $reprocessedCount = 0;

            DB::beginTransaction();

            foreach ($request->ids as $pointId) {
                try {
                    $point = Points::where('id', $pointId)
                        ->where('tenant_id', $this->admin['tenant_id'])
                        ->whereIn('status', ['Pending', 'Rejected'])
                        ->first();

                    if (!$point) {
                        $failedCount++;
                        continue;
                    }

                    // Get employee details
                    $employee = Employee::find($point->employee_id);
                    if (!$employee) {
                        $failedCount++;
                        continue;
                    }

                    // Update point record with missing department/category if needed
                    $pointUpdateData = ['status' => 'Redeemed', 'detail' => null];
                    if (!$point->department_id || !$point->category) {
                        $pointUpdateData['department_id'] = $employee->department_id ?? $this->admin['department_id'];
                        $pointUpdateData['category'] = $employee->category ?? $this->admin['category'];
                    }
                    
                    // Track if this was a rejected point being reprocessed
                    if ($point->status === 'Rejected') {
                        $reprocessedCount++;
                    }
                    
                    $point->update($pointUpdateData);

                    // Get target information
                    $target = Target::where('id', $point->target_id)
                        ->where('tenant_id', $this->admin['tenant_id'])
                        ->first();

                    if ($target) {
                        // Update target progress
                        $newProgress = $target->progress + $point->points;
                        
                        if ($newProgress >= $target->quantity) {
                            $target->update([
                                'progress' => $target->quantity,
                                'status' => '0',
                                'completion_date' => now(),
                            ]);
                            
                            $completedTargets[] = [
                                'name' => $target->name ?? "Target #{$point->target_id}",
                                'employee_id' => $point->employee_id
                            ];
                        } else {
                            $target->update([
                                'progress' => $newProgress,
                            ]);
                        }
                    }

                    // Update leaderboard using employee data
                    updateLeaderBoard(
                        $point->employee_id,
                        $employee->department_id ?? $this->admin['department_id'],
                        $employee->category ?? $this->admin['category'],
                        $this->admin['tenant_id']
                    );

                    $successCount++;

                } catch (\Exception $e) {
                    Log::error('Individual bulk approve failed', [
                        'point_id' => $pointId,
                        'error' => $e->getMessage()
                    ]);
                    $failedCount++;
                }
            }

            DB::commit();

            $message = "Bulk approval completed. {$successCount} points approved successfully";
            if ($failedCount > 0) {
                $message .= ", {$failedCount} failed";
            }
            if ($reprocessedCount > 0) {
                $message .= ". {$reprocessedCount} previously rejected points were reprocessed";
            }
            $message .= ".";

            return response()->json([
                'Error' => false,
                'Message' => $message,
                'Details' => [
                    'successful' => $successCount,
                    'failed' => $failedCount,
                    'completed_targets' => count($completedTargets),
                    'reprocessed' => $reprocessedCount
                ]
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            
            Log::error('Bulk approve points enhanced failed', [
                'admin_id' => $this->admin['employee_id'] ?? null,
                'tenant_id' => $this->admin['tenant_id'],
                'point_ids' => $request->ids,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Bulk approval failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Bulk approve points (original method for backward compatibility)
     */
    public function bulkApprovePoints(Request $request)
    {
        // For backward compatibility, call the enhanced version
        return $this->bulkApprovePointsEnhanced($request);
    }

    /**
     * Enhanced bulk reject points (works for both Pending and Approved points)
     */
    public function bulkRejectPoints(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'required|integer|exists:points,id',
            'details' => 'required|string|max:500',
        ]);

        try {
            $successCount = 0;
            $failedCount = 0;
            $reversedTargets = [];

            DB::beginTransaction();

            foreach ($request->ids as $pointId) {
                try {
                    $point = Points::where('id', $pointId)
                        ->where('tenant_id', $this->admin['tenant_id'])
                        ->whereIn('status', ['Pending', 'Redeemed'])
                        ->first();

                    if (!$point) {
                        $failedCount++;
                        continue;
                    }

                    // If point was approved, reverse the target progress
                    if ($point->status === 'Redeemed') {
                        $target = Target::where('id', $point->target_id)
                            ->where('tenant_id', $this->admin['tenant_id'])
                            ->first();

                        if ($target) {
                            $newProgress = max(0, $target->progress - $point->points);
                            
                            $updateData = ['progress' => $newProgress];
                            
                            // If target was completed, check if it should be reactivated
                            if ($target->status === '0' && $newProgress < $target->quantity) {
                                $updateData['status'] = '2';
                                $updateData['completion_date'] = null;
                                $reversedTargets[] = $point->target_id;
                            }
                            
                            $target->update($updateData);
                        }

                        // Update leaderboard to reflect the rejection
                        $employee = Employee::find($point->employee_id);
                        if ($employee) {
                            updateLeaderBoard(
                                $point->employee_id,
                                $employee->department_id ?? $this->admin['department_id'],
                                $employee->category ?? $this->admin['category'],
                                $this->admin['tenant_id']
                            );
                        }
                    }

                    // Update point status
                    $point->update([
                        'status' => 'Rejected',
                        'detail' => $request->details,
                    ]);

                    $successCount++;

                } catch (\Exception $e) {
                    Log::error('Individual bulk reject failed', [
                        'point_id' => $pointId,
                        'error' => $e->getMessage()
                    ]);
                    $failedCount++;
                }
            }

            DB::commit();

            $message = "Bulk rejection completed. {$successCount} points rejected successfully";
            if ($failedCount > 0) {
                $message .= ", {$failedCount} failed";
            }
            if (count($reversedTargets) > 0) {
                $message .= ". " . count($reversedTargets) . " completed targets were reactivated.";
            }
            $message .= ".";

            return response()->json([
                'Error' => false,
                'Message' => $message,
                'Details' => [
                    'successful' => $successCount,
                    'failed' => $failedCount,
                    'reversed_targets' => count($reversedTargets)
                ]
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            
            Log::error('Bulk reject points failed', [
                'admin_id' => $this->admin['employee_id'] ?? null,
                'tenant_id' => $this->admin['tenant_id'],
                'point_ids' => $request->ids,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Bulk rejection failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Get bulk operation statistics
     */
    public function getBulkOperationStats(Request $request)
    {
        try {
            $tenantId = $this->admin['tenant_id'];
            $category = $this->admin['category'];
            $departmentId = $this->admin['department_id'];

            // Get pending points count
            $pendingCount = Points::where('tenant_id', $tenantId)
                                ->where('category', $category)
                                ->where('department_id', $departmentId)
                                ->where('status', 'Pending')
                                ->count();

            // Get recent bulk operations (last 30 days)
            $recentBulkOps = Points::where('tenant_id', $tenantId)
                                 ->where('category', $category)
                                 ->where('department_id', $departmentId)
                                 ->where('updated_at', '>=', now()->subDays(30))
                                 ->whereIn('status', ['Redeemed', 'Rejected'])
                                 ->selectRaw('status, COUNT(*) as count, DATE(updated_at) as date')
                                 ->groupBy('status', 'date')
                                 ->orderBy('date', 'desc')
                                 ->get();

            return response()->json([
                'Error' => false,
                'Data' => [
                    'pending_count' => $pendingCount,
                    'recent_operations' => $recentBulkOps,
                    'can_bulk_approve' => $pendingCount > 0,
                    'can_bulk_reject' => $pendingCount > 0
                ]
            ]);

        } catch (\Throwable $t) {
            Log::error('Failed to get bulk operation stats', [
                'admin_id' => $this->admin['employee_id'] ?? null,
                'tenant_id' => $this->admin['tenant_id'],
                'error' => $t->getMessage()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Failed to get bulk operation statistics',
            ]);
        }
    }

    /**
     * Send leaderboard update notifications to ALL users in tenant
     */
    private function sendLeaderboardUpdateNotifications(string $tenantId, string $category, int $departmentId = null)
    {
        try {
            Log::info('Sending leaderboard update notifications', [
                'tenant_id' => $tenantId,
                'category' => $category,
                'department_id' => $departmentId
            ]);

            // Get current top 3 performers for the category
            $currentTop3 = $this->getTopPerformers($tenantId, $category, $departmentId);
            
            if (empty($currentTop3)) {
                Log::info('No top performers found for leaderboard notification');
                return;
            }

            // Check if top 3 has changed significantly
            $shouldNotify = $this->shouldSendLeaderboardNotification($tenantId, $category, $currentTop3);
            
            if (!$shouldNotify) {
                Log::info('Top 3 leaderboard unchanged, skipping notification');
                return;
            }

            // Store current top 3 for future comparison
            $this->storeCurrentTop3($tenantId, $category, $currentTop3);

            // Create leaderboard content for the specific category
            $categoryLeaderboardContent = $this->formatLeaderboardContent($currentTop3, $category);
            
            // Get overall tenant leaderboard (all categories combined)
            $overallTop3 = $this->getOverallTenantTop3($tenantId);
            $overallLeaderboardContent = $this->formatOverallLeaderboardContent($overallTop3);

            // Send to ALL employees in the tenant (regardless of category)
            $employeeSentCount = $this->sendLeaderboardToAllEmployees(
                $tenantId, 
                $category, 
                $categoryLeaderboardContent, 
                $overallLeaderboardContent
            );

            // Send to ALL admins in the tenant
            $adminSentCount = $this->sendLeaderboardToAllAdmins(
                $tenantId, 
                $category, 
                $categoryLeaderboardContent, 
                $overallLeaderboardContent
            );

            Log::info('Leaderboard notifications sent successfully', [
                'tenant_id' => $tenantId,
                'category' => $category,
                'employee_notifications_sent' => $employeeSentCount,
                'admin_notifications_sent' => $adminSentCount,
                'total_sent' => $employeeSentCount + $adminSentCount
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send leaderboard update notifications', [
                'tenant_id' => $tenantId,
                'category' => $category,
                'department_id' => $departmentId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

    // Additional helper methods for leaderboard notifications (same as in original file)
    private function shouldSendLeaderboardNotification(string $tenantId, string $category, array $currentTop3): bool
    {
        try {
            // Get previous top 3 from cache/database
            $cacheKey = "leaderboard_top3_{$tenantId}_{$category}";
            $previousTop3 = cache($cacheKey, []);
            
            // If no previous data, send notification
            if (empty($previousTop3)) {
                return true;
            }
            
            // Check if top 3 positions or scores changed
            for ($i = 0; $i < min(3, count($currentTop3)); $i++) {
                $current = $currentTop3[$i] ?? null;
                $previous = $previousTop3[$i] ?? null;
                
                if (!$current || !$previous) {
                    return true; // New person in top 3
                }
                
                // Check if employee or points changed
                if ($current->employee_id != $previous['employee_id'] || 
                    $current->total_points != $previous['total_points']) {
                    return true;
                }
            }
            
            return false; // No significant changes
            
        } catch (\Exception $e) {
            Log::error('Error checking leaderboard changes', [
                'error' => $e->getMessage()
            ]);
            return true; // Send notification on error to be safe
        }
    }

    private function storeCurrentTop3(string $tenantId, string $category, array $currentTop3): void
    {
        try {
            $cacheKey = "leaderboard_top3_{$tenantId}_{$category}";
            $top3Data = [];
            
            foreach ($currentTop3 as $performer) {
                $top3Data[] = [
                    'employee_id' => $performer->employee_id,
                    'name' => $performer->name,
                    'emoji' => $performer->emoji,
                    'total_points' => $performer->total_points
                ];
            }
            
            // Cache for 24 hours
            cache([$cacheKey => $top3Data], now()->addHours(24));
            
        } catch (\Exception $e) {
            Log::error('Error storing top 3 leaderboard', [
                'error' => $e->getMessage()
            ]);
        }
    }

    private function sendLeaderboardToAllEmployees(string $tenantId, string $category, string $categoryContent, string $overallContent): int
    {
        try {
            // Get ALL employees in tenant (regardless of category)
            $allEmployees = Employee::where('tenant_id', $tenantId)
                                  ->select('employee_id', 'name', 'emoji', 'category', 'department_id')
                                  ->get();

            $sentCount = 0;
            foreach ($allEmployees as $employee) {
                try {
                    // Determine if this is their category or not
                    $isTheirCategory = $employee->category === $category;
                    $title = $isTheirCategory 
                        ? "🏆 {$category} Leaderboard Updated!" 
                        : "🏆 Leaderboard Update: {$category} Category";
                    
                    $content = $isTheirCategory 
                        ? "Your category leaderboard has been updated!\n\n{$categoryContent}\n\n📊 Overall Tenant Rankings:\n{$overallContent}"
                        : "{$category} category leaderboard updated!\n\n{$categoryContent}\n\n📊 Overall Tenant Rankings:\n{$overallContent}";

                    $sent = $this->notificationService->sendToUser([
                        'tenant_id' => $tenantId,
                        'receiver_type' => 'employee',
                        'receiver_id' => $employee->employee_id,
                        'sender_type' => 'system',
                        'sender_id' => null,
                        'sender_name' => 'Leaderboard System',
                        'title' => $title,
                        'content' => $content,
                        'type' => 'leaderboard',
                        'priority' => $isTheirCategory ? 'high' : 'normal',
                        'related_id' => null
                    ]);

                    if ($sent) {
                        $sentCount++;
                    }

                } catch (\Exception $e) {
                    Log::error('Failed to send leaderboard notification to employee', [
                        'employee_id' => $employee->employee_id,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            return $sentCount;

        } catch (\Exception $e) {
            Log::error('Failed to send leaderboard notifications to employees', [
                'error' => $e->getMessage()
            ]);
            return 0;
        }
    }

    private function sendLeaderboardToAllAdmins(string $tenantId, string $category, string $categoryContent, string $overallContent): int
    {
        try {
            // Get ALL admins in tenant
            $allAdmins = Admin::where('tenant_id', $tenantId)
                             ->select('id', 'name', 'role', 'category')
                             ->get();

            $sentCount = 0;
            foreach ($allAdmins as $admin) {
                try {
                    $sent = $this->notificationService->sendToUser([
                        'tenant_id' => $tenantId,
                        'receiver_type' => 'admin',
                        'receiver_id' => $admin->id,
                        'sender_type' => 'system',
                        'sender_id' => null,
                        'sender_name' => 'Leaderboard System',
                        'title' => "🏆 Admin Alert: {$category} Leaderboard Updated",
                        'content' => "The {$category} leaderboard has been updated with new rankings!\n\n{$categoryContent}\n\n📊 Overall Tenant Performance:\n{$overallContent}",
                        'type' => 'leaderboard',
                        'priority' => 'normal',
                        'related_id' => null
                    ]);

                    if ($sent) {
                        $sentCount++;
                    }

                } catch (\Exception $e) {
                    Log::error('Failed to send leaderboard notification to admin', [
                        'admin_id' => $admin->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            return $sentCount;

        } catch (\Exception $e) {
            Log::error('Failed to send leaderboard notifications to admins', [
                'error' => $e->getMessage()
            ]);
            return 0;
        }
    }

    private function getOverallTenantTop3(string $tenantId): array
    {
        try {
            $overallTop3 = DB::table('leader_boards as lb')
                ->join('employees as e', 'lb.employee_id', '=', 'e.employee_id')
                ->where('lb.tenant_id', $tenantId)
                ->select(
                    'e.name',
                    'e.emoji',
                    'e.category',
                    'lb.total_points',
                    'lb.employee_id'
                )
                ->orderBy('lb.total_points', 'desc')
                ->limit(3)
                ->get()
                ->toArray();

            return $overallTop3;

        } catch (\Exception $e) {
            Log::error('Failed to get overall tenant top 3', [
                'tenant_id' => $tenantId,
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    private function formatOverallLeaderboardContent(array $overallTop3): string
    {
        if (empty($overallTop3)) {
            return "No overall rankings available yet.";
        }

        $content = "";
        $medals = ['🥇', '🥈', '🥉'];
        
        foreach ($overallTop3 as $index => $performer) {
            $medal = $medals[$index] ?? '🏅';
            $emoji = $performer->emoji ?? '👤';
            $category = $performer->category ?? 'Unknown';
            $content .= "{$medal} {$performer->name} {$emoji} ({$category}) - {$performer->total_points} pts\n";
        }
        
        return $content;
    }

    private function getTopPerformers(string $tenantId, string $category, int $departmentId = null): array
    {
        try {
            // Query to get top performers - adjust based on your leaderboard table structure
            $query = DB::table('leader_boards as lb')
                ->join('employees as e', 'lb.employee_id', '=', 'e.employee_id')
                ->where('lb.tenant_id', $tenantId)
                ->where('lb.category', $category);

            if ($departmentId) {
                $query->where('lb.department_id', $departmentId);
            }

            $topPerformers = $query->select(
                    'e.name',
                    'e.emoji',
                    'lb.total_points',
                    'lb.employee_id',
                    DB::raw('ROW_NUMBER() OVER (ORDER BY lb.total_points DESC) as rank')
                )
                ->orderBy('lb.total_points', 'desc')
                ->limit(3)
                ->get()
                ->toArray();

            return $topPerformers;

        } catch (\Exception $e) {
            Log::error('Failed to get top performers', [
                'tenant_id' => $tenantId,
                'category' => $category,
                'error' => $e->getMessage()
            ]);
            return [];
        }
    }

    private function formatLeaderboardContent(array $topPerformers, string $category): string
    {
        if (empty($topPerformers)) {
            return "No rankings available for {$category} category yet.";
        }

        $content = "";
        $medals = ['🥇', '🥈', '🥉'];
        
        foreach ($topPerformers as $index => $performer) {
            $medal = $medals[$index] ?? '🏅';
            $emoji = $performer->emoji ?? '👤';
            $content .= "{$medal} {$performer->name} {$emoji} - {$performer->total_points} points\n";
        }
        
        return $content;
    }
}