<?php

namespace App\Http\Controllers\PointsAdmin;

use App\Http\Controllers\Controller;
use App\Models\CategoryLeaderBoard;
use App\Models\DepartmentLeaderBoard;
use App\Models\Employee;
use App\Models\EmployeeLeaderBoard;
use App\Models\Points;
use App\Models\Product;
use App\Models\Target;
use App\Services\LeaderboardService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{
    protected $admin;
    protected $leaderboardService;

    //H-B : Constructor
    public function __construct()
    {
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->admin = session('isadmin');
            if ($this->admin) {
                $this->leaderboardService = new LeaderboardService($this->admin['tenant_id']);
            }
            return $next($request);
        });
    }

    //H-B : Points Admin Dashboard
    public function pAdminDashboard()
    {
        $targets = Target::leftjoin('products', 'targets.product_id', '=', 'products.id')
            ->where('targets.department_id', '=', $this->admin['department_id'])
            ->whereNot('targets.status', '1')
            ->where('targets.category', '=', $this->admin['category'])
            ->get([
                'targets.*',
                'products.name as name'
            ]);

        $employees = Points::join('employees', 'points.employee_id', '=', 'employees.employee_id')
            ->where('points.department_id', '=', $this->admin['department_id'])
            ->where('points.category', '=', $this->admin['category'])
            ->select(
                'employees.name as name',
                'employees.employee_id as id',
                DB::raw('SUM(CASE WHEN points.status = "redeemed" THEN points.points ELSE 0 END) as redeemed_points'),
                DB::raw('SUM(CASE WHEN points.status = "pending" THEN points.points ELSE 0 END) as pending_points'),
                DB::raw('MAX(CASE WHEN points.status = "redeemed" THEN points.updated_at ELSE NULL END) as date')
            )
            ->groupBy('employees.employee_id', 'employees.name')
            ->get();

        $currentDate = now()->format('F d, Y');
        $admin = $this->admin;
        
        // Get assigned custom emojis/GIFs for this points admin
        $assignedEmojis = $this->getAssignedEmojis($this->admin['employee_id']);
        
        return view('PointsAdmin.Dashboard', compact('admin', 'targets', 'currentDate', 'employees', 'assignedEmojis'));
    }

    private function getAssignedEmojis($adminId)
    {
        // Get assigned emojis/GIFs for this points admin directly
        return \App\Models\EmojiAssignment::with('emoji')
            ->where('assignable_id', $adminId)
            ->where('assignable_type', 'App\Models\Admin')
            ->where('tenant_id', $this->admin['tenant_id'])
            ->orderBy('created_at', 'desc')
            ->get();
    }

    //H-B : Points-Admin Claim Points Page
    public function pAdminPoints()
    {
        $admin = $this->admin;
        $products = Product::where('tenant_id', $this->admin['tenant_id'])->get();

        $targets = Target::where('department_id', $this->admin['department_id'])
            ->where('category', $this->admin['category'])
            ->where('tenant_id', $this->admin['tenant_id'])->get();

        $points = Points::select(
            'points.*',
            'employees.name as e_name',
            'targets.product_id as prod',
            'targets.custom as custom'
        )
            ->join('employees', 'points.employee_id', '=', 'employees.employee_id') // Join with employees table
            ->leftJoin('targets', 'points.target_id', '=', 'targets.id')  // Left join to include nullable fields
            ->where('points.department_id', $this->admin['department_id'])
            ->where('points.category', $this->admin['category'])
            ->where('points.tenant_id', $this->admin['tenant_id'])
            ->get();

        // dd($points->toArray());

        $employees = Employee::where('department_id', $this->admin['department_id'])
            ->where('tenant_id', $this->admin['tenant_id'])
            ->where('role', 'Employee')
            ->get(['name', 'status', 'employee_id', 'category']);


        return view('PointsAdmin.RequestedPoints', compact('admin', 'employees', 'points', 'targets', 'products'));
    }

    // H-B : LeaderBoard
    public function viewLeaderBoard()
    {
        $currentMonth = Carbon::now()->format('Y-m');

        $months = EmployeeLeaderBoard::select('month_year')
            ->distinct()
            ->orderBy('month_year', 'desc')
            ->pluck('month_year');

        $employees = EmployeeLeaderBoard::join('employees', function ($join) {
            $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id'); // Match tenant_id
        })
            ->where('employee_leader_boards.tenant_id', '=', $this->admin['tenant_id']) // Ensure tenant_id matches in employee_leader_boards
            ->where('month_year', $currentMonth)
            ->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee',
                'employees.category as category',
                'employees.image as image'
            ]);

        $departments = DepartmentLeaderBoard::join('departments', function ($join) {
            $join->on('department_leader_boards.department_id', '=', 'departments.id')
                ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id'); // Match tenant_id
        })
            ->join('employees', function ($join) {
                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
                    ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id') // Match tenant_id
                    ->where('employees.role', 'Manager');
            })
            ->where('department_leader_boards.tenant_id', '=', $this->admin['tenant_id']) // Match tenant_id
            ->where('month_year', $currentMonth)
            ->groupBy('department_leader_boards.id', 'department_leader_boards.rank', 'department_leader_boards.earned_points', 'departments.name') // Group by leaderboard ID to avoid duplicates
            ->orderBy('department_leader_boards.rank', 'asc')
            ->get([
                'department_leader_boards.id',
                'department_leader_boards.rank',
                'department_leader_boards.earned_points',
                'departments.name as department',
                \DB::raw('COUNT(DISTINCT employees.employee_id) as count'), // Count unique managers
                \DB::raw('MIN(employees.name) as manager'), // Get any one manager’s name
                \DB::raw('MIN(employees.image) as emp_image') // Get any one manager’s image
            ]);

        $categories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('category_leader_board.*', 'departments.name as name')
            ->where('month_year', $currentMonth)->get();

        // Get available categories for advanced filtering
        $availableCategories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('departments.name')
            ->where('category_leader_board.tenant_id', $this->admin['tenant_id'])
            ->distinct()
            ->pluck('departments.name')
            ->toArray();

        return view('PointsAdmin.Leaderboard', compact('employees', 'categories', 'departments', 'months', 'availableCategories'));
    }

    // H-B: Employee Filter LeaderBoard
    public function viewEmpLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"
        
        // Check if category filter is applied
        $category = request()->get('category');
        
        $query = EmployeeLeaderBoard::join('employees', function ($join) {
            $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id'); // Match tenant_id
        })
            ->where('employee_leader_boards.tenant_id', '=', $this->admin['tenant_id']) // Ensure tenant_id matches in employee_leader_boards
            ->where('month_year', $currentMonth);
            
        // Apply category filter if specified
        if ($category && $category !== 'all') {
            $query->where('employees.category', '=', $category);
        }
        
        $employees = $query->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee',
                'employees.category as category',
                'employees.image as image'
            ]);
        //        dd($employees);
        return view('partials.employeeLeaderBoard', compact('employees'))->render();
    }

    // H-B: Employee Filter LeaderBoard
    public function viewDepLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"
        $departments = DepartmentLeaderBoard::join('departments', function ($join) {
            $join->on('department_leader_boards.department_id', '=', 'departments.id')
                ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id'); // Match tenant_id
        })
            ->join('employees', function ($join) {
                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
                    ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id') // Match tenant_id
                    ->where('employees.role', 'Manager');
            })
            ->where('department_leader_boards.tenant_id', '=', $this->admin['tenant_id']) // Match tenant_id
            ->where('month_year', $currentMonth)
            ->groupBy('department_leader_boards.id', 'department_leader_boards.rank', 'department_leader_boards.earned_points', 'departments.name') // Group by leaderboard ID to avoid duplicates
            ->orderBy('department_leader_boards.rank', 'asc')
            ->get([
                'department_leader_boards.id',
                'department_leader_boards.rank',
                'department_leader_boards.earned_points',
                'departments.name as department',
                \DB::raw('COUNT(DISTINCT employees.employee_id) as count'), // Count unique managers
                \DB::raw('MIN(employees.name) as manager'), // Get any one manager’s name
                \DB::raw('MIN(employees.image) as emp_image') // Get any one manager’s image
            ]);
        //        dd($departments);
        return view('partials.departmentLeaderBoard', compact('departments'))->render();
    }

    // H-B: Employee Filter LeaderBoard
    public function viewCatLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"
        $categories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('category_leader_board.*', 'departments.name as name')
            ->where('category_leader_board.tenant_id', '=', $this->admin['tenant_id'])
            ->where('month_year', $currentMonth)
            ->get();
        //        dd($categories);
        return view('partials.categoryLeaderBoard', compact('categories'))->render();
    }

    // Enhanced Leaderboard Methods
    
    /**
     * Enhanced leaderboard view with all filters
     */
    public function viewEnhancedLeaderboard()
    {
        $currentMonth = Carbon::now()->format('Y-m');
        
        // Get available months
        $months = $this->leaderboardService->getAvailableMonths();
        
        // Get filter options
        $departments = $this->leaderboardService->getAvailableDepartments();
        $categories = $this->leaderboardService->getAvailableCategories();
        
        // Get initial data
        $leaderboardData = $this->leaderboardService->getLeaderboardData([
            'category' => 'employees',
            'time_period' => 'current_month'
        ]);
        
        return view('PointsAdmin.EnhancedLeaderboard', compact(
            'leaderboardData', 
            'months', 
            'departments', 
            'categories'
        ));
    }
    
    /**
     * AJAX endpoint for filtered leaderboard data
     */
    public function getFilteredLeaderboard(Request $request)
    {
        $filters = $request->only(['category', 'time_period', 'department_id', 'custom_date_range', 'specific_category', 'month']);
        
        try {
            // If specific_category is true, filter by the specific category
            if ($request->get('specific_category') && $request->get('category') !== 'all') {
                $data = $this->leaderboardService->getLeaderboardDataByCategory($filters);
            } else {
                $data = $this->leaderboardService->getLeaderboardData($filters);
            }
            
            $prizes = $this->leaderboardService->getPrizesForLeaderboard(
                $filters['category'] ?? 'all',
                $filters['time_period'] ?? 'current_month'
            );
            
            return response()->json([
                'success' => true,
                'data' => $data,
                'top3' => $data->take(3),
                'complete_ranking' => $data->skip(3),
                'total_count' => $data->count(),
                'prizes' => $prizes
            ]);
        } catch (\Exception $e) {
            Log::error('Error in PointsAdmin getFilteredLeaderboard: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load leaderboard data'
            ], 500);
        }
    }
    
    /**
     * Get BDC leaderboard specifically
     */
    public function getBDCLeaderboard(Request $request)
    {
        $timePeriod = $request->get('time_period', 'current_month');
        $data = $this->leaderboardService->getBDCLeaderboard($timePeriod);
        
        return view('partials.enhancedLeaderboard', [
            'data' => $data,
            'category' => 'BDC',
            'top3' => $data->take(3),
            'complete_ranking' => $data->skip(3)
        ])->render();
    }
    
    /**
     * Get Service Advisor leaderboard specifically
     */
    public function getServiceAdvisorLeaderboard(Request $request)
    {
        $timePeriod = $request->get('time_period', 'current_month');
        $data = $this->leaderboardService->getServiceAdvisorLeaderboard($timePeriod);
        
        return view('partials.enhancedLeaderboard', [
            'data' => $data,
            'category' => 'Service Advisor',
            'top3' => $data->take(3),
            'complete_ranking' => $data->skip(3)
        ])->render();
    }

}
