<?php

namespace App\Http\Controllers\PointsAdmin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Department;
use App\Models\Employee;
use App\Services\UniversalNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class EmployeeController extends Controller
{
    protected $admin;
    private UniversalNotificationService $notificationService;

    //H-B : Constructor
    public function __construct(UniversalNotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->admin = session('isadmin');
            return $next($request);
        });
    }

    // H - B : View Listing page
    public function listing()
    {
        $departments = Department::where('tenant_id', $this->admin['tenant_id'])->get();
        $employees = Employee::with('department')
            ->where('tenant_id', $this->admin['tenant_id'])
            ->get();

        // Get available emojis/GIFs created by superadmin
        $availableEmojis = \App\Models\CustomEmoji::forTenant($this->admin['tenant_id'])
            ->active()
            ->orderBy('type')
            ->orderBy('name')
            ->get();

        return view ('PointsAdmin.employee.listing', compact('employees', 'departments', 'availableEmojis'));
    }

    // H - B : Add And Edit Employee
    public function addAndEditEmployee(Request $request)
    {
        $employee = Employee::where('tenant_id', $this->admin['tenant_id'])
            ->where('employee_id', $request->employee_id)->first();

        $validationRules = [
            'employee_id' => 'required',
            'name' => 'required',
            'email' => 'required',
            'role' => 'required',
            'emoji' => 'nullable|string|max:10',
            'emoji_id' => 'nullable|integer', // Added custom emoji validation
            'emoji_message' => 'nullable|string|max:500' // Added emoji message validation
        ];
        
        if(!$employee) {
            $validationRules['password'] = 'required';
        }
        
        if($request->role === 'Manager') {
            $validationRules['department'] = 'required | integer';
        } elseif ($request->role === 'Admin') {
            $validationRules['department'] = 'required | integer ';
            $validationRules['category'] = 'required';
        } else {
            $validationRules['department'] = 'nullable';
        }
        
        $customMessages = [
            'department.required' => 'The department field is required.',
            'department.integer' => 'The department must have a valid value.',
            'emoji.max' => 'The emoji field must not exceed 10 characters.',
        ];
        
        $validate = $request->validate($validationRules, $customMessages);
        $image = null;
        $pas = null;
        
        // Store original emoji for comparison (for notifications)
        $originalEmoji = $employee ? $employee->emoji : null;
        $isNewEmployee = !$employee;

        if($employee) {
            if(isset($request->password)) {
                $pas = Hash::make($request->password);
            } else {
                $pas = $employee->password;
            }

            if(isset($request->image)) {
                // FIX: Add proper validation before deleting old image
                if (!empty($employee->image) && $employee->image !== '/' && file_exists(public_path($employee->image))) {
                    unlink(public_path($employee->image));
                }
                $image = saveFiles($request->image, 'Employees');
            } else {
                $image = $employee->image;
            }
        } else {
            $pas = Hash::make($request->password);
            if(isset($request->image)) {
                $image = saveFiles($request->image, 'Employees');
            } else {
                $image = null;
            }
        }

        $data = $request->only([
            'employee_id', 'name', 'email',
            'status', 'dob', 'phn', 'emoji', 'gender',
            'marital', 'nationality', 'city',
            'state', 'zip', 'address', 'role',
        ]);

        $data['password'] = $pas;
        $data['image'] = $image;
        $data['department_id'] = $request->department;
        $data['category'] = $request->category;
        $data['tenant_id'] = $this->admin['tenant_id'];
        
        // Handle emoji - if no emoji provided, generate one based on name
        $emojiWasAutoGenerated = false;
        if (empty($data['emoji'])) {
            $data['emoji'] = $this->generateEmojiFromName($data['name']);
            $emojiWasAutoGenerated = true;
        }

        try {
            if($employee) {
                $old_dep = $employee->department_id;
                $empCount = Department::where('id', $old_dep)->where('tenant_id', $this->admin['tenant_id'])->first();
                if($empCount) {
                    $count = $empCount->no_of_emp - 1;
                    $empCount->update(['no_of_emp' => $count]);
                }
                
                $employee->update($data);
                
                if($request->department != null) {
                    $count = Employee::where('tenant_id', $this->admin['tenant_id'])
                        ->where('department_id', $request->department)->count();
                    $test = Department::where('id', $request->department)->update(['no_of_emp' => $count]);
                }

                // Send emoji change notification if emoji changed
                if ($originalEmoji !== $data['emoji']) {
                    Log::info('Emoji changed by points admin, sending notification', [
                        'employee_id' => $employee->employee_id,
                        'old_emoji' => $originalEmoji,
                        'new_emoji' => $data['emoji'],
                        'tenant_id' => $employee->tenant_id,
                        'points_admin_id' => $this->admin['employee_id']
                    ]);
                    
                    // Send emoji change notification
                    $this->sendEmojiChangeNotification($employee, $originalEmoji, $data['emoji'], $emojiWasAutoGenerated);
                }

                return response()->json([
                    'Error' => false,
                    'Message' => 'Employee Updated successfully!',
                    'NotificationSent' => ($originalEmoji !== $data['emoji']) ? 'Emoji change notification sent' : 'No emoji change',
                ]);
            } else {
                $newEmployee = Employee::create($data);

                // Handle custom emoji assignment if provided
                if ($request->has('emoji_id') && $request->emoji_id) {
                    $this->assignEmojiToEmployee($newEmployee, $request->emoji_id, $request->emoji_message);
                }
                
                Log::info('New employee created by points admin, sending welcome notification', [
                    'employee_id' => $newEmployee->employee_id,
                    'emoji' => $data['emoji'],
                    'tenant_id' => $newEmployee->tenant_id,
                    'points_admin_id' => $this->admin['employee_id']
                ]);
                
                // Send welcome notification with emoji for new employee
                $this->sendWelcomeNotification($newEmployee, $data['emoji'], $emojiWasAutoGenerated);

                return response()->json([
                    'Error' => false,
                    'Message' => 'Employee Added successfully!',
                    'NotificationSent' => 'Welcome notification sent with emoji: ' . $data['emoji'],
                ]);
            }

        } catch(\Throwable $t) {
            Log::error('Points admin employee operation failed', [
                'error' => $t->getMessage(),
                'line' => $t->getLine(),
                'points_admin_id' => $this->admin['employee_id'],
                'trace' => $t->getTraceAsString()
            ]);
            
            return response()->json([
                'Error' => true,
                'Message' => "line no : ".$t->getLine(). ", Msg : ".$t->getMessage(),
            ]);
        }
    }

    private function generateEmojiFromName($name)
    {
        $emojis = ['😊', '😎', '🤓', '😄', '🙂', '😇', '🤗', '😃', '😁', '🤩', '🥳', '😌', '🤠', '👨‍💻', '👩‍💻', '👨‍💼', '👩‍💼'];
        
        // Use the first character of the name to determine emoji
        $nameLength = strlen($name);
        $index = $nameLength % count($emojis);
        
        return $emojis[$index];
    }

    // Send welcome notification for new employee
    private function sendWelcomeNotification($employee, $emoji, $wasAutoGenerated)
    {
        try {
            $emojiSource = $wasAutoGenerated ? 'automatically assigned' : 'set';
            $content = $wasAutoGenerated 
                ? "Welcome to the team! The Points Admin has assigned you a special emoji {$emoji} based on your name. This emoji represents you in our points system!"
                : "Welcome to the team! Your profile emoji {$emoji} has been set by the Points Admin. This emoji represents you in our points system!";

            // Send welcome notification
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->admin['employee_id'],
                'sender_name' => $this->admin['name'] ?? 'Points Admin',
                'title' => "Welcome! Your emoji is ready {$emoji}",
                'content' => $content,
                'type' => 'welcome_emoji',
                'priority' => 'high',
                'related_id' => $employee->employee_id
            ]);

            // Send celebratory emoji notification with points theme
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->admin['employee_id'],
                'sender_name' => $this->admin['name'] ?? 'Points Admin',
                'emoji' => $emoji . '🎯',
                'message' => 'Welcome to the points system! Here\'s your special emoji!',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Points admin welcome notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'points_admin_id' => $this->admin['employee_id'],
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);

            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send points admin welcome notification', [
                'employee_id' => $employee->employee_id,
                'points_admin_id' => $this->admin['employee_id'],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    // Send emoji change notification for existing employee
    private function sendEmojiChangeNotification($employee, $oldEmoji, $newEmoji, $wasAutoGenerated)
    {
        try {
            $changeType = $wasAutoGenerated ? 'automatically updated' : 'updated';
            $content = $wasAutoGenerated 
                ? "Hi {$employee->name}! The Points Admin has automatically updated your profile emoji from {$oldEmoji} to {$newEmoji} based on your profile changes. Your new emoji will be used in the points system!"
                : "Hi {$employee->name}! The Points Admin has updated your profile emoji from {$oldEmoji} to {$newEmoji}. Your new emoji represents you in the points system!";

            // Send change notification
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->admin['employee_id'],
                'sender_name' => $this->admin['name'] ?? 'Points Admin',
                'title' => "Your emoji has been {$changeType}! {$newEmoji}",
                'content' => $content,
                'type' => 'emoji_update',
                'priority' => 'normal',
                'related_id' => $employee->employee_id
            ]);

            // Send new emoji as reaction with points theme
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->admin['employee_id'],
                'sender_name' => $this->admin['name'] ?? 'Points Admin',
                'emoji' => $newEmoji . '⭐',
                'message' => 'Your new emoji is ready for the points system!',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Points admin emoji change notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'points_admin_id' => $this->admin['employee_id'],
                'old_emoji' => $oldEmoji,
                'new_emoji' => $newEmoji,
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);

            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send points admin emoji change notification', [
                'employee_id' => $employee->employee_id,
                'points_admin_id' => $this->admin['employee_id'],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    // H - B : Delete Employee
    public function deleteEmployee(Request $request)
    {
        $employee = Employee::where('employee_id', $request->id)->first();

        try {
            if ($employee) {
                // Send farewell notification before deletion
                $notificationSent = $this->sendFarewellNotification($employee);

                // FIX: Add proper validation before deleting image file
                if (!empty($employee->image) && $employee->image !== '/' && file_exists(public_path($employee->image))) {
                    unlink(public_path($employee->image));
                }
                
                $depart = $employee->department_id;
                $employee->delete();
                $count = Employee::where('tenant_id', $this->admin['tenant_id'])
                    ->where('department_id', $depart)->count();
                $test = Department::where('id', $depart)->update(['no_of_emp' => $count]);
                
                return response()->json([
                    'Error' => false,
                    'Message' => 'Employee record deleted successfully',
                    'NotificationSent' => $notificationSent ? 'Farewell notification sent' : 'Notification failed',
                ]);
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee Record not found!',
                ], 404);
            }
        } catch (\Throwable $t) {
            Log::error('Points admin employee deletion failed', [
                'employee_id' => $request->id,
                'points_admin_id' => $this->admin['employee_id'],
                'error' => $t->getMessage()
            ]);
            
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage()
            ]);
        }
    }

    // Send farewell notification before employee deletion
    private function sendFarewellNotification($employee)
    {
        try {
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->admin['employee_id'],
                'sender_name' => $this->admin['name'] ?? 'Points Admin',
                'title' => 'Farewell from Points System 👋',
                'content' => "Hi {$employee->name}, thank you for being part of our points system. Your emoji {$employee->emoji} will be remembered along with all your achievements. We wish you all the best in your future endeavors!",
                'type' => 'farewell',
                'priority' => 'high',
                'related_id' => $employee->employee_id
            ]);

            // Send farewell emoji with points theme
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->admin['employee_id'],
                'sender_name' => $this->admin['name'] ?? 'Points Admin',
                'emoji' => '👋🏆',
                'message' => 'Thank you for your participation in our points system!',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Points admin farewell notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'points_admin_id' => $this->admin['employee_id'],
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);

            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send points admin farewell notification', [
                'employee_id' => $employee->employee_id,
                'points_admin_id' => $this->admin['employee_id'],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    //Employee details view function
    public function employeeDetails(){
        $admin_count = Employee::where('role' , 'admin')->count();
        $manager_count = Employee::where('role' , 'manager')->count();
        return view('PointsAdmin.employee.employee_details' , compact('admin_count' , 'manager_count' ,));
    }

    /**
     * Assign custom emoji/GIF to employee
     */
    private function assignEmojiToEmployee($employee, $emojiId, $message = null)
    {
        try {
            // Get the current user (PointsAdmin)
            $assignedBy = [
                'id' => $this->admin['employee_id'],
                'type' => 'App\Models\Admin' // PointsAdmins are stored as Admins in the User model
            ];
            
            if ($assignedBy) {
                \App\Models\EmojiAssignment::create([
                    'assignable_id' => $employee->employee_id,
                    'assignable_type' => 'App\Models\Employee',
                    'assigned_by_id' => $assignedBy['id'],
                    'assigned_by_type' => $assignedBy['type'],
                    'emoji_id' => $emojiId,
                    'message' => $message,
                    'tenant_id' => $this->admin['tenant_id']
                ]);

                Log::info('Custom emoji assigned to employee by points admin', [
                    'employee_id' => $employee->employee_id,
                    'emoji_id' => $emojiId,
                    'assigned_by' => $assignedBy
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Failed to assign custom emoji to employee by points admin', [
                'employee_id' => $employee->employee_id,
                'emoji_id' => $emojiId,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get emoji assignments for a specific employee
     */
    public function getEmojiAssignments($employeeId)
    {
        try {
            Log::info('Points admin fetching emoji assignments', [
                'employee_id' => $employeeId,
                'tenant_id' => $this->admin['tenant_id'],
                'points_admin_data' => $this->admin
            ]);

            $assignments = \App\Models\EmojiAssignment::with('emoji')
                ->where('assignable_id', $employeeId)
                ->where('assignable_type', 'App\Models\Employee')
                ->where('tenant_id', $this->admin['tenant_id'])
                ->orderBy('created_at', 'desc')
                ->get();

            Log::info('Points admin found assignments', [
                'count' => $assignments->count(),
                'assignments' => $assignments->toArray()
            ]);

            return response()->json([
                'success' => true,
                'assignments' => $assignments
            ]);
        } catch (\Exception $e) {
            Log::error('Points admin error fetching emoji assignments', [
                'employee_id' => $employeeId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch emoji assignments'
            ], 500);
        }
    }

    /**
     * Assign emoji to employee (standalone assignment)
     */
    public function assignEmojiToEmployeeStandalone(Request $request)
    {
        try {
            $request->validate([
                'employee_id' => 'required|string',
                'emoji_id' => 'required|integer',
                'message' => 'nullable|string|max:500'
            ]);

            $employee = Employee::where('employee_id', $request->employee_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();

            if (!$employee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee not found'
                ], 404);
            }

            $emoji = \App\Models\CustomEmoji::where('id', $request->emoji_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->where('is_active', true)
                ->first();

            if (!$emoji) {
                return response()->json([
                    'success' => false,
                    'message' => 'Emoji/GIF not found or inactive'
                ], 404);
            }

            // Get the current user (PointsAdmin)
            $assignedBy = [
                'id' => $this->admin['employee_id'],
                'type' => 'App\Models\Admin' // PointsAdmins are stored as Admins in the User model
            ];
            
            if (!$assignedBy) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unable to identify current user'
                ], 400);
            }

            // Create the assignment
            $assignment = \App\Models\EmojiAssignment::create([
                'assignable_id' => $employee->employee_id,
                'assignable_type' => 'App\Models\Employee',
                'assigned_by_id' => $assignedBy['id'],
                'assigned_by_type' => $assignedBy['type'],
                'emoji_id' => $request->emoji_id,
                'message' => $request->message,
                'tenant_id' => $this->admin['tenant_id']
            ]);

            Log::info('Points admin emoji assigned to employee via standalone assignment', [
                'employee_id' => $employee->employee_id,
                'emoji_id' => $request->emoji_id,
                'assigned_by' => $assignedBy,
                'assignment_id' => $assignment->id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assigned successfully to ' . $employee->name,
                'assignment' => $assignment
            ]);

        } catch (\Exception $e) {
            Log::error('Points admin failed to assign emoji to employee', [
                'employee_id' => $request->employee_id,
                'emoji_id' => $request->emoji_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to assign emoji/GIF: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove emoji assignment
     */
    public function removeEmojiAssignment(Request $request)
    {
        try {
            $request->validate([
                'assignment_id' => 'required|integer'
            ]);

            $assignment = \App\Models\EmojiAssignment::where('id', $request->assignment_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();

            if (!$assignment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Assignment not found'
                ], 404);
            }

            $employeeName = 'Unknown';
            if ($assignment->assignable_type === 'App\Models\Employee') {
                $employee = Employee::where('employee_id', $assignment->assignable_id)
                    ->where('tenant_id', $this->admin['tenant_id'])
                    ->first();
                if ($employee) {
                    $employeeName = $employee->name;
                }
            }

            $assignment->delete();

            Log::info('Points admin emoji assignment removed', [
                'assignment_id' => $request->assignment_id,
                'employee_name' => $employeeName
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assignment removed successfully from ' . $employeeName
            ]);

        } catch (\Exception $e) {
            Log::error('Points admin failed to remove emoji assignment', [
                'assignment_id' => $request->assignment_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to remove emoji assignment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update existing emoji assignment
     */
    public function updateEmojiAssignment(Request $request)
    {
        try {
            $request->validate([
                'assignment_id' => 'required|integer',
                'emoji_id' => 'required|integer',
                'message' => 'nullable|string|max:500'
            ]);

            $assignment = \App\Models\EmojiAssignment::where('id', $request->assignment_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->first();

            if (!$assignment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Assignment not found'
                ], 404);
            }

            $emoji = \App\Models\CustomEmoji::where('id', $request->emoji_id)
                ->where('tenant_id', $this->admin['tenant_id'])
                ->where('is_active', true)
                ->first();

            if (!$emoji) {
                return response()->json([
                    'success' => false,
                    'message' => 'Emoji/GIF not found or inactive'
                ], 404);
            }

            // Update the assignment
            $assignment->update([
                'emoji_id' => $request->emoji_id,
                'message' => $request->message
            ]);

            Log::info('Points admin emoji assignment updated', [
                'assignment_id' => $assignment->id,
                'new_emoji_id' => $request->emoji_id,
                'message' => $request->message
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assignment updated successfully',
                'assignment' => $assignment->load('emoji')
            ]);

        } catch (\Exception $e) {
            Log::error('Points admin failed to update emoji assignment', [
                'assignment_id' => $request->assignment_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update emoji assignment: ' . $e->getMessage()
            ], 500);
        }
    }
}