<?php

namespace App\Http\Controllers;

use App\Models\Notification;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class NotificationController extends Controller
{
    private NotificationService $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Get current user info
     */
    private function getCurrentUser(Request $request)
    {
        $user = $request->attributes->get('authenticated_user');
        $tenantId = $request->attributes->get('tenant_id');
        // dd($user, $tenantId);
        if (get_class($user) === 'App\Models\Admin') {
            return [
                'tenant_id' => $tenantId,
                'type' => 'admin',
                'id' => $user->id,
            ];
        }
        
        if (isset($user->employee_id)) {
            return [
                'tenant_id' => $tenantId,
                'type' => 'employee',
                'id' => $user->employee_id,
            ];
        }
        
        abort(403, 'Unable to determine user identity.');
    }

    /**
     * Get notifications for authenticated user
     */
    public function index(Request $request)
    {
        $user = $this->getCurrentUser($request);
        $notifications = $this->notificationService->getUserNotifications(
            $user['tenant_id'],
            $user['type'],
            $user['id'],
            50
        );

        return view('notifications.index', compact('notifications'));
    }

    /**
     * Get notifications as JSON (for dropdown/AJAX)
     */
    public function getNotifications(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);
        $limit = $request->get('limit', 10);
        $onlyUnread = $request->boolean('unread_only', false);

        $notifications = $this->notificationService->getUserNotifications(
            $user['tenant_id'],
            $user['type'],
            $user['id'],
            $limit,
            $onlyUnread
        );

        $unreadCount = $this->notificationService->getUnreadCount(
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => true,
            'notifications' => $notifications->map(function ($notification) {
                return [
                    'id' => $notification->id,
                    'title' => $notification->title,
                    'content' => $notification->content,
                    'sender_name' => $notification->sender_name,
                    'is_read' => $notification->is_read,
                    'time_ago' => $notification->time_ago,
                    'type' => $notification->type,
                    'created_at' => $notification->created_at->format('Y-m-d H:i:s'),
                ];
            }),
            'unread_count' => $unreadCount,
        ]);
    }

    /**
     * Get unread count only
     */
    public function getUnreadCount(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        $count = $this->notificationService->getUnreadCount(
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => true,
            'count' => $count,
        ]);
    }

    /**
     * Mark notification as read
     */
    public function markAsRead(Request $request, $id): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        $success = $this->notificationService->markAsRead(
            $id,
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => $success,
            'message' => $success ? 'Notification marked as read' : 'Notification not found',
        ], $success ? 200 : 404);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        $count = $this->notificationService->markAllAsRead(
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => true,
            'message' => "Marked {$count} notifications as read",
            'count' => $count,
        ]);
    }

    /**
     * Delete notification
     */
    public function delete(Request $request, $id): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        $success = $this->notificationService->deleteNotification(
            $id,
            $user['tenant_id'],
            $user['type'],
            $user['id']
        );

        return response()->json([
            'success' => $success,
            'message' => $success ? 'Notification deleted' : 'Notification not found',
        ], $success ? 200 : 404);
    }

    /**
     * Send announcement (Admin only)
     */
    public function sendAnnouncement(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        // Only admins can send announcements
        if ($user['type'] !== 'admin') {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:1000',
            'target' => 'required|in:all_employees,all_admins,specific',
            'receivers' => 'required_if:target,specific|array',
            'receivers.*.type' => 'required_with:receivers|in:admin,employee',
            'receivers.*.id' => 'required_with:receivers|integer',
        ]);

        $title = $request->title;
        $content = $request->content;
        $target = $request->target;

        try {
            if ($target === 'all_employees') {
                $this->notificationService->sendToAllEmployees(
                    $user['tenant_id'],
                    $title,
                    $content,
                    $user['type'],
                    $user['id']
                );
            } elseif ($target === 'all_admins') {
                $this->notificationService->sendToAllAdmins(
                    $user['tenant_id'],
                    $title,
                    $content,
                    $user['type'],
                    $user['id']
                );
            } else {
                $this->notificationService->sendAnnouncement(
                    $user['tenant_id'],
                    $title,
                    $content,
                    $request->receivers
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Announcement sent successfully',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send announcement: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Send notification to specific user (Admin only)
     */
    public function sendToUser(Request $request): JsonResponse
    {
        $user = $this->getCurrentUser($request);

        // Only admins can send direct notifications
        if ($user['type'] !== 'admin') {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $request->validate([
            'receiver_type' => 'required|in:admin,employee',
            'receiver_id' => 'required|integer',
            'title' => 'required|string|max:255',
            'content' => 'required|string|max:1000',
            'type' => 'nullable|string|max:50',
        ]);

        try {
            $this->notificationService->sendToUser(
                $user['tenant_id'],
                $request->receiver_type,
                $request->receiver_id,
                $request->title,
                $request->content,
                $user['type'],
                $user['id'],
                $request->type ?? 'general'
            );

            return response()->json([
                'success' => true,
                'message' => 'Notification sent successfully',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send notification: ' . $e->getMessage(),
            ], 500);
        }
    }
}