<?php

namespace App\Http\Controllers\Manager;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use App\Models\Points;
use App\Models\Target;
use App\Models\Product;
use App\Services\PointsEmailNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PointsController extends Controller
{
    protected $mngr;
    private PointsEmailNotificationService $emailNotificationService;

    //H - B : Constructor
    public function __construct(PointsEmailNotificationService $emailNotificationService)
    {
        $this->emailNotificationService = $emailNotificationService;
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->mngr = session('ismngr');
            return $next($request);
        });
    }

    // H-B : Manager Add Points to Employee Directly
    public function mngrPoints(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required',
            'vin_no' => 'required',
            'target_id' => 'required|array',
            'department_id' => 'required', // Now required from frontend
            'category' => 'required', // Now required from frontend
        ]);

        // Verify employee exists
        $employee = Employee::find($request->employee_id);
        if (!$employee) {
            return response()->json([
                'Error' => true,
                'Message' => 'Employee not found!',
            ]);
        }

        $data = [
            'employee_id' => $request->employee_id,
            'department_id' => $request->department_id, // Use from frontend
            'category' => $request->category, // Use from frontend
            'vin_no' => $request->vin_no,
            'status' => 'Redeemed',
            'detail' => null,
            'tenant_id' => $this->mngr['tenant_id'],
        ];

        $targets = $request->target_id;
        $points = array_count_values($targets);

        try {
            DB::beginTransaction();
            $done = [];

            foreach($points as $target => $point) {
                $data['target_id'] = $target;
                $data['points'] = $point;
                $done[] = Points::create($data);
                
                $prog = Target::where('tenant_id', $this->mngr['tenant_id'])
                    ->where('department_id', $this->mngr['department_id'])
                    ->whereNot('status', '1')
                    ->where('id', $target)->first();
                    
                if($prog) {
                    if(($prog->progress + $point) >= $prog->quantity) {
                        $prog->update([
                            'progress' => $prog->quantity,
                            'status' => '0',
                            'completion_date' => now(),
                        ]);
                    } else {
                        $prog->update(['progress' => $prog->progress + $point]);
                    }
                }
            }

            if($done) {
                updateLeaderBoard(
                    $request->employee_id, 
                    $request->department_id, // Use from frontend
                    $request->category, // Use from frontend
                    $this->mngr['tenant_id']
                );

                // Send email notification for points awarded by Manager
                $totalPoints = array_sum($points);
                $targetNames = [];
                foreach($points as $targetId => $pointValue) {
                    $target = Target::find($targetId);
                    if($target) {
                        $targetNames[] = $target->name ?? "Target #{$targetId}";
                    } else {
                        $targetNames[] = "Target #{$targetId}";
                    }
                }
                
                $this->emailNotificationService->sendPointsRequestApprovedNotification(
                    $request->employee_id,
                    $this->mngr['tenant_id'],
                    $totalPoints,
                    implode(', ', $targetNames),
                    $this->mngr['employee_id']
                );

                DB::commit();
                return response()->json([
                    'Error' => false,
                    'Message' => 'Points Added Successfully!',
                ]);
            } else {
                DB::rollBack();
                return response()->json([
                    'Error' => true,
                    'Message' => 'Error: Attempt Failed!',
                ]);
            }

        } catch(\Throwable $t) {
            DB::rollBack();
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    // H-B : Manager Reject Points
    public function mngrRejectPoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required',
            'details' => 'required|string|max:250'
        ]);
        
        try {
            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Point record not found!',
                ]);
            }

            $point->update([
                'status' => 'Rejected',
                'detail' => $request->details,
            ]);

            // Send email notification for rejection
            $target = Target::where('id', $point->target_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();
            $targetName = $target ? ($target->name ?? "Target #{$point->target_id}") : "Target #{$point->target_id}";
            
            $this->emailNotificationService->sendPointsRequestRejectedNotification(
                $point->employee_id,
                $this->mngr['tenant_id'],
                $point->points,
                $targetName,
                $request->details,
                $this->mngr['employee_id']
            );

            return response()->json([
                'Error' => false,
                'Message' => 'Operation Successful!',
            ]);

        } catch (\Throwable $t) {
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    // H-B : Manager Approve Points
    public function mngrApprovePoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required',
        ]);
        
        try {
            DB::beginTransaction();

            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->mngr['tenant_id'])->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Point record not found!',
                ]);
            }

            // Get employee details
            $employee = Employee::find($point->employee_id);
            if (!$employee) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee not found!',
                ]);
            }

            // Update point record with missing department/category if needed
            $pointUpdateData = ['status' => 'Redeemed'];
            if (!$point->department_id || !$point->category) {
                $pointUpdateData['department_id'] = $employee->department_id;
                $pointUpdateData['category'] = $employee->category;
            }
            $point->update($pointUpdateData);

            $target = Target::where('id', $point->target_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->whereNot('status', '1')->first();

            if ($target) {
                $quantity = $target->quantity;
                $progress = $target->progress;
                $progress += $point->points;

                if ($progress >= $quantity) {
                    $target->update([
                        'progress' => $target->quantity,
                        'status' => '0',
                        'completion_date' => now(),
                    ]);
                } else {
                    $target->update([
                        'progress' => $progress,
                    ]);
                }
            }

            // Use employee data for leaderboard update to ensure non-null values
            updateLeaderBoard(
                $point->employee_id, 
                $employee->department_id,
                $employee->category,
                $this->mngr['tenant_id']
            );

            // Send email notification for approval
            $targetName = $target ? ($target->name ?? "Target #{$point->target_id}") : "Target #{$point->target_id}";
            $this->emailNotificationService->sendPointsRequestApprovedNotification(
                $point->employee_id,
                $this->mngr['tenant_id'],
                $point->points,
                $targetName,
                $this->mngr['employee_id']
            );

            DB::commit();
            return response()->json([
                'Error' => false,
                'Message' => 'Operation Successful!',
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    /**
     * Get point details for editing
     */
    public function getPoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer|exists:points,id',
        ]);

        try {
            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Point record not found!',
                ]);
            }

            return response()->json([
                'Error' => false,
                'Message' => 'Point details retrieved successfully!',
                'data' => [
                    'id' => $point->id,
                    'employee_id' => $point->employee_id,
                    'vin_no' => $point->vin_no,
                    'target_id' => $point->target_id,
                    'points' => $point->points,
                    'status' => $point->status,
                    'detail' => $point->detail,
                    'department_id' => $point->department_id,
                    'category' => $point->category,
                    'created_at' => $point->created_at,
                    'updated_at' => $point->updated_at,
                ]
            ]);

        } catch (\Throwable $t) {
            Log::error('Get points failed', [
                'point_id' => $request->id,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);
            
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    /**
     * Update points after editing
     */
    public function updatePoints(Request $request)
    {
        $validated = $request->validate([
            'point_id' => 'required|integer|exists:points,id',
            'employee_id' => 'required|integer|exists:employees,employee_id',
            'vin_no' => 'required|string|max:255',
            'target_id' => 'required|integer|exists:targets,id',
            'points' => 'required|integer|min:1',
            'status' => 'required|in:Pending,Redeemed,Rejected',
            'rejection_reason' => 'nullable|string|max:500',
        ]);

        try {
            DB::beginTransaction();

            $point = Points::where('id', $request->point_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Point record not found!',
                ]);
            }

            // Store original values for calculations
            $originalStatus = $point->status;
            $originalPoints = $point->points;
            $originalTargetId = $point->target_id;

            // Get employee details for department_id and category
            $employee = Employee::find($request->employee_id);
            if (!$employee) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee not found!',
                ]);
            }

            // Handle target progress changes
            $this->handleTargetProgressOnEdit(
                $originalTargetId,
                $request->target_id,
                $originalPoints,
                $request->points,
                $originalStatus,
                $request->status
            );

            // Update the point record
            $updateData = [
                'employee_id' => $request->employee_id,
                'vin_no' => $request->vin_no,
                'target_id' => $request->target_id,
                'points' => $request->points,
                'status' => $request->status,
                'department_id' => $employee->department_id,
                'category' => $employee->category,
                'detail' => $request->status === 'Rejected' ? $request->rejection_reason : null,
            ];

            $point->update($updateData);

            // Update leaderboard if needed
            if ($originalStatus !== $request->status || $originalPoints !== $request->points || $point->employee_id !== $request->employee_id) {
                // If employee changed, update both old and new employee leaderboards
                if ($point->employee_id !== $request->employee_id) {
                    // Update original employee leaderboard
                    updateLeaderBoard(
                        $point->employee_id,
                        $point->department_id,
                        $point->category,
                        $this->mngr['tenant_id']
                    );
                }
                
                // Update new/current employee leaderboard
                updateLeaderBoard(
                    $request->employee_id,
                    $employee->department_id,
                    $employee->category,
                    $this->mngr['tenant_id']
                );
            }

            // Send email notification for points update
            $target = Target::find($request->target_id);
            $targetName = $target ? ($target->name ?? "Target #{$request->target_id}") : "Target #{$request->target_id}";
            
            // Determine the appropriate notification based on status change
            if ($originalStatus !== $request->status) {
                if ($request->status === 'Redeemed') {
                    $this->emailNotificationService->sendPointsRequestApprovedNotification(
                        $request->employee_id,
                        $this->mngr['tenant_id'],
                        $request->points,
                        $targetName,
                        $this->mngr['employee_id']
                    );
                } elseif ($request->status === 'Rejected') {
                    $this->emailNotificationService->sendPointsRequestRejectedNotification(
                        $request->employee_id,
                        $this->mngr['tenant_id'],
                        $request->points,
                        $targetName,
                        $request->rejection_reason ?? 'No reason provided',
                        $this->mngr['employee_id']
                    );
                }
            } else {
                // Status didn't change, but points or other details were updated
                // Send a general update notification
                $this->emailNotificationService->sendPointsRequestApprovedNotification(
                    $request->employee_id,
                    $this->mngr['tenant_id'],
                    $request->points,
                    $targetName,
                    $this->mngr['employee_id']
                );
            }

            DB::commit();

            return response()->json([
                'Error' => false,
                'Message' => 'Point updated successfully!',
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            Log::error('Update points failed', [
                'point_id' => $request->point_id,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Update failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Reverse approved points back to pending
     */
    public function reversePoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer|exists:points,id',
        ]);

        try {
            DB::beginTransaction();

            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->where('status', 'Redeemed')
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Approved point record not found!',
                ]);
            }

            // Get employee details to ensure we have department_id and category
            $employee = Employee::find($point->employee_id);
            if (!$employee) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee not found!',
                ]);
            }

            // Update point record with missing department/category if needed
            $updatePointData = [
                'status' => 'Pending',
                'detail' => null,
            ];

            // Ensure department_id and category are not null
            if (!$point->department_id || !$point->category) {
                $updatePointData['department_id'] = $employee->department_id;
                $updatePointData['category'] = $employee->category;
            }

            $point->update($updatePointData);

            // Reverse target progress
            $target = Target::where('id', $point->target_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();

            if ($target) {
                $newProgress = max(0, $target->progress - $point->points);
                
                $updateData = ['progress' => $newProgress];
                
                // If target was completed, reactivate it
                if ($target->status === '0') {
                    $updateData['status'] = '2';
                    $updateData['completion_date'] = null;
                }
                
                $target->update($updateData);
            }

            // Use employee data for leaderboard update to ensure non-null values
            updateLeaderBoard(
                $point->employee_id,
                $employee->department_id,
                $employee->category,
                $this->mngr['tenant_id']
            );

            // Send email notification for reversal
            $targetName = $target ? ($target->name ?? "Target #{$point->target_id}") : "Target #{$point->target_id}";
            $this->emailNotificationService->sendPointsRequestReversedNotification(
                $point->employee_id,
                $this->mngr['tenant_id'],
                $point->points,
                $targetName,
                $this->mngr['employee_id']
            );

            DB::commit();

            return response()->json([
                'Error' => false,
                'Message' => 'Points reversed successfully! Status changed to pending.',
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            Log::error('Reverse points failed', [
                'point_id' => $request->id,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Reverse failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Reprocess rejected points back to pending
     */
    public function reprocessPoints(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer|exists:points,id',
        ]);

        try {
            $point = Points::where('id', $request->id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->where('status', 'Rejected')
                ->first();

            if (!$point) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Rejected point record not found!',
                ]);
            }

            // Update point status to pending
            $point->update([
                'status' => 'Pending',
                'detail' => null,
            ]);

            // Send email notification for reprocessing
            $target = Target::where('id', $point->target_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();
            $targetName = $target ? ($target->name ?? "Target #{$point->target_id}") : "Target #{$point->target_id}";
            
            $this->emailNotificationService->sendPointsRequestReprocessedNotification(
                $point->employee_id,
                $this->mngr['tenant_id'],
                $point->points,
                $targetName,
                $this->mngr['employee_id']
            );

            return response()->json([
                'Error' => false,
                'Message' => 'Points reprocessed successfully! Status changed to pending.',
            ]);

        } catch (\Throwable $t) {
            Log::error('Reprocess points failed', [
                'point_id' => $request->id,
                'error' => $t->getMessage(),
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Reprocess failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Handle target progress changes during point editing
     */
    private function handleTargetProgressOnEdit($originalTargetId, $newTargetId, $originalPoints, $newPoints, $originalStatus, $newStatus)
    {
        // If original status was approved, we need to reverse the progress first
        if ($originalStatus === 'Redeemed') {
            $originalTarget = Target::where('id', $originalTargetId)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();
                
            if ($originalTarget) {
                $newProgress = max(0, $originalTarget->progress - $originalPoints);
                
                $updateData = ['progress' => $newProgress];
                
                // If target was completed, check if it should be reactivated
                if ($originalTarget->status === '0' && $newProgress < $originalTarget->quantity) {
                    $updateData['status'] = '2';
                    $updateData['completion_date'] = null;
                }
                
                $originalTarget->update($updateData);
            }
        }

        // If new status is approved, add the new progress
        if ($newStatus === 'Redeemed') {
            $newTarget = Target::where('id', $newTargetId)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();
                
            if ($newTarget) {
                $newProgress = $newTarget->progress + $newPoints;
                
                $updateData = ['progress' => $newProgress];
                
                // Check if target is now completed
                if ($newProgress >= $newTarget->quantity) {
                    $updateData['progress'] = $newTarget->quantity;
                    $updateData['status'] = '0';
                    $updateData['completion_date'] = now();
                }
                
                $newTarget->update($updateData);
            }
        }
    }

    /**
     * Bulk approve points
     */
    public function bulkApprovePoints(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'required|integer|exists:points,id',
        ]);

        try {
            $successCount = 0;
            $failedCount = 0;
            $completedTargets = [];
            $employeeIds = [];

            DB::beginTransaction();

            foreach ($request->ids as $pointId) {
                try {
                    $point = Points::where('id', $pointId)
                        ->where('tenant_id', $this->mngr['tenant_id'])
                        ->where('status', 'Pending')
                        ->first();

                    if (!$point) {
                        $failedCount++;
                        continue;
                    }

                    // Get employee details
                    $employee = Employee::find($point->employee_id);
                    if (!$employee) {
                        $failedCount++;
                        continue;
                    }

                    // Update point record with missing department/category if needed
                    $pointUpdateData = ['status' => 'Redeemed'];
                    if (!$point->department_id || !$point->category) {
                        $pointUpdateData['department_id'] = $employee->department_id;
                        $pointUpdateData['category'] = $employee->category;
                    }
                    $point->update($pointUpdateData);

                    // Get target information
                    $target = Target::where('id', $point->target_id)
                        ->where('tenant_id', $this->mngr['tenant_id'])
                        ->first();

                    if ($target) {
                        // Update target progress
                        $newProgress = $target->progress + $point->points;
                        
                        if ($newProgress >= $target->quantity) {
                            $target->update([
                                'progress' => $target->quantity,
                                'status' => '0',
                                'completion_date' => now(),
                            ]);
                            
                            $completedTargets[] = [
                                'name' => $target->name ?? $target->custom ?? "Target #{$point->target_id}",
                                'employee_id' => $point->employee_id
                            ];
                        } else {
                            $target->update([
                                'progress' => $newProgress,
                            ]);
                        }
                    }

                    // Update leaderboard using employee data
                    updateLeaderBoard(
                        $point->employee_id,
                        $employee->department_id,
                        $employee->category,
                        $this->mngr['tenant_id']
                    );

                    $employeeIds[] = $point->employee_id;
                    $successCount++;

                } catch (\Exception $e) {
                    Log::error('Individual bulk approve failed', [
                        'point_id' => $pointId,
                        'error' => $e->getMessage()
                    ]);
                    $failedCount++;
                }
            }

            DB::commit();

            $message = "Bulk approval completed. {$successCount} points approved successfully";
            if ($failedCount > 0) {
                $message .= ", {$failedCount} failed";
            }
            $message .= ".";

            return response()->json([
                'Error' => false,
                'Message' => $message,
                'Details' => [
                    'successful' => $successCount,
                    'failed' => $failedCount,
                    'completed_targets' => count($completedTargets)
                ]
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            
            Log::error('Bulk approve points failed', [
                'manager_id' => $this->mngr['id'] ?? null,
                'tenant_id' => $this->mngr['tenant_id'],
                'point_ids' => $request->ids,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Bulk approval failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Bulk reject points (works for both Pending and Approved points)
     */
    public function bulkRejectPoints(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'required|integer|exists:points,id',
            'details' => 'required|string|max:500',
        ]);

        try {
            $successCount = 0;
            $failedCount = 0;
            $reversedTargets = [];

            DB::beginTransaction();

            foreach ($request->ids as $pointId) {
                try {
                    $point = Points::where('id', $pointId)
                        ->where('tenant_id', $this->mngr['tenant_id'])
                        ->whereIn('status', ['Pending', 'Redeemed']) // Allow both pending and approved
                        ->first();

                    if (!$point) {
                        $failedCount++;
                        continue;
                    }

                    // If point was approved, reverse the target progress
                    if ($point->status === 'Redeemed') {
                        $target = Target::where('id', $point->target_id)
                            ->where('tenant_id', $this->mngr['tenant_id'])
                            ->first();

                        if ($target) {
                            $newProgress = max(0, $target->progress - $point->points);
                            
                            $updateData = ['progress' => $newProgress];
                            
                            // If target was completed, check if it should be reactivated
                            if ($target->status === '0' && $newProgress < $target->quantity) {
                                $updateData['status'] = '2';
                                $updateData['completion_date'] = null;
                                $reversedTargets[] = $point->target_id;
                            }
                            
                            $target->update($updateData);
                        }

                        // Update leaderboard to reflect the rejection
                        $employee = Employee::find($point->employee_id);
                        if ($employee) {
                            updateLeaderBoard(
                                $point->employee_id,
                                $employee->department_id,
                                $employee->category,
                                $this->mngr['tenant_id']
                            );
                        }
                    }

                    // Update point status
                    $point->update([
                        'status' => 'Rejected',
                        'detail' => $request->details,
                    ]);

                    $successCount++;

                } catch (\Exception $e) {
                    Log::error('Individual bulk reject failed', [
                        'point_id' => $pointId,
                        'error' => $e->getMessage()
                    ]);
                    $failedCount++;
                }
            }

            DB::commit();

            $message = "Bulk rejection completed. {$successCount} points rejected successfully";
            if ($failedCount > 0) {
                $message .= ", {$failedCount} failed";
            }
            if (count($reversedTargets) > 0) {
                $message .= ". " . count($reversedTargets) . " completed targets were reactivated.";
            }
            $message .= ".";

            return response()->json([
                'Error' => false,
                'Message' => $message,
                'Details' => [
                    'successful' => $successCount,
                    'failed' => $failedCount,
                    'reversed_targets' => count($reversedTargets)
                ]
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            
            Log::error('Bulk reject points failed', [
                'manager_id' => $this->mngr['id'] ?? null,
                'tenant_id' => $this->mngr['tenant_id'],
                'point_ids' => $request->ids,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Bulk rejection failed: ' . $t->getMessage(),
            ]);
        }
    }

    /**
     * Bulk approve points (works for both Pending and Rejected points)
     */
    public function bulkApprovePointsEnhanced(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'required|integer|exists:points,id',
        ]);

        try {
            $successCount = 0;
            $failedCount = 0;
            $completedTargets = [];
            $reprocessedCount = 0;

            DB::beginTransaction();

            foreach ($request->ids as $pointId) {
                try {
                    $point = Points::where('id', $pointId)
                        ->where('tenant_id', $this->mngr['tenant_id'])
                        ->whereIn('status', ['Pending', 'Rejected']) // Allow both pending and rejected
                        ->first();

                    if (!$point) {
                        $failedCount++;
                        continue;
                    }

                    // Get employee details
                    $employee = Employee::find($point->employee_id);
                    if (!$employee) {
                        $failedCount++;
                        continue;
                    }

                    // Update point record with missing department/category if needed
                    $pointUpdateData = ['status' => 'Redeemed', 'detail' => null]; // Clear rejection reason
                    if (!$point->department_id || !$point->category) {
                        $pointUpdateData['department_id'] = $employee->department_id;
                        $pointUpdateData['category'] = $employee->category;
                    }
                    
                    // Track if this was a rejected point being reprocessed
                    if ($point->status === 'Rejected') {
                        $reprocessedCount++;
                    }
                    
                    $point->update($pointUpdateData);

                    // Get target information
                    $target = Target::where('id', $point->target_id)
                        ->where('tenant_id', $this->mngr['tenant_id'])
                        ->first();

                    if ($target) {
                        // Update target progress
                        $newProgress = $target->progress + $point->points;
                        
                        if ($newProgress >= $target->quantity) {
                            $target->update([
                                'progress' => $target->quantity,
                                'status' => '0',
                                'completion_date' => now(),
                            ]);
                            
                            $completedTargets[] = [
                                'name' => $target->name ?? $target->custom ?? "Target #{$point->target_id}",
                                'employee_id' => $point->employee_id
                            ];
                        } else {
                            $target->update([
                                'progress' => $newProgress,
                            ]);
                        }
                    }

                    // Update leaderboard using employee data
                    updateLeaderBoard(
                        $point->employee_id,
                        $employee->department_id,
                        $employee->category,
                        $this->mngr['tenant_id']
                    );

                    $successCount++;

                } catch (\Exception $e) {
                    Log::error('Individual bulk approve failed', [
                        'point_id' => $pointId,
                        'error' => $e->getMessage()
                    ]);
                    $failedCount++;
                }
            }

            DB::commit();

            $message = "Bulk approval completed. {$successCount} points approved successfully";
            if ($failedCount > 0) {
                $message .= ", {$failedCount} failed";
            }
            if ($reprocessedCount > 0) {
                $message .= ". {$reprocessedCount} previously rejected points were reprocessed";
            }
            $message .= ".";

            return response()->json([
                'Error' => false,
                'Message' => $message,
                'Details' => [
                    'successful' => $successCount,
                    'failed' => $failedCount,
                    'completed_targets' => count($completedTargets),
                    'reprocessed' => $reprocessedCount
                ]
            ]);

        } catch (\Throwable $t) {
            DB::rollBack();
            
            Log::error('Bulk approve points enhanced failed', [
                'manager_id' => $this->mngr['id'] ?? null,
                'tenant_id' => $this->mngr['tenant_id'],
                'point_ids' => $request->ids,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Bulk approval failed: ' . $t->getMessage(),
            ]);
        }
    }
}