<?php

namespace App\Http\Controllers\Manager;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\CategoryLeaderBoard;
use App\Models\Department;
use App\Models\DepartmentLeaderBoard;
use App\Models\Employee;
use App\Models\EmployeeLeaderBoard;
use App\Models\Points;
use App\Models\Product;
use App\Models\Target;
use App\Services\LeaderboardService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{
    protected $mngr;
    protected $leaderboardService;

    //H - B : Constructor
    public function __construct()
    {
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->mngr = session('ismngr');
            if ($this->mngr) {
                $this->leaderboardService = new LeaderboardService($this->mngr['tenant_id']);
            }
            return $next($request);
        });
    }

    //H-B : Manager Dashboard
    public function mngrDashboard()
    {
        $targets = Target::where('department_id', $this->mngr['department_id'])
            ->where('tenant_id', $this->mngr['tenant_id'])
            ->whereNot('status', '1')->get();

        $departments = Department::where('id', $this->mngr['department_id'])->first();

        $products = Product::where('tenant_id', $this->mngr['tenant_id'])->get();

        $points = Points::leftJoin('targets', function ($join) {
            $join->on('points.target_id', '=', 'targets.id')
                ->on('points.tenant_id', '=', 'targets.tenant_id'); // Match tenant_id in Points and Targets
        })
            ->leftJoin('products', function ($join) {
                $join->on('targets.product_id', '=', 'products.id')
                    ->on('targets.tenant_id', '=', 'products.tenant_id'); // Match tenant_id in Targets and Products
            })
            ->where('points.department_id', $this->mngr['department_id'])
            ->where('points.tenant_id', $this->mngr['tenant_id']) // Match tenant_id with $this->emp['tenant_id']
            ->get([
                'points.*',
                'products.name as name',
                'targets.custom as custom'
            ]);

        // Get assigned custom emojis/GIFs for this manager
        $assignedEmojis = $this->getAssignedEmojis($this->mngr['employee_id']);

        return view('Manager.Dashboard', compact('targets', 'departments', 'points', 'products', 'assignedEmojis'));
    }

    private function getAssignedEmojis($managerId)
    {
        // Get assigned emojis/GIFs for this manager directly
        return \App\Models\EmojiAssignment::with('emoji')
            ->where('assignable_id', $managerId)
            ->where('assignable_type', 'App\Models\Admin')
            ->where('tenant_id', $this->mngr['tenant_id'])
            ->orderBy('created_at', 'desc')
            ->get();
    }

 public function mManagerPoints()
    {
        $mngr = $this->mngr;
        $products = Product::where('tenant_id', $this->mngr['tenant_id'])->get();
        $targets = Target::where('department_id', $this->mngr['department_id'])
            ->where('tenant_id', $this->mngr['tenant_id'])->get();
        $points = Points::select(
            'points.*',
            'employees.name as e_name',
            'targets.product_id as prod',
            'targets.custom as custom'
        )
            ->join('employees', 'points.employee_id', '=', 'employees.employee_id') // Join with employees table
            ->leftJoin('targets', 'points.target_id', '=', 'targets.id')  // Left join to include nullable fields
            ->where('points.tenant_id', $this->mngr['tenant_id'])
            ->get();

        $employees = Employee::where('tenant_id', $this->mngr['tenant_id'])
            ->where('role', 'Employee')
            ->get(['name', 'status', 'employee_id', 'category', 'department_id']);


        return view('Manager.RequestedPoints', compact('mngr', 'employees', 'points', 'targets', 'products'));
    }
    //H-B : Department Employee List
    public function employeeList()
    {
        $department = Department::where('id', $this->mngr['department_id'])
            ->where('tenant_id', $this->mngr['tenant_id'])->first();
        $employees = Employee::where('department_id', $this->mngr['department_id'])
            ->where('tenant_id', $this->mngr['tenant_id'])->get();

        // Get available emojis/GIFs created by superadmin
        $availableEmojis = \App\Models\CustomEmoji::forTenant($this->mngr['tenant_id'])
            ->active()
            ->orderBy('type')
            ->orderBy('name')
            ->get();

        return view('Manager.Employee', compact('employees', 'department', 'availableEmojis'));
    }

    //H-B : All Target of Department
    public function depTargets()
    {
        $targets = Target::where('department_id', $this->mngr['department_id'])
            ->where('tenant_id', $this->mngr['tenant_id'])
            ->whereNot('status', '1')->get();
        $department = Department::where('id', $this->mngr['department_id'])
            ->where('tenant_id', $this->mngr['tenant_id'])->first();
        $products = Product::where('tenant_id', $this->mngr['tenant_id'])->get();
        return view('Manager.Targets.depTargets', compact('targets', 'department', 'products'));
    }

    // H-B: Category Target Listing page
    public function targetListing($category)
    {
        $path = request()->path();
        $categoryFromPath = last(explode('/', $path));
        $categoryCleaned = str_replace('+', ' ', $categoryFromPath);

        $category = $categoryCleaned;
        $targets = Target::leftjoin('products', 'targets.product_id', '=', 'products.id')
            ->where('targets.department_id', '=', $this->mngr['department_id'])
            ->where('targets.category', '=', $category)
            ->whereNot('targets.status', '1')
            ->get([
                'targets.*',
                'products.name as name'
            ]);

        $department = Department::where('id', $this->mngr['department_id'])
            ->where('tenant_id', $this->mngr['tenant_id'])->first();

        $products = Product::where('tenant_id', $this->mngr['tenant_id'])->get();

        $currentDate = now()->format('F d, Y');

        return view('Manager.Targets.categoryTargets', compact('targets', 'department', 'products', 'category', 'currentDate'));
    }

    // H-B: Employee Points Page
    public function employeePoints($category)
    {
        $path = request()->path();
        $categoryFromPath = last(explode('/', $path));
        $categoryCleaned = str_replace('+', ' ', $categoryFromPath);

        $category = $categoryCleaned;

        $products = Product::where('tenant_id', $this->mngr['tenant_id'])->get();

        $targets = Target::where('department_id', $this->mngr['department_id'])
            ->where('category', $category)
            ->where('tenant_id', $this->mngr['tenant_id'])->get();

        $points = Points::where('department_id', $this->mngr['department_id'])
            ->where('tenant_id', $this->mngr['tenant_id'])
            ->where('category', $category)->get();

        $employees = Employee::where('tenant_id', $this->mngr['tenant_id'])
            ->get(['name', 'status', 'employee_id', 'category', 'role']);


        return view('Manager.EmployeePoints', compact('points', 'category', 'employees', 'targets', 'products'));
    }

    // H-B: Category Admin Page
    public function categoryAdmin($category)
    {
        $path = request()->path();
        $categoryFromPath = last(explode('/', $path));
        $categoryCleaned = str_replace('+', ' ', $categoryFromPath);

        $category = $categoryCleaned;
        $admins = Employee::where('tenant_id', $this->mngr['tenant_id'])
            ->where('department_id', $this->mngr['department_id'])
            ->where('role', 'Admin')
            ->where('category', $category)->get();

        $department = Department::where('tenant_id', $this->mngr['tenant_id'])
            ->where('id', $this->mngr['department_id'])->first();

        $employees = Employee::where('tenant_id', $this->mngr['tenant_id'])
            ->where('department_id', $this->mngr['department_id'])
            ->where('role', '!=', 'Manager')->get();
        return view('Manager.Admins', compact('admins', 'department', 'employees', 'category'));
    }

    // H-B: Manager >> Admin Detail Page
    public function adminDetail($slug)
    {
        // Get the full path (e.g., "manager/admins/details/admin")
        $path = request()->path();

        // Split the path by '/' and get the last part (which will be 'admin' in this case)
        $slugFromPath = last(explode('/', $path)); // This will give you 'admin'

        $department = Department::where('tenant_id', $this->mngr['tenant_id'])
            ->where('id', $this->mngr['department_id'])->first();

        $employeeOrg = Employee::where('tenant_id', $this->mngr['tenant_id'])
            ->where('slug', $slugFromPath)->first();
        if (!$employeeOrg) {
            // Handle the case when no employee is found
            abort(404, 'Employee not found');
        }
        $employees = Employee::where('tenant_id', $this->mngr['tenant_id'])
            ->where('department_id', $this->mngr['department_id'])
            ->where('role', '!=', 'Manager')->get();

        $piePoints = Points::select(
            DB::raw("SUM(CASE WHEN status = 'Redeemed' THEN points ELSE 0 END) as redeemed"),
            DB::raw("SUM(CASE WHEN status = 'Pending' THEN points ELSE 0 END) as pending")
        )
            ->where('tenant_id', $employeeOrg->tenant_id)
            ->where('department_id', $employeeOrg->department_id)
            ->where('category', $employeeOrg->category)
            ->first();
        $piePoints->total_points = $piePoints->redeemed + $piePoints->pending;


        return view('Manager.adminDetail', compact('department', 'employeeOrg', 'piePoints', 'employees'));
    }

    // H-B : LeaderBoard
    public function viewLeaderBoard()
    {
        $currentMonth = Carbon::now()->format('Y-m');

        $months = EmployeeLeaderBoard::select('month_year')
            ->distinct()
            ->orderBy('month_year', 'desc')
            ->pluck('month_year');

        $employees = EmployeeLeaderBoard::join('employees', function ($join) {
            $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id'); // Match tenant_id
        })
            ->where('employee_leader_boards.tenant_id', '=', $this->mngr['tenant_id']) // Ensure tenant_id matches in employee_leader_boards
            ->where('month_year', $currentMonth)
            ->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee',
                'employees.category as category',
                'employees.image as image'
            ]);

        $departments = DepartmentLeaderBoard::join('departments', function ($join) {
            $join->on('department_leader_boards.department_id', '=', 'departments.id')
                ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id'); // Match tenant_id
        })
            ->join('employees', function ($join) {
                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
                    ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id') // Match tenant_id
                    ->where('employees.role', 'Manager');
            })
            ->where('department_leader_boards.tenant_id', '=', $this->mngr['tenant_id']) // Match tenant_id
            ->where('month_year', $currentMonth)
            ->groupBy('department_leader_boards.id', 'department_leader_boards.rank', 'department_leader_boards.earned_points', 'departments.name') // Group by leaderboard ID to avoid duplicates
            ->orderBy('department_leader_boards.rank', 'asc')
            ->get([
                'department_leader_boards.id',
                'department_leader_boards.rank',
                'department_leader_boards.earned_points',
                'departments.name as department',
                \DB::raw('COUNT(DISTINCT employees.employee_id) as count'), // Count unique managers
                \DB::raw('MIN(employees.name) as manager'), // Get any one manager's name
                \DB::raw('MIN(employees.image) as emp_image') // Get any one manager's image
            ]);

        $categories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('category_leader_board.*', 'departments.name as name')
            ->where('month_year', $currentMonth)->get();

        // Get available categories for advanced filtering
        $availableCategories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('departments.name')
            ->where('category_leader_board.tenant_id', $this->mngr['tenant_id'])
            ->distinct()
            ->pluck('departments.name')
            ->toArray();

        return view('Manager.Leaderboard', compact('employees', 'categories', 'departments', 'months', 'availableCategories'));
    }

    // H-B: Employee Filter LeaderBoard
    public function viewEmpLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"
        
        // Check if category filter is applied
        $category = request()->get('category');
        
        $query = EmployeeLeaderBoard::join('employees', function ($join) {
            $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id'); // Match tenant_id
        })
            ->where('employee_leader_boards.tenant_id', '=', $this->mngr['tenant_id']) // Ensure tenant_id matches in employee_leader_boards
            ->where('month_year', $currentMonth);
            
        // Apply category filter if specified
        if ($category && $category !== 'all') {
            $query->where('employees.category', '=', $category);
        }
        
        $employees = $query->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                'employee_leader_boards.*',
                'employees.name as employee',
                'employees.category as category',
                'employees.image as image'
            ]);
        //        dd($employees);
        return view('partials.employeeLeaderBoard', compact('employees'))->render();
    }

    // H-B: Employee Filter LeaderBoard
    public function viewDepLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"

        $departments = DepartmentLeaderBoard::join('departments', function ($join) {
            $join->on('department_leader_boards.department_id', '=', 'departments.id')
                ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id'); // Match tenant_id
        })
            ->join('employees', function ($join) {
                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
                    ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id') // Match tenant_id
                    ->where('employees.role', 'Manager');
            })
            ->where('department_leader_boards.tenant_id', '=', $this->mngr['tenant_id']) // Match tenant_id
            ->where('month_year', $currentMonth)
            ->groupBy('department_leader_boards.id', 'department_leader_boards.rank', 'department_leader_boards.earned_points', 'departments.name') // Group by leaderboard ID to avoid duplicates
            ->orderBy('department_leader_boards.rank', 'asc')
            ->get([
                'department_leader_boards.id',
                'department_leader_boards.rank',
                'department_leader_boards.earned_points',
                'departments.name as department',
                \DB::raw('COUNT(DISTINCT employees.employee_id) as count'), // Count unique managers
                \DB::raw('MIN(employees.name) as manager'), // Get any one manager’s name
                \DB::raw('MIN(employees.image) as emp_image') // Get any one manager’s image
            ]);
        //        dd($departments);
        return view('partials.departmentLeaderBoard', compact('departments'))->render();
    }

    // H-B: Employee Filter LeaderBoard
    public function viewCatLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"
        $categories = CategoryLeaderBoard::join('departments', 'category_leader_board.department_id', '=', 'departments.id')
            ->select('category_leader_board.*', 'departments.name as name')
            ->where('category_leader_board.tenant_id', '=', $this->mngr['tenant_id'])
            ->where('month_year', $currentMonth)
            ->get();
        //        dd($categories);
        return view('partials.categoryLeaderBoard', compact('categories'))->render();
    }

    // Enhanced Leaderboard Methods
    
    /**
     * Enhanced leaderboard view with all filters
     */
    public function viewEnhancedLeaderboard()
    {
        $currentMonth = Carbon::now()->format('Y-m');
        
        // Get available months
        $months = $this->leaderboardService->getAvailableMonths();
        
        // Get filter options
        $departments = $this->leaderboardService->getAvailableDepartments();
        $categories = $this->leaderboardService->getAvailableCategories();
        
        // Get initial data
        $leaderboardData = $this->leaderboardService->getLeaderboardData([
            'category' => 'employees',
            'time_period' => 'current_month'
        ]);
        
        return view('Manager.EnhancedLeaderboard', compact(
            'leaderboardData', 
            'months', 
            'departments', 
            'categories'
        ));
    }
    
    /**
     * AJAX endpoint for filtered leaderboard data
     */
    public function getFilteredLeaderboard(Request $request)
    {
        $filters = $request->only(['category', 'time_period', 'department_id', 'custom_date_range', 'specific_category', 'month']);
        
        try {
            // If specific_category is true, filter by the specific category
            if ($request->get('specific_category') && $request->get('category') !== 'all') {
                $data = $this->leaderboardService->getLeaderboardDataByCategory($filters);
            } else {
                $data = $this->leaderboardService->getLeaderboardData($filters);
            }
            
            $prizes = $this->leaderboardService->getPrizesForLeaderboard(
                $filters['category'] ?? 'all',
                $filters['time_period'] ?? 'current_month'
            );
            
            return response()->json([
                'success' => true,
                'data' => $data,
                'top3' => $data->take(3),
                'complete_ranking' => $data->skip(3),
                'total_count' => $data->count(),
                'prizes' => $prizes
            ]);
        } catch (\Exception $e) {
            Log::error('Error in Manager getFilteredLeaderboard: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load leaderboard data'
            ], 500);
        }
    }
    
    /**
     * Get BDC leaderboard specifically
     */
    public function getBDCLeaderboard(Request $request)
    {
        $timePeriod = $request->get('time_period', 'current_month');
        $data = $this->leaderboardService->getBDCLeaderboard($timePeriod);
        
        return view('partials.enhancedLeaderboard', [
            'data' => $data,
            'category' => 'BDC',
            'top3' => $data->take(3),
            'complete_ranking' => $data->skip(3)
        ])->render();
    }
    
    /**
     * Get Service Advisor leaderboard specifically
     */
    public function getServiceAdvisorLeaderboard(Request $request)
    {
        $timePeriod = $request->get('time_period', 'current_month');
        $data = $this->leaderboardService->getServiceAdvisorLeaderboard($timePeriod);
        
        return view('partials.enhancedLeaderboard', [
            'data' => $data,
            'category' => 'Service Advisor',
            'top3' => $data->take(3),
            'complete_ranking' => $data->skip(3)
        ])->render();
    }

}
