<?php

namespace App\Http\Controllers\Manager;

use App\Http\Controllers\Controller;
use App\Models\Department;
use App\Models\Employee;
use App\Services\UniversalNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class EmployeeController extends Controller
{
    protected $mngr;
    private UniversalNotificationService $notificationService;

    //H - B : Constructor
    public function __construct(UniversalNotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->mngr = session('ismngr');
            return $next($request);
        });
    }

    // H - B : Add And Edit Employee
    public function addAndEditEmployee(Request $request)
    {
        $employee = Employee::where('employee_id', $request->employee_id)->first();

        $validationRules = [
            'employee_id' => 'required',
            'name' => 'required',
            'email' => 'required',
            'role' => 'required',
            'department' => 'required',
            'category' => 'required',
            'emoji' => 'nullable|string|max:10', // Added emoji validation
            'emoji_id' => 'nullable|integer', // Added custom emoji validation
            'emoji_message' => 'nullable|string|max:500' // Added emoji message validation
        ];
        
        if(!$employee) {
            $validationRules['password'] = 'required';
        }
        
        $customMessages = [
            'department.required' => 'The department field is required.',
            'department.integer' => 'The department must have a valid value.',
            'emoji.max' => 'The emoji field must not exceed 10 characters.',
        ];
        
        $validate = $request->validate($validationRules, $customMessages);
        $image = null;
        $pas = null;
        
        // Store original emoji for comparison (for notifications)
        $originalEmoji = $employee ? $employee->emoji : null;
        $isNewEmployee = !$employee;

        if($employee) {
            if(isset($request->password)) {
                $pas = Hash::make($request->password);
            } else {
                $pas = $employee->password;
            }

            if (isset($request->image)) {
                // Delete old image if it exists
                if (!empty($employee->image) && $employee->image !== '/' && file_exists(public_path($employee->image))) {
                    unlink(public_path($employee->image));
                }
                // Always save the new image
                $image = saveFiles($request->image, 'Employees');
            } else {
                $image = $employee->image;
            }
        } else {
            $pas = Hash::make($request->password);
            if(isset($request->image)) {
                $image = saveFiles($request->image, 'Employees');
            } else {
                $image = null;
            }
        }

        $data = $request->only([
            'employee_id', 'name', 'email',
            'status', 'dob', 'emoji', 'phn', 'gender',
            'marital', 'nationality', 'city',
            'state', 'zip', 'address', 'role',
        ]);

        $data['password'] = $pas;
        $data['image'] = $image;
        $data['department_id'] = $request->department;
        $data['category'] = $request->category;
        $data['tenant_id'] = $this->mngr['tenant_id'];

        // Handle emoji - if no emoji provided, generate one based on name
        $emojiWasAutoGenerated = false;
        if (empty($data['emoji'])) {
            $data['emoji'] = $this->generateEmojiFromName($data['name']);
            $emojiWasAutoGenerated = true;
        }

        try {
            if($employee) {
                Employee::where('employee_id', $request->employee_id)->update($data);

                $count = Employee::where('tenant_id', $this->mngr['tenant_id'])
                    ->where('department_id', $request->department)->count();
                $test = Department::where('id', $request->department)->update(['no_of_emp' => $count]);

                // Send emoji change notification if emoji changed
                if ($originalEmoji !== $data['emoji']) {
                    Log::info('Emoji changed by manager, sending notification', [
                        'employee_id' => $employee->employee_id,
                        'old_emoji' => $originalEmoji,
                        'new_emoji' => $data['emoji'],
                        'tenant_id' => $employee->tenant_id,
                        'manager_id' => $this->mngr['employee_id']
                    ]);
                    
                    // Send emoji change notification
                    $this->sendEmojiChangeNotification($employee, $originalEmoji, $data['emoji'], $emojiWasAutoGenerated);
                }

                return response()->json([
                    'Error' => false,
                    'Message' => 'Employee Updated successfully!',
                    'NotificationSent' => ($originalEmoji !== $data['emoji']) ? 'Emoji change notification sent' : 'No emoji change',
                ]);
            } else {
                $newEmployee = Employee::create($data);

                // Handle custom emoji assignment if provided
                if ($request->has('emoji_id') && $request->emoji_id) {
                    $this->assignEmojiToEmployee($newEmployee, $request->emoji_id, $request->emoji_message);
                }

                $count = Employee::where('tenant_id', $this->mngr['tenant_id'])
                    ->where('department_id', $request->department)->count();
                $test = Department::where('id', $request->department)->update(['no_of_emp' => $count]);

                Log::info('New employee created by manager, sending welcome notification', [
                    'employee_id' => $newEmployee->employee_id,
                    'emoji' => $data['emoji'],
                    'tenant_id' => $newEmployee->tenant_id,
                    'manager_id' => $this->mngr['employee_id']
                ]);
                
                // Send welcome notification with emoji for new employee
                $this->sendWelcomeNotification($newEmployee, $data['emoji'], $emojiWasAutoGenerated);

                return response()->json([
                    'Error' => false,
                    'Message' => 'Employee Added successfully!',
                    'NotificationSent' => 'Welcome notification sent with emoji: ' . $data['emoji'],
                ]);
            }

        } catch(\Throwable $t) {
            Log::error('Manager employee operation failed', [
                'error' => $t->getMessage(),
                'line' => $t->getLine(),
                'manager_id' => $this->mngr['employee_id'],
                'trace' => $t->getTraceAsString()
            ]);
            
            return response()->json([
                'Error' => true,
                'Message' => "Line no: " . $t->getLine() . ", Msg: " . $t->getMessage(),
            ]);
        }
    }

    /**
     * Generate an emoji based on the employee's name
     * This is a simple implementation - you can make it more sophisticated
     */
    private function generateEmojiFromName($name)
    {
        $emojis = ['😊', '😎', '🤓', '😄', '🙂', '😇', '🤗', '😃', '😁', '🤩', '🥳', '😌', '🤠', '👨‍💻', '👩‍💻', '👨‍💼', '👩‍💼'];
        
        // Use the first character of the name to determine emoji
        $nameLength = strlen($name);
        $index = $nameLength % count($emojis);
        
        return $emojis[$index];
    }

    // Send welcome notification for new employee
    private function sendWelcomeNotification($employee, $emoji, $wasAutoGenerated)
    {
        try {
            $emojiSource = $wasAutoGenerated ? 'automatically assigned' : 'set';
            $content = $wasAutoGenerated 
                ? "Welcome to the team! Your manager has assigned you a special emoji {$emoji} based on your name. This emoji represents you across the platform!"
                : "Welcome to the team! Your manager has set your profile emoji {$emoji}. This emoji represents you across the platform!";

            // Send welcome notification
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin', // Manager acts as admin
                'sender_id' => $this->mngr['employee_id'],
                'sender_name' => $this->mngr['name'] ?? 'Manager',
                'title' => "Welcome! Your emoji is ready {$emoji}",
                'content' => $content,
                'type' => 'welcome_emoji',
                'priority' => 'high',
                'related_id' => $employee->employee_id
            ]);

            // Send celebratory emoji notification
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->mngr['employee_id'],
                'sender_name' => $this->mngr['name'] ?? 'Manager',
                'emoji' => $emoji . '🎉',
                'message' => 'Welcome to the team! Here\'s your special emoji!',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Manager welcome notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'manager_id' => $this->mngr['employee_id'],
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);

            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send manager welcome notification', [
                'employee_id' => $employee->employee_id,
                'manager_id' => $this->mngr['employee_id'],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    // Send emoji change notification for existing employee
    private function sendEmojiChangeNotification($employee, $oldEmoji, $newEmoji, $wasAutoGenerated)
    {
        try {
            $changeType = $wasAutoGenerated ? 'automatically updated' : 'updated';
            $content = $wasAutoGenerated 
                ? "Hi {$employee->name}! Your manager has automatically updated your profile emoji from {$oldEmoji} to {$newEmoji} based on your profile changes."
                : "Hi {$employee->name}! Your manager has updated your profile emoji from {$oldEmoji} to {$newEmoji}. Your new emoji represents you across the platform!";

            // Send change notification
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->mngr['employee_id'],
                'sender_name' => $this->mngr['name'] ?? 'Manager',
                'title' => "Your emoji has been {$changeType}! {$newEmoji}",
                'content' => $content,
                'type' => 'emoji_update',
                'priority' => 'normal',
                'related_id' => $employee->employee_id
            ]);

            // Send new emoji as reaction
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->mngr['employee_id'],
                'sender_name' => $this->mngr['name'] ?? 'Manager',
                'emoji' => $newEmoji . '✨',
                'message' => 'Your new emoji is ready!',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Manager emoji change notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'manager_id' => $this->mngr['employee_id'],
                'old_emoji' => $oldEmoji,
                'new_emoji' => $newEmoji,
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);

            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send manager emoji change notification', [
                'employee_id' => $employee->employee_id,
                'manager_id' => $this->mngr['employee_id'],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    // H - B : Delete Employee (actually removes from department)
    public function deleteEmployee(Request $request)
    {
        $employee = Employee::where('employee_id', $request->id)->first();

        try {
            if ($employee) {
                // Send removal notification before updating
                $notificationSent = $this->sendRemovalNotification($employee);

                $employee->update([
                    'department_id' => null,
                    'category' => null,
                    'status' => '1',
                    'role' => 'Employee',
                ]);

                if($employee) {
                    return response()->json([
                        'Error' => false,
                        'Message' => 'Employee removed successfully',
                        'NotificationSent' => $notificationSent ? 'Removal notification sent' : 'Notification failed',
                    ]);
                } else {
                    return response()->json([
                        'Error' => false,
                        'Message' => 'Error : Attempt Failed!',
                    ]);
                }
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Employee Record not found!',
                ], 404);
            }
        } catch (\Throwable $t) {
            Log::error('Manager employee removal failed', [
                'employee_id' => $request->id,
                'manager_id' => $this->mngr['employee_id'],
                'error' => $t->getMessage()
            ]);
            
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage()
            ]);
        }
    }

    // Send removal notification when employee is removed from department
    private function sendRemovalNotification($employee)
    {
        try {
            $sent1 = $this->notificationService->sendToUser([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->mngr['employee_id'],
                'sender_name' => $this->mngr['name'] ?? 'Manager',
                'title' => 'Department Assignment Update 📋',
                'content' => "Hi {$employee->name}, your manager has updated your department assignment. Your emoji {$employee->emoji} remains with you. Please check with your manager for next steps.",
                'type' => 'department_change',
                'priority' => 'high',
                'related_id' => $employee->employee_id
            ]);

            // Send informational emoji
            $sent2 = $this->notificationService->sendEmoji([
                'tenant_id' => $employee->tenant_id,
                'receiver_type' => 'employee',
                'receiver_id' => $employee->employee_id,
                'sender_type' => 'admin',
                'sender_id' => $this->mngr['employee_id'],
                'sender_name' => $this->mngr['name'] ?? 'Manager',
                'emoji' => '📋💼',
                'message' => 'Department assignment updated by your manager',
                'related_id' => $employee->employee_id
            ]);

            Log::info('Manager removal notifications sent successfully', [
                'employee_id' => $employee->employee_id,
                'manager_id' => $this->mngr['employee_id'],
                'text_notification' => $sent1 ? 'success' : 'failed',
                'emoji_notification' => $sent2 ? 'success' : 'failed'
            ]);

            return $sent1 && $sent2;

        } catch (\Exception $e) {
            Log::error('Failed to send manager removal notification', [
                'employee_id' => $employee->employee_id,
                'manager_id' => $this->mngr['employee_id'],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Assign custom emoji/GIF to employee
     */
    private function assignEmojiToEmployee($employee, $emojiId, $message = null)
    {
        try {
            // Get the current user (Manager)
            $assignedBy = [
                'id' => $this->mngr['employee_id'],
                'type' => 'App\Models\Admin' // Managers are stored as Admins in the User model
            ];
            
            if ($assignedBy) {
                \App\Models\EmojiAssignment::create([
                    'assignable_id' => $employee->employee_id,
                    'assignable_type' => 'App\Models\Employee',
                    'assigned_by_id' => $assignedBy['id'],
                    'assigned_by_type' => $assignedBy['type'],
                    'emoji_id' => $emojiId,
                    'message' => $message,
                    'tenant_id' => $this->mngr['tenant_id']
                ]);

                Log::info('Custom emoji assigned to employee by manager', [
                    'employee_id' => $employee->employee_id,
                    'emoji_id' => $emojiId,
                    'assigned_by' => $assignedBy
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Failed to assign custom emoji to employee by manager', [
                'employee_id' => $employee->employee_id,
                'emoji_id' => $emojiId,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get emoji assignments for a specific employee
     */
    public function getEmojiAssignments($employeeId)
    {
        try {
            Log::info('Manager fetching emoji assignments', [
                'employee_id' => $employeeId,
                'tenant_id' => $this->mngr['tenant_id'],
                'manager_data' => $this->mngr
            ]);

            $assignments = \App\Models\EmojiAssignment::with('emoji')
                ->where('assignable_id', $employeeId)
                ->where('assignable_type', 'App\Models\Employee')
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->orderBy('created_at', 'desc')
                ->get();

            Log::info('Manager found assignments', [
                'count' => $assignments->count(),
                'assignments' => $assignments->toArray()
            ]);

            return response()->json([
                'success' => true,
                'assignments' => $assignments
            ]);
        } catch (\Exception $e) {
            Log::error('Manager error fetching emoji assignments', [
                'employee_id' => $employeeId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch emoji assignments'
            ], 500);
        }
    }

    /**
     * Assign emoji to employee (standalone assignment)
     */
    public function assignEmojiToEmployeeStandalone(Request $request)
    {
        try {
            $request->validate([
                'employee_id' => 'required|string',
                'emoji_id' => 'required|integer',
                'message' => 'nullable|string|max:500'
            ]);

            $employee = Employee::where('employee_id', $request->employee_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();

            if (!$employee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Employee not found'
                ], 404);
            }

            $emoji = \App\Models\CustomEmoji::where('id', $request->emoji_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->where('is_active', true)
                ->first();

            if (!$emoji) {
                return response()->json([
                    'success' => false,
                    'message' => 'Emoji/GIF not found or inactive'
                ], 404);
            }

            // Get the current user (Manager)
            $assignedBy = [
                'id' => $this->mngr['employee_id'],
                'type' => 'App\Models\Admin' // Managers are stored as Admins in the User model
            ];
            
            if (!$assignedBy) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unable to identify current user'
                ], 400);
            }

            // Create the assignment
            $assignment = \App\Models\EmojiAssignment::create([
                'assignable_id' => $employee->employee_id,
                'assignable_type' => 'App\Models\Employee',
                'assigned_by_id' => $assignedBy['id'],
                'assigned_by_type' => $assignedBy['type'],
                'emoji_id' => $request->emoji_id,
                'message' => $request->message,
                'tenant_id' => $this->mngr['tenant_id']
            ]);

            Log::info('Manager emoji assigned to employee via standalone assignment', [
                'employee_id' => $employee->employee_id,
                'emoji_id' => $request->emoji_id,
                'assigned_by' => $assignedBy,
                'assignment_id' => $assignment->id
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assigned successfully to ' . $employee->name,
                'assignment' => $assignment
            ]);

        } catch (\Exception $e) {
            Log::error('Manager failed to assign emoji to employee', [
                'employee_id' => $request->employee_id,
                'emoji_id' => $request->emoji_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to assign emoji/GIF: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove emoji assignment
     */
    public function removeEmojiAssignment(Request $request)
    {
        try {
            $request->validate([
                'assignment_id' => 'required|integer'
            ]);

            $assignment = \App\Models\EmojiAssignment::where('id', $request->assignment_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();

            if (!$assignment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Assignment not found'
                ], 404);
            }

            $employeeName = 'Unknown';
            if ($assignment->assignable_type === 'App\Models\Employee') {
                $employee = Employee::where('employee_id', $assignment->assignable_id)
                    ->where('tenant_id', $this->mngr['tenant_id'])
                    ->first();
                if ($employee) {
                    $employeeName = $employee->name;
                }
            }

            $assignment->delete();

            Log::info('Manager emoji assignment removed', [
                'assignment_id' => $request->assignment_id,
                'employee_name' => $employeeName
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assignment removed successfully from ' . $employeeName
            ]);

        } catch (\Exception $e) {
            Log::error('Manager failed to remove emoji assignment', [
                'assignment_id' => $request->assignment_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to remove emoji assignment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update existing emoji assignment
     */
    public function updateEmojiAssignment(Request $request)
    {
        try {
            $request->validate([
                'assignment_id' => 'required|integer',
                'emoji_id' => 'required|integer',
                'message' => 'nullable|string|max:500'
            ]);

            $assignment = \App\Models\EmojiAssignment::where('id', $request->assignment_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->first();

            if (!$assignment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Assignment not found'
                ], 404);
            }

            $emoji = \App\Models\CustomEmoji::where('id', $request->emoji_id)
                ->where('tenant_id', $this->mngr['tenant_id'])
                ->where('is_active', true)
                ->first();

            if (!$emoji) {
                return response()->json([
                    'success' => false,
                    'message' => 'Emoji/GIF not found or inactive'
                ], 404);
            }

            // Update the assignment
            $assignment->update([
                'emoji_id' => $request->emoji_id,
                'message' => $request->message
            ]);

            Log::info('Manager emoji assignment updated', [
                'assignment_id' => $assignment->id,
                'new_emoji_id' => $request->emoji_id,
                'message' => $request->message
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Emoji/GIF assignment updated successfully',
                'assignment' => $assignment->load('emoji')
            ]);

        } catch (\Exception $e) {
            Log::error('Manager failed to update emoji assignment', [
                'assignment_id' => $request->assignment_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update emoji assignment: ' . $e->getMessage()
            ], 500);
        }
    }
}