<?php

namespace App\Http\Controllers\Manager;

use App\Http\Controllers\Controller;
use App\Models\Target;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DepTargetsController extends Controller
{

    protected $mngr;

    //H - B : Constructor
    public function __construct()
    {
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->mngr = session('ismngr');
            return $next($request);
        });
    }

    //H - B : Add and Edit Target
    public function addAndEditTarget(Request $request)
    {
        $validated = $request->validate([
            'department_id' => 'required',
            'category' => 'required',
            'quantity' => 'required',
            'last_date' => 'required',
            'status' => 'required',
        ]);

        $trgt = Target::find($request->id);
        
        try {
            if($request->custom != null)
            {
                $match = DB::table('targets')
                ->where('custom', $request->custom)
                ->where('department_id', $request->department_id)
                ->whereBetween('last_date', [
                    Carbon::now()->startOfMonth(),
                    Carbon::now()->endOfMonth()
                ])
                ->whereIn('status', ['2', '0'])
                ->exists(); // Check if any record exists
            if($match)
            {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Target with the same name already exist, Custom Target must be renamed.',
                ]);
            }
            }
        } catch (\Exception $e)
        {
            return response()->json([
                'Error' => true,
                'Message' => 'Invalid target Matching',
            ]);
        }

        $formattedDate = Carbon::createFromFormat('m/d/Y', $request->last_date)->format('Y-m-d');
        $data = $request->only([
            'department_id',
            'category',
            'product_id',
            'custom',
            'quantity',
            'status'
        ]);
        $data['last_date'] = $formattedDate;
        $data['tenant_id'] = $this->mngr['tenant_id'];

        try {
            if ($trgt) {
                $attempt = $trgt->update($data);
            } else {
                $attempt = Target::create($data);
            }

            if ($attempt) {
                return response()->json([
                    'Error' => false,
                    'Message' => "Target Assigned Successfully!",
                ]);
            } else {
                return response()->json([
                    'Error' => true,
                    'Message' => "Target Assignment Failed!",
                ]);
            }
        } catch (\Throwable $t)
        {
            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }

    }

    // H - B : Delete Target
    public function deleteTarget(Request $request)
    {
        $trgt = Target::find($request->id);
        if(!$trgt)
        {
            return response()->json([
                'Error'=>true,
                'Message'=>"Target Not Found!"
            ]);
        }
        else
        {
            $trgt->update(['status' => '1']);
            return response()->json([
                'Error'=>false,
                'Message'=>"Target Removed Successfully!"
            ]);
        }
    }

    // H - B : Renew Target
    public function renewTarget(Request $request)
    {
        $trgt = Target::find($request->id);
        if(!$trgt)
        {
            return response()->json([
                'Error'=>true,
                'Message'=>"Target Not Found!"
            ]);
        }
        else
        {
            if($trgt->progress == $trgt->quantity && $trgt->completion_date != null)
            {
                $daysDifference = $trgt->created_at->diffInDays($trgt->last_date);
                $newDate = Carbon::now()->addDays($daysDifference + 1)->format('Y-m-d');

                $data = [
                    'department_id' => $trgt->department_id,
                    'category' => $trgt->category,
                    'product_id' => $trgt->product_id,
                    'custom' => $trgt->custom,
                    'quantity' => $trgt->quantity,
                    'status' => '2'
                ];
                $data['last_date'] = $newDate;
                $data['tenant_id'] = $this->mngr['tenant_id'];

                //Created New Target With Same info
                $attempt = Target::create($data);
                //Making base target inactive
                $inactive = $trgt->update(['status' => '1']);

                if($attempt && $inactive)
                {
                    return response()->json([
                        'Error'=>false,
                        'Message'=>"Target Renewed Successfully!"
                    ]);
                }
                else
                {
                    return response()->json([
                        'Error'=>true,
                        'Message'=>"Target Renewal Failed!"
                    ]);
                }
            }
            else
            {
                return response()->json([
                    'Error'=>true,
                    'Message'=>"For Renewal, Target Must Be Completed First!"
                ]);
            }
        }
    }
}
