<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\Points;
use App\Models\Target;
use App\Services\EmployeePointsNotificationService;
use App\Services\PointsEmailNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PointsController extends Controller
{
    protected $emp;
    private EmployeePointsNotificationService $employeeNotificationService;
    private PointsEmailNotificationService $emailNotificationService;

    //H - B : Constructor
    public function __construct(
        EmployeePointsNotificationService $employeeNotificationService,
        PointsEmailNotificationService $emailNotificationService
    ) {
        $this->employeeNotificationService = $employeeNotificationService;
        $this->emailNotificationService = $emailNotificationService;
        
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->emp = session('isemp');
            return $next($request);
        });
    }

    // H-B : Employee Add Points
    public function addPoints(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required',
            'vin_no' => 'required',
            'target_id' => 'required | array',
        ]);

        $data = [
            'employee_id'=>$request->employee_id,
            'department_id'=> $this->emp['department_id'],
            'category'=> $this->emp['category'],
            'vin_no'=>$request->vin_no,
            'status'=>'Pending',
            'detail'=>null,
        ];
        $data['tenant_id'] = $this->emp['tenant_id'];
        $targets = $request->target_id;
        $points = array_count_values($targets);

        try {
            $totalPoints = 0;
            $targetNames = [];
            $done = [];

            foreach($points as $target => $point)
            {
                $data['target_id'] = $target;
                $data['points'] = $point;
                $totalPoints += $point;
                
                // Get target name for notification
                $targetInfo = Target::where('id', $target)
                                   ->where('tenant_id', $this->emp['tenant_id'])
                                   ->first();
                if ($targetInfo) {
                    $targetNames[] = $targetInfo->name ?? "Target #{$target}";
                }
                
                $done[] = Points::create($data);
            }
            if($done)
            {
                // Send push notifications (existing functionality)
                $this->employeeNotificationService->sendEmployeeRequestConfirmation(
                    $request->employee_id,
                    $this->emp['tenant_id'],
                    $totalPoints,
                    $targetNames
                );

                $this->employeeNotificationService->notifyAdminsAboutPointsRequest(
                    $request->employee_id,
                    $this->emp['tenant_id'],
                    $totalPoints,
                    $targetNames,
                    $request->vin_no
                );

                // Send email notifications (new functionality)
                $this->emailNotificationService->sendPointsRequestSubmittedNotification(
                    $request->employee_id,
                    $this->emp['tenant_id'],
                    $totalPoints,
                    $targetNames,
                    $request->vin_no
                );

                return response()->json([
                    'Error' => false,
                    'Message' => 'Points Request Generated Successfully!',
                    'Details' => [
                        'total_points' => $totalPoints,
                        'targets' => count($targetNames),
                        'status' => 'Pending Admin Approval',
                        'vin_no' => $request->vin_no
                    ]
                ]);
            }
            else
            {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Request Generation Failed!',
                ]);
            }

        } catch(\Throwable $t) {
            Log::error('Points request creation failed', [
                'employee_id' => $request->employee_id,
                'vin_no' => $request->vin_no,
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => $t->getMessage(),
            ]);
        }
    }

    /**
     * Get employee's points history with status
     */
    public function getPointsHistory(Request $request)
    {
        try {
            $points = Points::where('employee_id', $this->emp['employee_id'])
                           ->where('tenant_id', $this->emp['tenant_id'])
                           ->with(['target' => function($query) {
                               $query->select('id', 'name', 'quantity', 'progress');
                           }])
                           ->orderBy('created_at', 'desc')
                           ->paginate(20);

            return response()->json([
                'Error' => false,
                'Data' => $points,
                'Message' => 'Points history retrieved successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to retrieve points history', [
                'employee_id' => $this->emp['employee_id'],
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Failed to retrieve points history: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get employee's pending points requests
     */
    public function getPendingRequests(Request $request)
    {
        try {
            $pendingPoints = Points::where('employee_id', $this->emp['employee_id'])
                                  ->where('tenant_id', $this->emp['tenant_id'])
                                  ->where('status', 'Pending')
                                  ->with(['target' => function($query) {
                                      $query->select('id', 'name', 'quantity', 'progress');
                                  }])
                                  ->orderBy('created_at', 'desc')
                                  ->get();

            $totalPendingPoints = $pendingPoints->sum('points');

            return response()->json([
                'Error' => false,
                'Data' => $pendingPoints,
                'Count' => $pendingPoints->count(),
                'TotalPendingPoints' => $totalPendingPoints,
                'Message' => 'Pending requests retrieved successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to retrieve pending requests', [
                'employee_id' => $this->emp['employee_id'],
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Failed to retrieve pending requests: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Cancel a pending points request
     */
    public function cancelPointsRequest(Request $request)
    {
        $validated = $request->validate([
            'points_id' => 'required|exists:points,id'
        ]);

        try {
            $pointsRecord = Points::where('id', $request->points_id)
                                 ->where('employee_id', $this->emp['employee_id'])
                                 ->where('tenant_id', $this->emp['tenant_id'])
                                 ->where('status', 'Pending')
                                 ->with('target')
                                 ->first();

            if (!$pointsRecord) {
                return response()->json([
                    'Error' => true,
                    'Message' => 'Points request not found or cannot be cancelled'
                ]);
            }

            // Update status to cancelled
            $pointsRecord->update([
                'status' => 'Cancelled',
                'detail' => 'Cancelled by employee on ' . now()->format('Y-m-d H:i:s')
            ]);

            // Get target name for notification
            $targetName = $pointsRecord->target 
                ? $pointsRecord->target->name 
                : "Target #{$pointsRecord->target_id}";

            // Notify admins about cancellation
            $this->employeeNotificationService->notifyAdminsAboutCancellation(
                $this->emp['employee_id'],
                $this->emp['tenant_id'],
                $pointsRecord->points,
                $targetName,
                $pointsRecord->vin_no
            );

            return response()->json([
                'Error' => false,
                'Message' => 'Points request cancelled successfully',
                'Details' => [
                    'cancelled_points' => $pointsRecord->points,
                    'target_name' => $targetName,
                    'vin_no' => $pointsRecord->vin_no
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to cancel points request', [
                'employee_id' => $this->emp['employee_id'],
                'points_id' => $request->points_id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Failed to cancel request: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get employee's points summary
     */
    public function getPointsSummary(Request $request)
    {
        try {
            $summary = [
                'total_earned' => Points::where('employee_id', $this->emp['employee_id'])
                                       ->where('tenant_id', $this->emp['tenant_id'])
                                       ->where('status', 'Redeemed')
                                       ->sum('points'),
                'pending_points' => Points::where('employee_id', $this->emp['employee_id'])
                                         ->where('tenant_id', $this->emp['tenant_id'])
                                         ->where('status', 'Pending')
                                         ->sum('points'),
                'pending_requests' => Points::where('employee_id', $this->emp['employee_id'])
                                           ->where('tenant_id', $this->emp['tenant_id'])
                                           ->where('status', 'Pending')
                                           ->count(),
                'rejected_requests' => Points::where('employee_id', $this->emp['employee_id'])
                                            ->where('tenant_id', $this->emp['tenant_id'])
                                            ->where('status', 'Rejected')
                                            ->count(),
                'total_requests' => Points::where('employee_id', $this->emp['employee_id'])
                                         ->where('tenant_id', $this->emp['tenant_id'])
                                         ->count(),
                'approved_requests' => Points::where('employee_id', $this->emp['employee_id'])
                                            ->where('tenant_id', $this->emp['tenant_id'])
                                            ->where('status', 'Redeemed')
                                            ->count(),
                'cancelled_requests' => Points::where('employee_id', $this->emp['employee_id'])
                                             ->where('tenant_id', $this->emp['tenant_id'])
                                             ->where('status', 'Cancelled')
                                             ->count()
            ];

            // Calculate success rate
            $summary['approval_rate'] = $summary['total_requests'] > 0 
                ? round(($summary['approved_requests'] / $summary['total_requests']) * 100, 2)
                : 0;

            return response()->json([
                'Error' => false,
                'Data' => $summary,
                'Message' => 'Points summary retrieved successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to retrieve points summary', [
                'employee_id' => $this->emp['employee_id'],
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Failed to retrieve summary: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Submit bulk points requests for multiple targets
     */
  /**
     * Submit bulk points requests for multiple targets
     * Updated to send category-filtered notifications
     */
    public function submitBulkPointsRequest(Request $request)
    {
        dd($request->all());
        $validated = $request->validate([
            'requests' => 'required|array|min:1|max:10', // Limit to 10 requests at once
            'requests.*.vin_no' => 'required|string',
            'requests.*.target_id' => 'required|exists:targets,id',
            'requests.*.points' => 'required|integer|min:1|max:1000',
            'requests.*.description' => 'nullable|string|max:500'
        ]);

        try {
            $successfulRequests = [];
            $failedRequests = [];
            $totalPoints = 0;
            $targetNames = [];

            foreach ($request->requests as $index => $requestData) {
                try {
                    // Verify target belongs to tenant
                    $target = Target::where('id', $requestData['target_id'])
                                   ->where('tenant_id', $this->emp['tenant_id'])
                                   ->first();
                    dd($target);
                    if (!$target) {
                        $failedRequests[] = [
                            'index' => $index,
                            'reason' => 'Target not found or not accessible'
                        ];
                        continue;
                    }

                    $pointData = [
                        'employee_id' => $this->emp['employee_id'],
                        'department_id' => $this->emp['department_id'],
                        'category' => $this->emp['category'],
                        'vin_no' => $requestData['vin_no'],
                        'target_id' => $requestData['target_id'],
                        'points' => $requestData['points'],
                        'tenant_id' => $this->emp['tenant_id'],
                        'status' => 'Pending',
                        'detail' => $requestData['description'] ?? null,
                    ];

                    $pointsRecord = Points::create($pointData);
                    $successfulRequests[] = [
                        'index' => $index,
                        'points_id' => $pointsRecord->id,
                        'target_name' => $target->name,
                        'points' => $requestData['points']
                    ];

                    $totalPoints += $requestData['points'];
                    $targetNames[] = $target->name ?? "Target #{$target->id}";

                } catch (\Exception $e) {
                    $failedRequests[] = [
                        'index' => $index,
                        'reason' => $e->getMessage()
                    ];
                }
            }

            // Send notifications if there were successful requests
            if (!empty($successfulRequests)) {
                // Send confirmation to employee
                $this->employeeNotificationService->sendEmployeeRequestConfirmation(
                    $this->emp['employee_id'],
                    $this->emp['tenant_id'],
                    $totalPoints,
                    array_unique($targetNames)
                );

                // Create a summary VIN for bulk notification
                $bulkVinNo = 'BULK_' . time() . '_' . count($successfulRequests) . '_REQUESTS';
                
                // Notify admins about bulk request (category-filtered)
                $this->employeeNotificationService->notifyAdminsAboutBulkPointsRequest(
                    $this->emp['employee_id'],
                    $this->emp['tenant_id'],
                    $totalPoints,
                    array_unique($targetNames),
                    $bulkVinNo,
                    count($successfulRequests)
                );

                // Send email notifications for bulk points request
                $this->emailNotificationService->sendPointsRequestSubmittedNotification(
                    $this->emp['employee_id'],
                    $this->emp['tenant_id'],
                    $totalPoints,
                    array_unique($targetNames),
                    $bulkVinNo
                );
            }

            $message = count($successfulRequests) > 0 
                ? "Bulk request completed. " . count($successfulRequests) . " requests submitted successfully."
                : "Bulk request failed. No requests were processed.";

            if (!empty($failedRequests)) {
                $message .= " " . count($failedRequests) . " requests failed.";
            }

            return response()->json([
                'Error' => empty($successfulRequests),
                'Message' => $message,
                'Details' => [
                    'successful_requests' => count($successfulRequests),
                    'failed_requests' => count($failedRequests),
                    'total_points_requested' => $totalPoints,
                    'unique_targets' => count(array_unique($targetNames)),
                    'successful_details' => $successfulRequests,
                    'failed_details' => $failedRequests,
                    'employee_category' => $this->emp['category'],
                    'notified_admin_category' => $this->emp['category']
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Bulk points request failed', [
                'employee_id' => $this->emp['employee_id'],
                'employee_category' => $this->emp['category'],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Bulk request failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get recent activity (last 30 days)
     */
    public function getRecentActivity(Request $request)
    {
        try {
            $recentActivity = Points::where('employee_id', $this->emp['employee_id'])
                                   ->where('tenant_id', $this->emp['tenant_id'])
                                   ->where('created_at', '>=', now()->subDays(30))
                                   ->with(['target' => function($query) {
                                       $query->select('id', 'name');
                                   }])
                                   ->orderBy('created_at', 'desc')
                                   ->limit(50)
                                   ->get()
                                   ->map(function($point) {
                                       return [
                                           'id' => $point->id,
                                           'points' => $point->points,
                                           'status' => $point->status,
                                           'target_name' => $point->target ? $point->target->name : 'Unknown Target',
                                           'vin_no' => $point->vin_no,
                                           'created_at' => $point->created_at->format('Y-m-d H:i:s'),
                                           'days_ago' => $point->created_at->diffInDays(now()),
                                           'detail' => $point->detail
                                       ];
                                   });

            return response()->json([
                'Error' => false,
                'Data' => $recentActivity,
                'Count' => $recentActivity->count(),
                'Message' => 'Recent activity retrieved successfully'
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to retrieve recent activity', [
                'employee_id' => $this->emp['employee_id'],
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'Error' => true,
                'Message' => 'Failed to retrieve recent activity: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Request reminder about pending requests (employee can trigger this)
     */
    public function requestPendingReminder(Request $request)
    {
        try {
            $pendingRequests = Points::where('employee_id', $this->emp['employee_id'])
                                    ->where('tenant_id', $this->emp['tenant_id'])
                                    ->where('status', 'Pending')
                                    ->get();

            if ($pendingRequests->isEmpty()) {
                return response()->json([
                    'Error' => false,
                    'Message' => 'You have no pending points requests'
                ]);
            }

            $totalPendingPoints = $pendingRequests->sum('points');
            
            $this->employeeNotificationService->sendPendingRequestReminder(
                $this->emp['employee_id'],
                $this->emp['tenant_id'],
                $pendingRequests->count(),
                $totalPendingPoints
            );

            return response()->json([
                'Error' => false,
                'Message' => 'Reminder sent successfully',
                'Details' => [
                    'pending_requests' => $pendingRequests->count(),
                    'total_pending_points' => $totalPendingPoints
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'Error' => true,
                'Message' => 'Failed to send reminder: ' . $e->getMessage()
            ]);
        }
    }
}