<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use App\Models\CategoryLeaderBoard;
use App\Models\DepartmentLeaderBoard;
use App\Models\EmployeeLeaderBoard;
use App\Models\Points;
use App\Models\Product;
use App\Models\Target;
use App\Models\Department;
use App\Services\LeaderboardService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class HomeController extends Controller
{
    protected $emp;
    protected $leaderboardService;

    //H - B : Constructor
    public function __construct()
    {
        // Use middleware to delay session initialization
        $this->middleware(function ($request, $next) {
            $this->emp = session('isemp');
            if ($this->emp) {
                $this->leaderboardService = new LeaderboardService($this->emp['tenant_id']);
            }
            return $next($request);
        });
    }

    // H-B : LeaderBoard
    public function viewLeaderBoard()
    {
        $currentMonth = Carbon::now()->format('Y-m');

        // Get available months with data
        $months = collect([
            EmployeeLeaderBoard::where('tenant_id', $this->emp['tenant_id'])->select('month_year')->distinct()->pluck('month_year'),
            DepartmentLeaderBoard::where('tenant_id', $this->emp['tenant_id'])->select('month_year')->distinct()->pluck('month_year'),
            CategoryLeaderBoard::where('tenant_id', $this->emp['tenant_id'])->select('month_year')->distinct()->pluck('month_year')
        ])->flatten()->unique()->sort()->reverse()->values();

        // If no months found, add current month
        if ($months->isEmpty()) {
            $months = collect([$currentMonth]);
        }

        // Get employees data
        $employees = $this->getEmployeeLeaderboardData($currentMonth);

        // Get departments data  
        $departments = $this->getDepartmentLeaderboardData($currentMonth);

        // Get categories data
        $categories = $this->getCategoryLeaderboardData($currentMonth);

        return view('Employee.Leaderboard', compact('employees', 'categories', 'departments', 'months'));
    }

    // H-B: Employee Filter LeaderBoard
    public function viewEmpLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"

        try {
            $employees = $this->getEmployeeLeaderboardData($currentMonth);
            return view('partials.employeeLeaderBoard', compact('employees'))->render();
        } catch (\Exception $e) {
            Log::error('Error in viewEmpLeaderBoard: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to load employee leaderboard'], 500);
        }
    }

    // H-B: Department Filter LeaderBoard
    public function viewDepLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"
        try {
            $departments = $this->getDepartmentLeaderboardData($currentMonth);
            return view('partials.departmentLeaderBoard', compact('departments'))->render();
        } catch (\Exception $e) {
            Log::error('Error in viewDepLeaderBoard: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to load department leaderboard'], 500);
        }
    }

    // H-B: Category Filter LeaderBoard
    public function viewCatLeaderBoard($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"

        try {
            $categories = $this->getCategoryLeaderboardData($currentMonth);
            return view('partials.categoryLeaderBoard', compact('categories'))->render();
        } catch (\Exception $e) {
            Log::error('Error in viewCatLeaderBoard: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to load category leaderboard'], 500);
        }
    }

    // Helper method to get employee leaderboard data
    private function getEmployeeLeaderboardData($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"

        return EmployeeLeaderBoard::join('employees', function ($join) {
            $join->on('employee_leader_boards.employee_id', '=', 'employees.employee_id')
                ->on('employee_leader_boards.tenant_id', '=', 'employees.tenant_id');
        })
            ->where('employee_leader_boards.tenant_id', $this->emp['tenant_id'])
            ->where('employee_leader_boards.month_year', $currentMonth)
            ->orderBy('employee_leader_boards.rank', 'asc')
            ->get([
                    'employee_leader_boards.*',
                    'employees.name as employee',
                    'employees.category as category',
                    'employees.image as image',
                    'employees.emoji as emoji'
                ]);
    }

    // Helper method to get department leaderboard data
    private function getDepartmentLeaderboardData($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"

        return DepartmentLeaderBoard::join('departments', function ($join) {
            $join->on('department_leader_boards.department_id', '=', 'departments.id')
                ->on('department_leader_boards.tenant_id', '=', 'departments.tenant_id');
        })
            ->leftJoin('employees', function ($join) {
                $join->on('department_leader_boards.department_id', '=', 'employees.department_id')
                    ->on('department_leader_boards.tenant_id', '=', 'employees.tenant_id')
                    ->where('employees.role', 'Manager');
            })
            ->where('department_leader_boards.tenant_id', $this->emp['tenant_id'])
            ->where('department_leader_boards.month_year', $currentMonth)
            ->groupBy([
                    'department_leader_boards.id',
                    'department_leader_boards.rank',
                    'department_leader_boards.earned_points',
                    'departments.name'
                ])
            ->orderBy('department_leader_boards.rank', 'asc')
            ->get([
                    'department_leader_boards.id',
                    'department_leader_boards.rank',
                    'department_leader_boards.earned_points',
                    'departments.name as department',
                    DB::raw('COUNT(DISTINCT employees.employee_id) as count'),
                    DB::raw('MIN(employees.name) as manager'),
                    DB::raw('MIN(employees.image) as emp_image'),
                    DB::raw('MIN(employees.emoji) as emoji')
                ]);
    }

    // Helper method to get category leaderboard data
    private function getCategoryLeaderboardData($currentMonth)
    {
        $segments = request()->segments();
        $currentMonth = end($segments); // Get the last segment, e.g., "2025-07"

        return CategoryLeaderBoard::join('departments', function ($join) {
            $join->on('category_leader_board.department_id', '=', 'departments.id')
                ->on('category_leader_board.tenant_id', '=', 'departments.tenant_id');
        })
            ->where('category_leader_board.tenant_id', $this->emp['tenant_id'])
            ->where('category_leader_board.month_year', $currentMonth)
            ->orderBy('category_leader_board.rank', 'asc')
            ->get([
                    'category_leader_board.*',
                    'departments.name as name'
                ]);
    }

    // ... rest of your existing methods remain the same ...

    // H - B : Employee Dashboard
    public function empDashboard()
    {
        $points = Points::leftJoin('targets', function ($join) {
            $join->on('points.target_id', '=', 'targets.id')
                ->where('targets.status', '<>', 1)
                ->on('points.tenant_id', '=', 'targets.tenant_id');
        })
            ->leftJoin('products', function ($join) {
                $join->on('targets.product_id', '=', 'products.id')
                    ->where('targets.status', '<>', 1)
                    ->on('targets.tenant_id', '=', 'products.tenant_id');
            })
            ->where('points.department_id', $this->emp['department_id'])
            ->where('points.category', $this->emp['category'])
            ->where('points.employee_id', $this->emp['employee_id'])
            ->where('points.status', 'Redeemed')
            ->where('points.tenant_id', $this->emp['tenant_id'])
            ->get([
                    'points.*',
                    'products.name as name',
                    'targets.custom as custom'
                ]);

        $claims = Points::join('targets', function ($join) {
            $join->on('points.target_id', '=', 'targets.id')
                ->where('targets.status', '!=', 1)
                ->on('points.tenant_id', '=', 'targets.tenant_id');
        })
            ->leftJoin('products', function ($join) {
                $join->on('targets.product_id', '=', 'products.id')
                    ->where('targets.status', '!=', 1)
                    ->on('targets.tenant_id', '=', 'products.tenant_id');
            })
            ->where('points.employee_id', $this->emp['employee_id'])
            ->where('points.status', 'redeemed')
            ->where('targets.status', '!=', '1')
            ->where('points.tenant_id', $this->emp['tenant_id'])
            ->groupBy('points.target_id', 'targets.quantity', 'points.employee_id', 'targets.custom', 'products.name')
            ->orderBy('points.target_id')
            ->get([
                    'points.target_id',
                    DB::raw('SUM(points.points) as redeemed_points'),
                    'targets.quantity',
                    'points.employee_id',
                    'targets.custom as custom',
                    'products.name as name'
                ]);

        $targets = Target::leftJoin('products', function ($join) {
            $join->on('targets.product_id', '=', 'products.id')
                ->where('targets.status', '<>', '1')
                ->on('targets.tenant_id', '=', 'products.tenant_id');
        })
            ->where('targets.department_id', $this->emp['department_id'])
            ->where('targets.category', $this->emp['category'])
            ->where('targets.status', '<>', '1')
            ->where('targets.tenant_id', $this->emp['tenant_id'])
            ->where(function ($query) {
                $query->whereNull('parent_target')
                    ->orWhere('parent_target', '!=', 'parent');
            })
            ->get([
                    'targets.*',
                    'products.name as name'
                ]);

        $piePoints = Points::select(
            DB::raw("SUM(CASE WHEN status = 'Redeemed' THEN points ELSE 0 END) as redeemed"),
            DB::raw("SUM(CASE WHEN status = 'Pending' THEN points ELSE 0 END) as pending")
        )
            ->where('tenant_id', $this->emp['tenant_id'])
            ->where('employee_id', $this->emp['employee_id'])
            ->first();

        $piePoints->total_points = $piePoints->redeemed + $piePoints->pending;

        $currentDate = now()->format('F d, Y');
        $employee = $this->emp;

        if (!isset($employee['emoji']) || empty($employee['emoji'])) {
            $employee['emoji'] = $this->getEmployeeEmoji($employee['employee_id']);
        }

        // Get assigned custom emojis/GIFs for this employee
        $assignedEmojis = $this->getAssignedEmojis($employee['employee_id']);

        return view('Employee.Dashboard', compact('employee', 'points', 'claims', 'piePoints', 'currentDate', 'targets', 'assignedEmojis'));
    }

    private function getEmployeeEmoji($employeeId)
    {
        $employeeRecord = \App\Models\Employee::where('employee_id', $employeeId)
            ->where('tenant_id', $this->emp['tenant_id'])
            ->first(['emoji']);

        return $employeeRecord->emoji ?? '😊';
    }

    private function getAssignedEmojis($employeeId)
    {
        // Get assigned emojis/GIFs for this employee directly
        return \App\Models\EmojiAssignment::with('emoji')
            ->where('assignable_id', $employeeId)
            ->where('assignable_type', 'App\Models\Employee')
            ->where('tenant_id', $this->emp['tenant_id'])
            ->orderBy('created_at', 'desc')
            ->get();
    }

    // H - B : Filter On Dashboard
    public function empPointsData(Request $request)
    {
        $filter = $request->input('filter');

        $query = Points::where('tenant_id', $this->emp['tenant_id'])
            ->where('employee_id', $this->emp['employee_id'])
            ->where('status', 'Redeemed')
            ->select(DB::raw('SUM(points) as total_points'));

        switch ($filter) {
            case 'today':
                $query->addSelect(DB::raw('HOUR(updated_at) as time_group'))
                    ->whereDate('updated_at', today())
                    ->groupBy('time_group')
                    ->orderBy('time_group');
                break;
            case 'week':
                $query->addSelect(DB::raw('DATE(updated_at) as time_group'))
                    ->whereBetween('updated_at', [now()->subDays(6)->startOfDay(), now()->endOfDay()])
                    ->groupBy('time_group')
                    ->orderBy('time_group', 'desc');
                break;
            case 'month':
                $query->addSelect(DB::raw('DATE(updated_at) as time_group'))
                    ->whereMonth('updated_at', now()->month)
                    ->groupBy('time_group')
                    ->orderBy('time_group');
                break;
            case '6_months':
                $query->addSelect(DB::raw('MONTH(updated_at) as time_group, YEAR(updated_at) as year_group'))
                    ->whereBetween('updated_at', [now()->subMonths(6), now()])
                    ->groupBy('year_group', 'time_group')
                    ->orderByRaw('year_group, time_group');
                break;
            case 'year':
                $query->addSelect(DB::raw('MONTH(updated_at) as time_group, YEAR(updated_at) as year_group'))
                    ->whereYear('updated_at', now()->year)
                    ->groupBy('year_group', 'time_group')
                    ->orderByRaw('year_group, time_group');
                break;
        }

        $data = $query->get();

        if (in_array($filter, ['6_months', 'year'])) {
            $data->transform(function ($item) {
                $item->time_group = date('M Y', mktime(0, 0, 0, $item->time_group, 1, $item->year_group));
                unset($item->year_group);
                return $item;
            });
        }

        return response()->json($data);
    }

    // H-B : Employee Points
    public function empPoints()
    {
        $employee = $this->emp;

        if (!isset($employee['emoji']) || empty($employee['emoji'])) {
            $employee['emoji'] = $this->getEmployeeEmoji($employee['employee_id']);
        }

        $products = Product::where('tenant_id', $this->emp['tenant_id'])->get();
        $targets = Target::where('tenant_id', $this->emp['tenant_id'])
            ->where('category', $this->emp['category'])
            ->where(function ($query) {
                $query->whereNull('parent_target')
                    ->orWhere('parent_target', '!=', 'parent');
            })
            ->get();
        $points = Points::where('tenant_id', $this->emp['tenant_id'])
            ->where('employee_id', $this->emp['employee_id'])->get();
        return view('Employee.claimPoints', compact('employee', 'points', 'targets', 'products'));
    }

    // Enhanced Leaderboard Methods
    
    /**
     * Enhanced leaderboard view with all filters
     */
    public function viewEnhancedLeaderboard()
    {
        $currentMonth = Carbon::now()->format('Y-m');
        
        // Get available months
        $months = $this->leaderboardService->getAvailableMonths();
        
        // Get filter options
        $departments = $this->leaderboardService->getAvailableDepartments();
        $categories = $this->leaderboardService->getAvailableCategories();
        
        // Get initial data
        $leaderboardData = $this->leaderboardService->getLeaderboardData([
            'category' => 'employees',
            'time_period' => 'current_month'
        ]);
        
        return view('Employee.EnhancedLeaderboard', compact(
            'leaderboardData', 
            'months', 
            'departments', 
            'categories'
        ));
    }
    
    /**
     * AJAX endpoint for filtered leaderboard data
     */
    public function getFilteredLeaderboard(Request $request)
    {
        $filters = $request->only(['category', 'time_period', 'department_id', 'custom_date_range']);
        
        try {
            $data = $this->leaderboardService->getLeaderboardData($filters);
            $prizes = $this->leaderboardService->getPrizesForLeaderboard(
                $filters['category'] ?? 'all',
                $filters['time_period'] ?? 'current_month'
            );
            
            return response()->json([
                'success' => true,
                'data' => $data,
                'top3' => $data->take(3),
                'complete_ranking' => $data->skip(3),
                'total_count' => $data->count(),
                'prizes' => $prizes
            ]);
        } catch (\Exception $e) {
            Log::error('Error in getFilteredLeaderboard: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to load leaderboard data'
            ], 500);
        }
    }
    
    /**
     * Get BDC leaderboard specifically
     */
    public function getBDCLeaderboard(Request $request)
    {
        $timePeriod = $request->get('time_period', 'current_month');
        $data = $this->leaderboardService->getBDCLeaderboard($timePeriod);
        
        return view('partials.enhancedLeaderboard', [
            'data' => $data,
            'category' => 'BDC',
            'top3' => $data->take(3),
            'complete_ranking' => $data->skip(3)
        ])->render();
    }
    
    /**
     * Get Service Advisor leaderboard specifically
     */
    public function getServiceAdvisorLeaderboard(Request $request)
    {
        $timePeriod = $request->get('time_period', 'current_month');
        $data = $this->leaderboardService->getServiceAdvisorLeaderboard($timePeriod);
        
        return view('partials.enhancedLeaderboard', [
            'data' => $data,
            'category' => 'Service Advisor',
            'top3' => $data->take(3),
            'complete_ranking' => $data->skip(3)
        ])->render();
    }
}